/*
 * Decompiled with CFR 0.152.
 */
package me.magnet.consultant;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.SetMultimap;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicBoolean;
import me.magnet.consultant.Check;
import me.magnet.consultant.ConfigListener;
import me.magnet.consultant.ConfigUpdater;
import me.magnet.consultant.ConfigValidator;
import me.magnet.consultant.ConsultantException;
import me.magnet.consultant.PropertiesUtil;
import me.magnet.consultant.Service;
import me.magnet.consultant.ServiceIdentifier;
import me.magnet.consultant.ServiceRegistration;
import me.magnet.consultant.SettingListener;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Consultant {
    private static final int HEALTH_CHECK_INTERVAL = 10;
    private static Logger log = LoggerFactory.getLogger(Consultant.class);
    private final AtomicBoolean registered = new AtomicBoolean();
    private final CloseableHttpClient http;
    private final ScheduledExecutorService executor;
    private final String host;
    private final ServiceIdentifier id;
    private final ObjectMapper mapper;
    private final ConfigValidator validator;
    private final Properties validated;
    private final boolean pullConfig;
    private final Multimap<String, SettingListener> settingListeners;
    private final Set<ConfigListener> configListeners;

    public static Builder builder() {
        return new Builder();
    }

    private Consultant(ScheduledExecutorService executor, String host, ServiceIdentifier identifier, SetMultimap<String, SettingListener> settingListeners, Set<ConfigListener> configListeners, ConfigValidator validator, CloseableHttpClient http, boolean pullConfig) {
        this.settingListeners = Multimaps.synchronizedSetMultimap(settingListeners);
        this.configListeners = Sets.newConcurrentHashSet(configListeners);
        this.mapper = new ObjectMapper();
        this.validator = validator;
        this.executor = executor;
        this.host = host;
        this.id = identifier;
        this.pullConfig = pullConfig;
        this.validated = new Properties();
        this.http = http;
    }

    private void init(Properties initProperties) {
        this.updateValidatedConfig(initProperties);
        if (!this.pullConfig) {
            return;
        }
        log.info("Fetching initial configuration from Consul...");
        ConfigUpdater poller = new ConfigUpdater(this.executor, this.http, this.host, null, this.id, this.mapper, null, properties -> {
            if (this.validator == null) {
                this.updateValidatedConfig(properties);
            } else {
                try {
                    this.validator.validateConfig(properties);
                    this.updateValidatedConfig(properties);
                }
                catch (RuntimeException e) {
                    log.warn("New config did not pass validation: " + e.getMessage(), (Throwable)e);
                }
            }
        });
        try {
            this.executor.submit(poller).get();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        catch (ExecutionException e) {
            throw new ConsultantException(e.getCause());
        }
    }

    public void registerService(int port) {
        if (!this.registered.compareAndSet(false, true)) {
            log.warn("Cannot register the service, as service was already registered!");
            return;
        }
        String url = this.host + "/v1/agent/service/register";
        log.info("Registering service with Consul: {}", (Object)this.id);
        try {
            String serviceId = this.id.getInstance().get();
            String serviceName = this.id.getServiceName();
            String serviceHost = this.id.getHostName().orElse(InetAddress.getLoopbackAddress().getHostAddress());
            Check check = new Check("http://" + serviceHost + ":" + port + "/_health", 10);
            ServiceRegistration registration = new ServiceRegistration(serviceId, serviceName, serviceHost, port, check, new String[0]);
            String serialized = this.mapper.writeValueAsString((Object)registration);
            HttpPut request = new HttpPut(url);
            request.setEntity((HttpEntity)new StringEntity(serialized));
            request.setHeader("User-Agent", "Consultant");
            Throwable throwable = null;
            try (CloseableHttpResponse response = this.http.execute((HttpUriRequest)request);){
                int statusCode = response.getStatusLine().getStatusCode();
                if (statusCode >= 200 && statusCode < 400) {
                    return;
                }
                try {
                    log.error("Could not register service, status: " + statusCode);
                    throw new ConsultantException("Could not register service. Consul returned: " + statusCode);
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
            }
        }
        catch (IOException | RuntimeException e) {
            this.registered.set(false);
            log.error("Could not register service!", (Throwable)e);
            throw new ConsultantException(e);
        }
    }

    public void deregisterService() {
        if (!this.registered.compareAndSet(true, false)) {
            log.warn("Cannot deregister the service, as service wasn't registered or was already deregistered!");
            return;
        }
        String serviceId = this.id.getInstance().get();
        String url = this.host + "/v1/agent/service/deregister/" + serviceId;
        log.info("Deregistering service from Consul: {}", (Object)this.id);
        HttpDelete request = new HttpDelete(url);
        request.setHeader("User-Agent", "Consultant");
        try {
            Throwable throwable = null;
            try (CloseableHttpResponse response = this.http.execute((HttpUriRequest)request);){
                int statusCode = response.getStatusLine().getStatusCode();
                if (statusCode >= 200 && statusCode < 400) {
                    return;
                }
                try {
                    log.error("Could not deregister service, status: " + statusCode);
                    throw new ConsultantException("Could not deregister service. Consul returned: " + statusCode);
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
            }
        }
        catch (IOException | RuntimeException e) {
            this.registered.set(true);
            log.error("Could not deregister service!", (Throwable)e);
            throw new ConsultantException(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public List<Service> list(String serviceName) {
        String url = this.host + "/v1/catalog/service/" + serviceName;
        HttpGet request = new HttpGet(url);
        request.setHeader("User-Agent", "Consultant");
        try (CloseableHttpResponse response = this.http.execute((HttpUriRequest)request);){
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode >= 200 && statusCode < 400) {
                InputStream content = response.getEntity().getContent();
                List list = (List)this.mapper.readValue(content, (TypeReference)new TypeReference<List<Service>>(){});
                return list;
            }
            log.error("Could not locate service: " + serviceName + ", status: " + statusCode);
            throw new ConsultantException("Could not locate service: " + serviceName + ". Consul returned: " + statusCode);
        }
        catch (IOException | RuntimeException e) {
            log.error("Could not locate service: " + serviceName);
            throw new ConsultantException(e);
        }
    }

    public Optional<InetSocketAddress> locate(String serviceName) {
        return this.list(serviceName).stream().findFirst().map(service -> {
            String address = Optional.ofNullable(Strings.emptyToNull((String)service.getServiceAddress())).orElse(service.getAddress());
            return new InetSocketAddress(address, (int)service.getServicePort());
        });
    }

    public void addConfigListener(ConfigListener listener) {
        this.configListeners.add(listener);
    }

    public boolean removeConfigListener(ConfigListener listener) {
        return this.configListeners.remove(listener);
    }

    public void addSettingListener(String key, SettingListener listener) {
        this.settingListeners.put((Object)key, (Object)listener);
    }

    public boolean removeSettingListener(String key, SettingListener listener) {
        return this.settingListeners.remove((Object)key, (Object)listener);
    }

    private void updateValidatedConfig(Properties newConfig) {
        Map<String, Pair<String, String>> changes = PropertiesUtil.sync(newConfig, this.validated);
        if (changes.isEmpty()) {
            return;
        }
        for (ConfigListener configListener : this.configListeners) {
            configListener.onConfigUpdate(this.validated);
        }
        for (Map.Entry entry : changes.entrySet()) {
            String key = (String)entry.getKey();
            Collection listeners = this.settingListeners.get((Object)key);
            if (listeners == null || listeners.isEmpty()) continue;
            for (SettingListener listener : listeners) {
                Pair change = (Pair)entry.getValue();
                listener.onSettingUpdate(key, (String)change.getLeft(), (String)change.getRight());
            }
        }
    }

    public void shutdown() {
        try {
            this.deregisterService();
        }
        catch (ConsultantException e) {
            log.error("Error occurred while deregistering", (Throwable)e);
        }
        if (this.pullConfig && !this.executor.isShutdown()) {
            this.executor.shutdownNow();
        }
        try {
            this.http.close();
        }
        catch (IOException e) {
            log.error("Error occurred on shutdown: " + e.getMessage(), (Throwable)e);
        }
    }

    public ServiceIdentifier getServiceIdentifier() {
        return this.id;
    }

    public Properties getProperties() {
        return this.validated;
    }

    public static class Builder {
        private ScheduledExecutorService executor;
        private String host;
        private ServiceIdentifier id;
        private ConfigValidator validator;
        private CloseableHttpClient http;
        private Properties properties;
        private boolean pullConfig = true;
        private final SetMultimap<String, SettingListener> settingListeners = HashMultimap.create();
        private final Set<ConfigListener> configListeners = Sets.newHashSet();

        private Builder() {
            this.properties = new Properties();
        }

        public Builder usingExecutor(ScheduledExecutorService executor) {
            this.executor = executor;
            return this;
        }

        public Builder withConsulHost(String host) {
            this.host = host;
            return this;
        }

        public Builder identifyAs(String serviceName) {
            return this.identifyAs(serviceName, this.fromEnvironment("SERVICE_DC"), this.fromEnvironment("SERVICE_HOST"), Optional.ofNullable(this.fromEnvironment("SERVICE_INSTANCE")).orElse(UUID.randomUUID().toString()));
        }

        public Builder identifyAs(String serviceName, String datacenter) {
            return this.identifyAs(serviceName, datacenter, this.fromEnvironment("SERVICE_HOST"), Optional.ofNullable(this.fromEnvironment("SERVICE_INSTANCE")).orElse(UUID.randomUUID().toString()));
        }

        public Builder identifyAs(String serviceName, String datacenter, String hostname) {
            return this.identifyAs(serviceName, datacenter, hostname, Optional.ofNullable(this.fromEnvironment("SERVICE_INSTANCE")).orElse(UUID.randomUUID().toString()));
        }

        public Builder identifyAs(String serviceName, String datacenter, String hostname, String instanceName) {
            Preconditions.checkArgument((!Strings.isNullOrEmpty((String)serviceName) ? 1 : 0) != 0, (Object)"You must specify a 'serviceName'!");
            this.id = new ServiceIdentifier(serviceName, datacenter, hostname, instanceName);
            return this;
        }

        @VisibleForTesting
        Builder usingHttpClient(CloseableHttpClient httpClient) {
            this.http = httpClient;
            return this;
        }

        public Builder onValidConfig(ConfigListener listener) {
            this.configListeners.add(listener);
            return this;
        }

        public Builder onSettingUpdate(String key, SettingListener listener) {
            this.settingListeners.put((Object)key, (Object)listener);
            return this;
        }

        public Builder validateConfigWith(ConfigValidator validator) {
            this.validator = validator;
            return this;
        }

        public Builder pullConfigFromConsul(boolean pullConfig) {
            this.pullConfig = pullConfig;
            return this;
        }

        public Builder startWith(Properties properties) {
            Preconditions.checkArgument((properties != null ? 1 : 0) != 0, (Object)"You must specify a non-null Properties object!");
            this.properties = properties;
            return this;
        }

        public Consultant build() {
            if (Strings.isNullOrEmpty((String)this.host)) {
                this.host = this.fromEnvironment("CONSUL_HOST");
                if (Strings.isNullOrEmpty((String)this.host)) {
                    this.host = "http://localhost:8500";
                }
            }
            boolean executorSpecified = true;
            if (this.executor == null) {
                executorSpecified = false;
                this.executor = new ScheduledThreadPoolExecutor(1);
            }
            if (this.http == null) {
                PoolingHttpClientConnectionManager manager = new PoolingHttpClientConnectionManager();
                manager.setMaxTotal(5);
                manager.setDefaultMaxPerRoute(5);
                this.http = HttpClientBuilder.create().setConnectionManager((HttpClientConnectionManager)manager).build();
            }
            if (this.id == null) {
                this.id = new ServiceIdentifier((String)Preconditions.checkNotNull((Object)this.fromEnvironment("SERVICE_NAME"), (Object)"You must specify the name of the service using SERVICE_NAME=<service_name>"), this.fromEnvironment("SERVICE_DC"), this.fromEnvironment("SERVICE_HOST"), Optional.ofNullable(this.fromEnvironment("SERVICE_INSTANCE")).orElse(UUID.randomUUID().toString()));
            }
            Consultant consultant = new Consultant(this.executor, this.host, this.id, this.settingListeners, this.configListeners, this.validator, this.http, this.pullConfig);
            consultant.init(this.properties);
            if (!executorSpecified) {
                Runtime.getRuntime().addShutdownHook(new Thread(consultant::shutdown));
            }
            return consultant;
        }

        private String fromEnvironment(String key) {
            String property = System.getProperty(key);
            if (property != null) {
                return property;
            }
            property = System.getenv(key);
            if (property != null) {
                return property;
            }
            return null;
        }
    }
}

