/*
 * Copyright 2022-2023 Omico
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package me.omico.gradm.internal.codegen

import com.squareup.kotlinpoet.ClassName
import com.squareup.kotlinpoet.FileSpec
import com.squareup.kotlinpoet.FunSpec
import com.squareup.kotlinpoet.KModifier
import com.squareup.kotlinpoet.ParameterizedTypeName.Companion.parameterizedBy
import com.squareup.kotlinpoet.TypeSpec
import me.omico.gradm.GRADM_DEPENDENCY_PACKAGE_NAME
import me.omico.gradm.GRADM_PACKAGE_NAME
import me.omico.gradm.GradmExperimentalConfiguration
import me.omico.gradm.GradmGeneratedPluginType
import me.omico.gradm.VersionsMeta
import me.omico.gradm.internal.YamlDocument
import me.omico.gradm.internal.config.buildInRepositories
import me.omico.gradm.internal.config.plugins
import me.omico.gradm.internal.config.repositories
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.initialization.Settings
import org.gradle.api.plugins.ExtensionAware
import java.net.URI
import java.nio.file.Path

internal fun CodeGenerator.generatePluginSourceFile() =
    generatePluginSourceFile<ExtensionAware>(
        generatedSourcesDirectory = generatedSourcesDirectory,
        type = GradmGeneratedPluginType.General,
        overrideApplyFunctionBuilder = {
            controlFlow("if (target is %T)", Settings::class) {
                declarePluginsInSettings(gradmConfigDocument, versionsMeta)
                declareRepositoriesInSettings(gradmConfigDocument)
                declareDependenciesInSettings(dependencies)
                declareVersionsInSettings()
            }
            controlFlow("if (target is %T)", Project::class) {
                declareDependenciesInProject(dependencies)
                declareVersionsInProject()
            }
        },
    )

private inline fun <reified T> generatePluginSourceFile(
    generatedSourcesDirectory: Path,
    type: GradmGeneratedPluginType,
    overrideApplyFunctionBuilder: FunSpec.Builder.() -> Unit = {},
) = FileSpec.builder(type.packageName, type.className)
    .addSuppressWarningTypes(types = defaultSuppressWarningTypes + "UnstableApiUsage")
    .addGradmComment()
    .apply {
        TypeSpec.classBuilder(type.className)
            .addSuperinterface(Plugin::class.parameterizedBy(T::class))
            .addOverrideApplyFunction<T>(overrideApplyFunctionBuilder)
            .build()
            .also(::addType)
    }
    .build()
    .writeTo(generatedSourcesDirectory)

private inline fun <reified T> TypeSpec.Builder.addOverrideApplyFunction(
    overrideApplyFunctionBuilder: FunSpec.Builder.() -> Unit = {},
): TypeSpec.Builder =
    apply {
        FunSpec.builder("apply")
            .addModifiers(KModifier.OVERRIDE)
            .addParameter("target", T::class)
            .apply(overrideApplyFunctionBuilder)
            .build()
            .also(::addFunction)
    }

private fun FunSpec.Builder.declarePluginsInSettings(document: YamlDocument, versionsMeta: VersionsMeta) =
    controlFlow("target.pluginManagement.plugins") {
        document.plugins
            .sortedBy { plugin -> plugin.id }
            .forEach { plugin ->
                val version = versionsMeta.resolveVariable(plugin.module, plugin.version)
                addStatement("id(\"${plugin.id}\").version(\"${version}\").apply(false)")
            }
    }

private fun FunSpec.Builder.declareRepositoriesInSettings(document: YamlDocument) =
    controlFlow("target.dependencyResolutionManagement.repositories") {
        document.repositories.forEach { repository ->
            if (repository.id == "mavenLocal") {
                addStatement("mavenLocal()")
                return@forEach
            }
            if (repository.noUpdates) return@forEach
            buildInRepositories.find { it.id == repository.id }
                ?.let { addStatement("${it.id}()") }
                ?: addStatement("maven { url = %T.create(\"${repository.url}\") }", URI::class)
        }
    }

private fun FunSpec.Builder.declareDependenciesInSettings(dependencies: CodegenDependencies) =
    controlFlow("target.gradle.rootProject") {
        controlFlow("allprojects") {
            declareDependencies(dependencies)
        }
    }

private fun FunSpec.Builder.declareDependenciesInProject(dependencies: CodegenDependencies) =
    controlFlow("with(target)") {
        declareDependencies(dependencies)
    }

private fun FunSpec.Builder.declareDependencies(dependencies: CodegenDependencies) {
    dependencies.keys.forEach { name -> addDependencyExtension(path = "dependencies", name = name) }
    if (GradmExperimentalConfiguration.kotlinMultiplatformSupport) {
        addComment("Kotlin Multiplatform Support")
        dependencies.keys.forEach { name ->
            if (name in GradmExperimentalConfiguration.kotlinMultiplatformIgnoredExtensions) return@forEach
            addDependencyExtension(name = name)
        }
    }
}

private fun FunSpec.Builder.addDependencyExtension(path: String? = null, name: String) {
    val extensionsPath = when (path) {
        null -> "extensions"
        else -> {
            require(path.isNotBlank()) { "path must not be empty or blank" }
            require(extensionsPathRegex.matches(path)) { "path is invalid" }
            "$path.extensions"
        }
    }
    addExtensionsIfNeeds(
        extensionsPath = extensionsPath,
        name = name,
        className = ClassName(GRADM_DEPENDENCY_PACKAGE_NAME, name.capitalize()),
    )
}

private fun FunSpec.Builder.declareVersionsInSettings() =
    controlFlow("target.gradle.rootProject") {
        controlFlow("allprojects") {
            declareVersions()
        }
    }

private fun FunSpec.Builder.declareVersionsInProject() =
    controlFlow("with(target)") {
        declareVersions()
    }

private fun FunSpec.Builder.declareVersions() {
    addExtensionsIfNeeds(name = "versions", className = ClassName(GRADM_PACKAGE_NAME, "Versions"))
}

private fun FunSpec.Builder.addExtensionsIfNeeds(
    extensionsPath: String = "extensions",
    name: String,
    className: ClassName,
) {
    addStatement(format = "$extensionsPath.findByName(\"${name}\") ?: $extensionsPath.add(\"${name}\", %T)", className)
}

private val extensionsPathRegex = """^(\w+\.)*\w+$""".toRegex()
