package com.cdk.core.security.service.impl;

import java.util.Arrays;

import me.parakh.core.model.util.DateUtility;

import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.HttpClient;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.HttpClientBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import com.cdk.core.security.dto.OAuthListTokenDto;
import com.cdk.core.security.dto.OAuthTokenDto;
import com.cdk.core.security.dto.OAuthValidateDto;
import com.cdk.core.security.service.IdentityService;

/**
 * read more @ https://confluence.cdk.com/display/DSCP/OAuth+Token+Management
 * 
 * APIs to generate, update, revoke and validate tokens
 * 
 * http://api-int.dit.adpedge.com/identityservice/1.1.1/rest/oauth/tokens/user/<>
 * 
 * @author Kevendra Patidar
 */
@Service
public class IdentityServiceImpl implements IdentityService{

	/* ************************************ Static Fields ************************************ */
    private static final Logger LOGGER = LoggerFactory.getLogger(IdentityServiceImpl.class);
    
    public static final String IAM_BASE_SERVICE = "/identityservice/1.1.1/rest/";
    public static final String IAM_USER_ENDPOINT = IAM_BASE_SERVICE + "oauth/";
    public static final String IAM_ADMIN_ENDPOINT = IAM_BASE_SERVICE + "admin/oauth/";

    public static final String IAM_USER_VALIDATE = IAM_USER_ENDPOINT + "validate/";//IAMService Token Validate URL
    public static final String IAM_USER_SESSION_TOKEN = IAM_USER_ENDPOINT + "session-token/";//Create Session token URL
    public static final String IAM_USER_TOKENS = IAM_USER_ENDPOINT + "tokens/user/";//Token list URL 
    public static final String IAM_ADMIN_TOKENS =  IAM_ADMIN_ENDPOINT + "tokens/user/";//Create Long lived token URL
    public static final String IAM_ADMIN_DELETE_TOKENS =  IAM_ADMIN_ENDPOINT + "tokens/";//Create Long lived token URL


	/* ************************************ Instance Fields ************************************ */
    /*
     * For production :
     * Url --> http://api-int.adpedge.com OR http://api-int.prod.adpedge.com
     * Credential --> prod-remote/prod-remsleep
     * 
     * For dit :
     * Url --> http://api-int.dit.adpedge.com
     * Credential --> dit-remote/dit-remsleep
     * 
     */
/*
 * p4 path //Java/CobaltDeploymentCenterApp/main/configurations/daemon/applications/social.xml
 * 
    <property name="iam.service.user">
        <description>The remoting username for IAM service</description>
        <value environment="default">dit-remote</value>
        <value environment="qa-3a">staging-remote</value>
        <value environment="qa-3b">staging-remote</value>
        <value environment="uat-orda">staging-remote</value>
        <value environment="uat-ordb">staging-remote</value>
        <value environment="proda">prod-remote</value>
        <value environment="prodb">prod-remote</value>
    </property>

    <property name="iam.service.password">
        <description>The remoting password for IAM service</description>
        <value environment="default">dit-remsleep</value>
        <value environment="qa-3a">staging-remsleep</value>
        <value environment="qa-3b">staging-remsleep</value>
        <value environment="uat-orda">staging-remsleep</value>
        <value environment="uat-ordb">staging-remsleep</value>
        <value environment="proda">prod-remsleep</value>
        <value environment="prodb">prod-remsleep</value>
    </property>    
 */
    @Value("${iam.service.user:dit-remote}")
    private String username;
    @Value("${iam.service.password:dit-remsleep}")
    private String password;
    @Value("${iam.service.url:http://api-int.dit.adpedge.com}")
    private String iamUrl;
    @Value("${token.expiry.days:45}")
    private Integer tokenExpiryDays;
    
	/* ************************************ Public Methods ************************************ */
    /**
     * HttpClientErrorException 400 Bad Request if token in not available
     * @param token
     */
    public void deleteToken(String token) {
    	String url = iamUrl + IAM_ADMIN_DELETE_TOKENS + token;
        getRestTemplate().delete(url);
    }
    /**
     * generate/create a new long lived access token for this current logged in user
     * Request:
     * {
		  "tokenRequest": {
		    "deviceId": "firefox",
		    "deviceType": "browser",
		    "expirationTimestamp": "2017-01-04T14:36:17-0800", // or it can be a long value 1423861196185
		    "comments": "long term token"
		  }
		}
     */
    public OAuthTokenDto generateLongLivedToken(String username, String deviceId, String deviceType) throws RestClientException {
    	OAuthTokenDto tokenRequest = new OAuthTokenDto();
    	tokenRequest.setDeviceId(deviceId);
    	tokenRequest.setDeviceType(deviceType);
    	tokenRequest.setComments("Long Lived Token - created by EDGE"); 
    	tokenRequest.setExpirationTimestamp(DateUtility.addDays(tokenExpiryDays));
    	
    	String adminUrl = iamUrl + IAM_ADMIN_TOKENS + username;
    	
    	HttpHeaders headers = getHttpHeaders();
    	HttpEntity<OAuthTokenDto> entity = new HttpEntity<>(tokenRequest, headers);
    	
        RestTemplate restTemplate = getRestTemplate();
        restTemplate.setMessageConverters(BasicRequestFactory.getJsonMessageConverters());
    	LOGGER.debug("Creating long term token using IAM Service");
    	return restTemplate.postForObject(adminUrl, entity, OAuthTokenDto.class);
    }
    /**
     * Create a session token valid for 24 hours for this current logged in user.
     * 
     * HttpClientErrorException 400 Bad Request if username not fond
     */
	public OAuthTokenDto generateSessionToken(final String username) throws RestClientException {
    	LOGGER.debug("Getting tokes from IAM Service");
    	String url = iamUrl + IAM_USER_SESSION_TOKEN + username;
    	HttpHeaders headers = getHttpHeaders();
    	HttpEntity<String> entity = new HttpEntity<String>("parameters", headers);
    	
    	//IAM Team created a POST request with no request object :-)
    	return getRestTemplate().postForObject(url, entity, OAuthTokenDto.class);
    }
	/**
	 * if boolean authenticated = checkToken(HttpServletRequest httpRequest, HttpServletResponse httpResponse) 
	 * then in GenericFilterBean.doFilter continue --> FilterChain.doFilter(request, response);
	 * 
	 * httpServletResponse.setStatus(HttpServletResponse.SC_OK); 
	 * Response:
		<oauthResponse>
			<status>Success</status>
			<userName>kevendra.patidar@cdk.com</userName>
			<userGuid>F12E7CB9C258F56CE04013ACF8FB54EE</userGuid>
		</oauthResponse>
	 * 
	 * HttpServletResponse.sendError(HttpServletResponse.SC_UNAUTHORIZED) 401
	 * Response after conversion:
	    "oauthTokenRequest": {
	      "status": "Error",
	      "userName": "",
	      "userGuid": null
	    }		
	 */
    public OAuthValidateDto validateToken(String token) {
        String url = iamUrl + IAM_USER_VALIDATE + token;
        
        RestTemplate restTemplate = getRestTemplate();
        restTemplate.setMessageConverters(BasicRequestFactory.getJaxbMessageConverters());
        return restTemplate.getForObject(url, OAuthValidateDto.class);
    }
    /**
     * List of all available token for this current logged in user
     * 
     * HttpClientErrorException 400 Bad Request if username not fond
     */
    public OAuthListTokenDto getAllTokens(String username) throws RestClientException {
    	String url = iamUrl + IAM_USER_TOKENS + username;
        return getRestTemplate().getForObject(url, OAuthListTokenDto.class);
    }
    
	/* ************************************ Private Methods ************************************ */
    private HttpHeaders getHttpHeaders() {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        headers.setContentType(MediaType.APPLICATION_JSON);
        return headers;
    }
    private RestTemplate getRestTemplate() {
        return BasicRequestFactory.createBasicAuthTemplate(username, password);
    }
    public RestTemplate createBasicAuthTemplate(String username, String password) {
        BasicCredentialsProvider credentialsProvider = new BasicCredentialsProvider();
        credentialsProvider.setCredentials(AuthScope.ANY, new UsernamePasswordCredentials(username, password));
        HttpClient httpClient = HttpClientBuilder.create()
                .setDefaultCredentialsProvider(credentialsProvider)
                .build();

        ClientHttpRequestFactory requestFactory = new HttpComponentsClientHttpRequestFactory(httpClient);
        RestTemplate template = new RestTemplate(requestFactory);
        return template;
    } 
    
}
