package com.cdk.core.security.service.impl;

import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import com.cdk.core.security.dto.OAuthListTokenDto;
import com.cdk.core.security.dto.OAuthTokenDto;
import com.cdk.core.security.dto.OAuthValidateDto;
import com.cdk.core.security.dto.UserCredentialDto;
import com.cdk.core.security.service.AuthTestType;
import com.cdk.core.security.service.AuthenticationServiceProvider;

/**
 * Auhentication Service provider
 * 
 * @author Kevendra Patidar
 */
@Service
public class AuthenticationServiceProviderImpl implements AuthenticationServiceProvider{

	/* ************************************ Static Fields ************************************ */
    private static Logger LOG = LoggerFactory.getLogger(AuthenticationServiceProviderImpl.class);
    
	/* ************************************ Instance Fields ************************************ */    
    @Resource
    private SiteminderServiceImpl siteminderService;
    @Resource
    private IdentityServiceImpl identityService;
    @Value("${useCdkIdentityService:false}")
    private boolean useCdkIdentityService;
    
	/* ************************************ Public Methods ************************************ */
    @Override
	public Map<String, Object> test(AuthTestType type, String token, String username, String deviceId, String deviceType){
    	Map<String, Object> data = new HashMap<String, Object>();
    	switch(type){
    	case validateToken:
    		OAuthValidateDto oauthValidateResponseV = identityService.validateToken(token);	
        	data.put("oauthTokenRequest", oauthValidateResponseV);
    		break;
    	case deleteToken:
    		/** Logouts the user - token is invalidated/forgotten. */
    		identityService.deleteToken(token);
    		break;
    	case listToken:
    		OAuthListTokenDto oauthListTokenResponse = identityService.getAllTokens(username);
    		data.put("oauthListTokenResponse", oauthListTokenResponse);
    		break;
    	case generateLongLivedToken:
    		OAuthTokenDto oauthTokenRequest = identityService.generateLongLivedToken(username, deviceId, deviceType);	
        	data.put("oauthTokenRequest", oauthTokenRequest);    		
    		break;
    	case generateSessionToken:
    		OAuthTokenDto oauthTokenRequestX = identityService.generateSessionToken(username);	
        	data.put("oauthTokenRequestX", oauthTokenRequestX);
    		break;    		
    	}    	
		return data;
	}    
    @Override
	public OAuthTokenDto getToken(final UserCredentialDto userCredentials, boolean genSortLivedToken){
       	updateDefaultIfEmpty(userCredentials);
       	OAuthTokenDto oAuthTokenDto = createUserDto(userCredentials, genSortLivedToken, genSortLivedToken);
       	return oAuthTokenDto;
    }
	
	/* ************************************ Private Methods ************************************ */
    /**
     * Create the OAuthTokenRequest which includes the new token. 
     * If the user is already authenticated, then
     * pass in true to bypass SiteMinder authentication.
     */
    private OAuthTokenDto createUserDto(UserCredentialDto userCredential, boolean isAuthenticated, boolean isSessionToken){
    	String userName = userCredential.getUsername();
    	OAuthTokenDto oauthToken = null;      
    	if( ! isAuthenticated){
    		/*
    		 * if user not Authenticated then first authenticate
    		 * siteminderService authenticate and return long lived token 
    		 * OR 
    		 * if oAuthListTokenResponse = null then user 401 Authorization Required or call failed
    		 */
    		OAuthListTokenDto oAuthListTokenResponse = siteminderService.authenticate(userCredential);//SiteMinder authentication
            if(null == oAuthListTokenResponse){
    			//401 with siteminderService
    			return null;
            }
        	/*
        	 * token list will be same as 
        	 * http://api-int.dit.adpedge.com/identityservice/1.1.1/rest/oauth/tokens/user/kevendra.patidar@cdk.com
        	 */
            List<OAuthTokenDto> existingTokens = oAuthListTokenResponse.getTokens();
            oauthToken = AuthUtility.existingTokens(userCredential.getDeviceId(), userCredential.getDeviceType(), existingTokens);
            if(null != oauthToken){
            	//return if valid token found
            	return oauthToken;
            }
    	}else{
    		LOG.error("FIXME ");
    		//FIXME validate that user is authenticated or whoever claim user authenticated they should take care
    	}
    	if(useCdkIdentityService){
	    	/*
	    	 * no previously created token found(during authenticate) for given device id and device type 
	    	 * need to generate new token
	    	 */ 
	        if(isSessionToken){
	        	//createSessionTokenForAuthenticatedUser
	        	oauthToken = identityService.generateSessionToken(userName);
	    	}else{
		    	/*
		    	 * Get or Create token - createLongLivedTokenForAuthenticatedUser 
		    	 */ 
				oauthToken = identityService.generateLongLivedToken(userName, userCredential.getDeviceId(), userCredential.getDeviceType());
	    	}
    	}
        return oauthToken;
    }
    /**
     * User Token Service is only able to handle a max of 10 characters for device type
     */
    private String getDeviceType(String deviceType) {
        if(deviceType.length() > 10) {
            return deviceType.substring(0, 9);
        }
        return deviceType;
    }
    private void updateDefaultIfEmpty(final UserCredentialDto userCredentials) {
    	String deviceId = userCredentials.getDeviceId();
    	String deviceType = userCredentials.getDeviceType();
		if(StringUtils.isEmpty(deviceId)){
    		deviceId = getSecureRandom() +"";//findbugs:DMI_RANDOM_USED_ONLY_ONCE --> new Random().nextInt()+"";//default random 
    	}
		if(StringUtils.isEmpty(deviceType)){
        	LOG.debug("No deviceType provided");
    		deviceType = "N/A"; //default -NA-
    	}else{
    		deviceType = getDeviceType(deviceType);
    	}
    	userCredentials.setDeviceId(deviceId);
    	userCredentials.setDeviceType(deviceType);
	}  

	private int getSecureRandom() {
	    try {
			// Create a secure random number generator using the SHA1PRNG algorithm
			SecureRandom secureRandomGenerator = SecureRandom.getInstance("SHA1PRNG");
			
			// Get 128 random bytes
			byte[] randomBytes = new byte[128];
			secureRandomGenerator.nextBytes(randomBytes);
	
			// Create two secure number generators with the same seed
			int seedByteCount = 5;
			byte[] seed = secureRandomGenerator.generateSeed(seedByteCount);
	
			SecureRandom secureRandom1 = SecureRandom.getInstance("SHA1PRNG");
			secureRandom1.setSeed(seed);
			
			return secureRandom1.nextInt();
	    } catch (NoSuchAlgorithmException e) {
	    }
	    return 0;
	  }    
}
