package com.cdk.core.security.api;


import java.io.IOException;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import com.cdk.core.security.dto.OAuthValidateDto;
import com.cdk.core.security.service.impl.IdentityServiceImpl;
import com.cdk.core.security.service.impl.SiteminderServiceImpl;

/**
 * Interceptor that validates all the requests to the REST apis to check
 * if the valid access token is present in the header.
 *
    <mvc:interceptors>
        <mvc:interceptor>
            <mvc:mapping path="/**" /> all request
            <mvc:mapping path="/api/**" />
            <bean class="com.cdk.edge.security.api.OauthTokenValidationInterceptor"></bean>
        </mvc:interceptor>
    </mvc:interceptors>
 * 
 * 
 * 
 * HandlerInterceptor and AbstractAuthenticationProcessingFilter may be used for the same purpose 
 * but normally in a Spring based project authentication/authorization is handled by 
 * Spring Security (AbstractAuthenticationProcessingFilter).
 * 
 * HandlerInterceptor belongs to Spring MVC and used for custom logging, time measurement, 
 * HTTP header manipulation or (user) request context enhancement.
 *  
 * A HandlerInterceptor may be placed before all or a specific Spring MVC controller
 * and "lives" within the DispatcherServlet.
 * 
 * In contrast, Spring Security's filter chain integrates with javax.servlet.Filter.
 * A simplified request flow looks like this:
 * Container connector --> Filter (Spring Security) --> 
 * DispatcherServlet (Spring MVC) --> HandlerInterceptor (Spring MVC) --> Controller (Spring MVC)
 * 
 * Spring Security may be used independently from Spring MVC (Take a look at AbstractAuthenticationProcessingFilter 
 * and you can see that it derives from GenericFilterBean which in turn implements Filter). 
 * That said, you can incorporate authentication/authorization with Spring Security in any other web framework
 * 
 * we should rely on a matured and feature rich framework like Spring Security or Apache Shiro for authentication/authorization. 
 * Most of the time such frameworks already have all the features our project requires.
 * A popular framework that is used by many people reduces the probability of severe bugs or security holes 
 * due to frequent bug reports and security checks.
 * 
 * @author Kevendra Patidar
 */
@Component
public class OauthTokenValidationInterceptor implements HandlerInterceptor {

	/* ************************************ Static Fields ************************************ */
	private static final Logger LOG = LoggerFactory.getLogger(OauthTokenValidationInterceptor.class);
	//private static final String ACCESS_TOKEN_HEADER_NAME = "access_token";
	//can be used/passed as part of URL(query string), from body parameter, cookies, or HTTP Header like below
    private static final String SECURITY_TOKEN_HEADER = "x-access-token";
    
	/* ************************************ Constructors ************************************ */

	/* ************************************ Instance Fields ************************************ */
    @Resource
    private SiteminderServiceImpl siteminderService;
    @Resource
    private IdentityServiceImpl identityService;

	/* ************************************ Public Methods ************************************ */
    /**
     * Http403ForbiddenEntryPoint : Authentication credentials not found
     * Http401
     */
    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        String accessToken = request.getHeader(SECURITY_TOKEN_HEADER);
        LOG.info("Access token {}", accessToken);
        if (StringUtils.isEmpty(accessToken)) {
            LOG.error("Access token is null for a request to {}", request.getRequestURL());
            // If no access_token, check this request for signin
            String url = request.getRequestURL().toString();
            if("".equals(url)){
            	return true;
            }else{
            	return unauthorizedRequest(response);
            }
        }
        return checkToken(request, response);
    }

	/* ************************************ Private Methods ************************************ */
    private boolean checkToken(final HttpServletRequest request, final HttpServletResponse response) throws IOException {
    	String accessToken = request.getHeader(SECURITY_TOKEN_HEADER);
    	try{
    		//Validate against IAM or UserService service
	    	OAuthValidateDto authValidateDto = identityService.validateToken(accessToken);
	    	if(null != authValidateDto && "Success".equalsIgnoreCase(authValidateDto.getStatus())){
	            /* 
	             * return ‘true’ to let Spring know to process the request through another interceptor 
	             * or to send it to handler method if there are no further interceptors.
	             */
	    		return true;
	    	}
    	}catch(Exception e){
    	}
    	LOG.debug("Access token is invalid {}", accessToken);
    	return unauthorizedRequest(response);
    }
    private boolean unauthorizedRequest(final HttpServletResponse response) throws IOException{
    	response.sendError(HttpServletResponse.SC_UNAUTHORIZED);
        /*
         * request has been handled by the interceptor itself and no further processing is needed
         */    	
    	return false;    	
    }

	/* ************************************ Default Impl ************************************ */
    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler, ModelAndView modelAndView) throws Exception {
    }
    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
    }

}
