package me.parakh.core.security.filter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import me.parakh.core.security.jwt.JwtClaimEdge;
import me.parakh.core.service.common.UserService;
import me.parakh.core.service.security.AuthenticateService;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.InternalAuthenticationServiceException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.security.web.authentication.SavedRequestAwareAuthenticationSuccessHandler;
import org.springframework.security.web.authentication.SimpleUrlAuthenticationFailureHandler;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
//import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.GenericFilterBean;

/**
 * StatelessAuthenticationFilter
 * 
 * @author Kevendra Patidar
 */
@Configuration
public class CustomAuthenticationFilter extends GenericFilterBean {

	/* ************************************ Static Fields ************************************ */
    public static final String DEFAULT_TOKEN_HEADER = "Authorization";//"x-access-token";
	public static final String SPRING_SECURITY_FORM_USERNAME_KEY = "username";
	public static final String SPRING_SECURITY_FORM_PASSWORD_KEY = "password";
    
	/* ************************************ Instance Fields ************************************ */
	private String userNameParameter = SPRING_SECURITY_FORM_USERNAME_KEY;
	private String passwordParameter = SPRING_SECURITY_FORM_PASSWORD_KEY;
    private String tokenHeader = DEFAULT_TOKEN_HEADER;
    
    @Value("${url.signin:/api/auth/signin}")
    private String signinUrl;
    @Value("${useCdkIdentityService:false}")
    private boolean useCdkIdentityService;
    private RequestMatcher loginRequestMatcher;
    
	private AuthenticationSuccessHandler successHandler = new SavedRequestAwareAuthenticationSuccessHandler();
	private AuthenticationFailureHandler failureHandler = new SimpleUrlAuthenticationFailureHandler();	
/*
 * @see AbstractAuthenticationFilterConfigurer
		authFilter.setAuthenticationSuccessHandler(successHandler);
		authFilter.setAuthenticationFailureHandler(failureHandler);

 */
    @Resource
    private AuthenticationManager authenticationManager;
    @Resource
    private JwtTokenService jwtTokenService;
	@Resource
	private AuthenticateService authenticateService;
	@Resource
	private PasswordEncoder passwordEncoder;
	@Resource
	private UserService userService;
	
	/* ************************************ Constructors ************************************ */
	@PostConstruct
	public void initIt() throws Exception {
		this.loginRequestMatcher = new AntPathRequestMatcher(signinUrl, "POST");
	}
	
	/* ************************************ Public Methods ************************************ */
    @Override
    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws IOException, ServletException {
		HttpServletRequest request = (HttpServletRequest) req;
		HttpServletResponse response = (HttpServletResponse) res;    	
		/*
		 * requiresAuthentication @return <code>true</code> if the filter should attempt authentication,
		 * <code>false</code> otherwise.
		 */
//		if (! requiresAuthentication(request, response)) {
//			chain.doFilter(request, response);
//			return;
//		}
        if (loginRequestMatcher.matches(request)) {
        	//sign in go here
            try {
                Authentication authentication = attemptAuthentication(request, response);
                //Authentication authentication = authenticationService.getAuthentication(request);
                
                if (authentication == null) {
    				// return immediately as subclass has indicated that it hasn't completed
    				// authentication
    				return;
    			}                
      		  successfulLogin(request, response, authentication);
      		  SecurityContextHolder.getContext().setAuthentication(authentication);
            } catch (InternalAuthenticationServiceException failed) {
                logger.error("An internal error occurred while trying to authenticate the user", failed);
                unsuccessfulLogin(request, response, failed);
                return; // always return after failed login request
            } catch (AuthenticationException failed) {
                // Authentication failed
                unsuccessfulLogin(request, response, failed);
                return; // always return after failed login request
            }            
        }else{
        	//secured endpoint go here
	        String token = request.getHeader(tokenHeader);
	        if(StringUtils.isEmpty(token)){
	        	//-OR- throw new InsufficientAuthenticationException("Authorization token not found");
	        	unsuccessfulLogin(request, response, new CustomAuthenticationException("Empty token"));
	        	return;
	        }
	        if("Authorization".equalsIgnoreCase(tokenHeader)){
		        // remove schema from token
	            String authorizationSchema = "Bearer";
	            if (token.indexOf(authorizationSchema) == -1) {
	                //throw new InsufficientAuthenticationException("Authorization schema not found");
	            }
	            token = token.substring(authorizationSchema.length()).trim();
	        }
            try {
                Authentication authResult = attemptTokenAuthentication(token);
                if (authResult != null) {
                    SecurityContextHolder.getContext().setAuthentication(authResult);
                }
            } catch (Exception e) {
                logger.debug("JWT verification failed: ", e);
                unsuccessfulLogin(request, response, new CustomAuthenticationException("Invalid token"));
                return;
            }
        }
        /*
        TODO compair two approach
        SecurityContextHolder.getContext().setAuthentication(authentication);
        chain.doFilter(req, res);
        SecurityContextHolder.getContext().setAuthentication(null);
        
        -vs-
        
        chain.doFilter(req, res);
        */
        chain.doFilter(req, res);
    }

    /* ************************************ Protected Methods ************************************ */
    protected Authentication attemptAuthentication(HttpServletRequest request, HttpServletResponse response) {
        String username = obtainUsername(request);
        String password = obtainPassword(request);
    	if(StringUtils.isEmpty(username)){        	
			username = "";
		}
    	if(StringUtils.isEmpty(password)){        	
			password = "";
		}
		username = username.trim();
		
        UsernamePasswordAuthenticationToken authRequest = new UsernamePasswordAuthenticationToken(username, password);
        JwtClaimEdge claims = new JwtClaimEdge();
        claims.setDid(request.getParameter("deviceId"));
        claims.setDtp(request.getParameter("deviceType"));
        //claims.setSecurableTypes((List<String>)request.getParameter("securableTypes"));//FIXME string to array
        authRequest.setDetails(claims);
        return authenticationManager.authenticate(authRequest);
    }
	protected String obtainUsername(HttpServletRequest request) {
		return request.getParameter(userNameParameter);
	}
	protected String obtainPassword(HttpServletRequest request) {
		return request.getParameter(passwordParameter);
    }
    protected void successfulLogin(HttpServletRequest request, HttpServletResponse response, Authentication authResult) throws IOException, ServletException {
        logger.debug("Authentication request success: " + authResult);
        JwtClaimEdge claims = null;
		try {
			claims = provideClaims(request, authResult);
		} catch (Exception e) {
			logger.error("provide claims failed", e);
		}
        String token = jwtTokenService.sign(claims);
        response.setHeader(tokenHeader, token);
    }
    protected void successfulAuthentication(HttpServletRequest request, HttpServletResponse response, FilterChain chain, Authentication authResult) throws IOException, ServletException {
		SecurityContextHolder.getContext().setAuthentication(authResult);
		successHandler.onAuthenticationSuccess(request, response, authResult);
	}    
    protected void unsuccessfulLogin(HttpServletRequest request, HttpServletResponse response, AuthenticationException failed) throws IOException, ServletException {
        SecurityContextHolder.clearContext();
        failureHandler.onAuthenticationFailure(request, response, failed);
    }
    protected JwtClaimEdge provideClaims(HttpServletRequest request, Authentication authResult) {
        UserDetails userDetails = (UserDetails) authResult.getPrincipal();
        String userName = userDetails.getUsername();
        List<String> roles = new ArrayList<String>();
        for (GrantedAuthority authority : userDetails.getAuthorities()) {
            roles.add(authority.getAuthority());
        }
        JwtClaimEdge claims = (JwtClaimEdge) authResult.getDetails();
        String encodePassword = null;
        if(null != userDetails.getPassword()){
        	encodePassword = passwordEncoder.encode(userDetails.getPassword());
        }
        claims.setUid(getUid(userName, encodePassword));
        claims.setUsr(userName);
        claims.setRol(roles);        
        return claims;
    }
	protected Authentication attemptTokenAuthentication(String token) throws AuthenticationException, IOException, ServletException {
		JwtClaimEdge claims = jwtTokenService.verify(token);
		String userName = claims.getUsr();
        List<GrantedAuthority> authorities = obtainAuthoritiesFromClaims(claims.getRol());
        if(useCdkIdentityService){
	        if(userName == null || authorities == null) {
	            throw new IllegalStateException("Claims 'username' and/or 'authorities' cannot be null");
	        }else if(! authenticateService.tokenVerify(claims.getIamToken())) { 
	        	/*
	        	 * authenticateService.tokenVerify(claims)
	        	 * only iamToken require to pass, but this module not aware against which key iamToken store
	        	 */
	        	throw new IllegalStateException("invalid IAM token ");
	        }
        }
        AbstractAuthenticationToken authentication = new UsernamePasswordAuthenticationToken(userName, null, authorities);
        authentication.setDetails(claims);
        return authentication;
    }
	protected String obtainUsernameFromClaims(Map<String, Object> claims) {
        return (String) claims.get("username");
    }
	protected List<GrantedAuthority> obtainAuthoritiesFromClaims(List<String> roles) {
        List<GrantedAuthority> authorities = new ArrayList<GrantedAuthority>();
        for (String role : roles) {
            authorities.add(new SimpleGrantedAuthority(role));
        }
        return authorities;
    }
    /* ************************************ Private Methods ************************************ */
	private Long getUid(final String username, final String password) {
		return userService.getUidCreateIfNotFound(username, password);
	}
	
}
