package me.parakh.core.security.filter;

import javax.annotation.PostConstruct;

import me.parakh.core.model.util.DateUtility;
import me.parakh.core.security.jwt.JwtClaim;
import me.parakh.core.security.jwt.JwtClaimEdge;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.crypto.codec.Base64;
import org.springframework.security.jwt.Jwt;
import org.springframework.security.jwt.JwtHelper;
import org.springframework.security.jwt.crypto.sign.MacSigner;
import org.springframework.security.jwt.crypto.sign.SignerVerifier;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;


/**
 * Hash-based message authentication code
 * 
 * @author Kevendra Patidar
 */
@Service
public class JwtTokenServiceImpl implements JwtTokenService {

	/* ************************************ Static Fields ************************************ */
	private static final Logger LOG = LoggerFactory.getLogger(JwtTokenServiceImpl.class);

	/* ************************************ Instance Fields ************************************ */
    private ObjectMapper objectMapper = new ObjectMapper();
    private SignerVerifier signerVerifier;
    //@Value("#{secprop['signer.secret']?:'secretkey'}")//working with util:properties id="secprop"
    @Value("${signer.secret:secretkey}")
    public String secret;
    @Value("${signer.secret.base64.encoded:false}")
    public Boolean secretBase64Encoded;
    @Value("${token.expiry.days:45}")
    private Integer tokenExpiryDays;
    
	/* ************************************ Constructors ************************************ */
    @PostConstruct
   	public void initIt() throws Exception {
        Assert.notNull(secret, "secret must not be null");
        /*
        * 
        *  with org.apache.commons.codec.binary.Base64	
        *  String  base64EncodedToken  = Base64.encodeBase64URLSafeString(secret.getBytes());
        *  
        *  with org.springframework.security.crypto.codec.Base64
        *  byte[] base64Token= Base64.encode(secret.getBytes());
        *  String base64EncodedToken=new String(base64Token);
        *  -and decode-
        *  new String(org.springframework.security.crypto.codec.Base64.decode(encodedStr.getBytes()))
        *  
        *  no change 
        *  secret = Utf8.decode(secret.getBytes());
        */
        if(secretBase64Encoded){
            byte[] base64Decode = Base64.decode(secret.getBytes());
            String base64DecodeSecret = new String(base64Decode);
            this.signerVerifier = new MacSigner(base64DecodeSecret);
        }else{
            this.signerVerifier = new MacSigner(secret);
        }
   	} 
    
	/* ************************************ Public Methods ************************************ */
    @Override
	public String sign(final JwtClaim claims) {
    	updateMetadeta(claims);
    	String jsonClaims = toJson(claims);
    	if(StringUtils.isEmpty(jsonClaims)){
    		return null;
    	}
        Jwt jwt = JwtHelper.encode(jsonClaims, signerVerifier); 
        return jwt.getEncoded();
    }
    @Override
	public JwtClaimEdge verify(String token) {
        Jwt jwt = JwtHelper.decodeAndVerify(token, signerVerifier);
        return toJava(jwt.getClaims(), JwtClaimEdge.class);
    }

    /* ************************************ Private Methods ************************************ */
    private void updateMetadeta(final JwtClaim claims) {
    	claims.setIss("CDK Edge");
    	claims.setExp(DateUtility.addDays(tokenExpiryDays));//setExpirationTimestamp
    }
	/**
	 * Convert JSON to Java object
	 * @param json String
	 * @param Java object Type
	 * @return Java object
	 */
    private <T> T toJava(String json, Class<T> valueType){
    	try{
    		return objectMapper.readValue(json, valueType);
		}catch(Exception e){
			LOG.error("error during convert JSON String into Java object", e);
		}
    	return null;
    }
    /**
     * Convert Java object to JSON format
     * @see DealerCookie.getAsJSON()
     * @param valueType
     * @return json
     */
    private String toJson(Object value){
		try{
	    	return objectMapper.writeValueAsString(value) ;
	    }catch(JsonProcessingException e){
	    	LOG.error("Json Processing Exception", e);
	    }catch(Exception e){
			LOG.error("error during convert Java object to JSON String ", e);
		}
    	return null;
    }
}
