package snailgun

import java.io.PrintStream
import java.io.InputStream
import java.util.concurrent.atomic.AtomicBoolean

import snailgun.protocol.Defaults
import snailgun.protocol.Streams
import snailgun.logging.SnailgunLogger

import scopt.OParser
import java.net.ConnectException

/**
  * An implementation of a CLI via case-app.
  *
  * Unfortunately, GraalVM Native Image doesn't correctly generate a native
  * image because of parsing errors and warnings generated by the use of macros
  * in case-app via Shapeless. For that reason, this class is left here but it's
  * not used by default, preferring a CLI implementation that requires no macros
  * and works with GraalVM Native Image.
  */
abstract class Cli(in: InputStream, out: PrintStream, err: PrintStream) {
  def exit(code: Int): Unit
  def run(args: Array[String]): Unit = {
    var setServer: Boolean = false
    var setPort: Boolean = false
    val cliParser = {
      val builder = OParser.builder[CliParams]
      val nailgunServerOpt = builder
        .opt[String]("nailgun-server")
        .action((server, params) => { setServer = true; params.copy(nailgunServer = server) })
        .text("Specify the host name of the target Nailgun server")
      val nailgunPortOpt = builder
        .opt[Int]("nailgun-port")
        .action((port, params) => { setPort = true; params.copy(nailgunPort = port) })
        .text("Specify the port of the target Nailgun server")
      val helpOpt = builder
        .opt[Unit]('h', "help")
        .action((_, params) => params.copy(help = true))
        .text("Print help of the Nailgun server")
      val nailgunShowVersionOpt = builder
        .opt[Unit]("nailgun-showversion")
        .action((_, params) => params.copy(nailgunShowVersion = true))
        .text("Print version of Nailgun client before running command")
      val nailgunHelpOpt = builder
        .help("nailgun-help")
        .text("Print help of the Nailgun client")
      val verboseOpt = builder
        .opt[Unit]("verbose")
        .action((_, params) => params.copy(verbose = true))
        .text("Enable verbosity of the Nailgun client")
      val cmdOpt = builder
        .arg[String]("<cmd>...")
        .optional()
        .unbounded()
        .action((arg, params) => params.copy(args = params.args ++ List(arg)))
        .text("The command and arguments for the Nailgun server")
      OParser
        .sequence(
          builder.programName("nailgun"),
          builder.head("nailgun", Defaults.Version),
          nailgunServerOpt,
          nailgunPortOpt,
          helpOpt,
          nailgunHelpOpt,
          nailgunShowVersionOpt,
          verboseOpt,
          cmdOpt
        )
    }

    def errorAndExit(msg: String): Unit = { err.println(msg); exit(1) }
    OParser.parse(cliParser, args, CliParams()) match {
      case None => exit(1)
      case Some(params) =>
        if (params.nailgunShowVersion)
          out.println(s"Nailgun v${Defaults.Version}")

        def process(cmd: String, cmdArgs: Array[String]) = {
          val streams = Streams(in, out, err)
          val hostServer =
            if (setServer) params.nailgunServer
            else Defaults.env.getOrElse("NAILGUN_SERVER", params.nailgunServer)
          val portServer =
            if (setPort) params.nailgunPort
            else Defaults.env.getOrElse("NAILGUN_PORT", params.nailgunPort.toString).toInt
          val client = TcpClient(hostServer, portServer)
          val noCancel = new AtomicBoolean(false)
          val logger = new SnailgunLogger("log", out, isVerbose = params.verbose)
          try {
            val code =
              client.run(cmd, cmdArgs, Defaults.cwd, Defaults.env, streams, logger, noCancel, true)
            logger.debug(s"Return code is $code")
            exit(code)
          } catch {
            case _: ConnectException =>
              errorAndExit(s"No server running in $hostServer:$portServer!")
          }
        }

        params.args match {
          case Nil if params.help => process("help", Array.empty)
          case Nil                => errorAndExit("Missing command for Nailgun server!")
          case cmd :: cmdArgs     => process(cmd, cmdArgs.toArray)
        }
    }
  }
}

object Cli extends Cli(System.in, System.out, System.err) {
  def main(args: Array[String]): Unit = run(args)
  override def exit(code: Int) = System.exit(code)
}
