package me.youm.core.mqtt.config;

import me.youm.core.mqtt.props.MqttCommonProperties;
import me.youm.core.mqtt.props.MqttInboundProperties;
import me.youm.core.mqtt.props.MqttOutboundProperties;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.integration.annotation.ServiceActivator;
import org.springframework.integration.channel.DirectChannel;
import org.springframework.integration.core.MessageProducer;
import org.springframework.integration.mqtt.core.DefaultMqttPahoClientFactory;
import org.springframework.integration.mqtt.core.MqttPahoClientFactory;
import org.springframework.integration.mqtt.inbound.MqttPahoMessageDrivenChannelAdapter;
import org.springframework.integration.mqtt.outbound.MqttPahoMessageHandler;
import org.springframework.integration.mqtt.support.DefaultPahoMessageConverter;
import org.springframework.messaging.MessageChannel;
import org.springframework.messaging.MessageHandler;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;

/**
 * @author youta
 */
@ConditionalOnClass(MqttConnectOptions.class)
@Configuration
@EnableConfigurationProperties({MqttInboundProperties.class, MqttOutboundProperties.class, MqttCommonProperties.class})
@ConditionalOnProperty(value = "youm.mqtt.enable", havingValue = "true", matchIfMissing = true)
public class MqttClientAutoConfiguration {

    MqttPahoMessageDrivenChannelAdapter adapter;

    @Resource
    MqttInboundProperties mqttInboundProperties;


    @Resource
    MqttOutboundProperties mqttOutboundProperties;

    @Resource
    MqttCommonProperties mqttCommonProperties;

//    @Autowired(required = false)
//    @Qualifier(value = "messageHandler")
//    private MessageHandler messageHandler;

    @Resource
    @Qualifier(value = "defaultMessageHandler")
    private MessageHandler defaultMessageHandler;

    @Bean
    public MqttConnectOptions mqttConnectOptions() {
        MqttConnectOptions mqttConnectOptions = new MqttConnectOptions();
        mqttConnectOptions.setUserName(mqttCommonProperties.getUserName());
        mqttConnectOptions.setPassword(mqttCommonProperties.getPassword().toCharArray());
        mqttConnectOptions.setServerURIs(new String[]{mqttCommonProperties.getUrl()});
        mqttConnectOptions.setKeepAliveInterval(2);
        return mqttConnectOptions;
    }

    /**
     * mqtt客户端工厂类
     *
     * @param mqttConnectOptions options
     * @return MqttPahoClientFactory
     */
    @Bean
    public MqttPahoClientFactory mqttClientFactory(@Autowired MqttConnectOptions mqttConnectOptions) {
        DefaultMqttPahoClientFactory factory = new DefaultMqttPahoClientFactory();
        factory.setConnectionOptions(mqttConnectOptions);
        return factory;
    }

    /**
     * 发送消息处理器（推送）,配置发送消息客户端
     *
     * @param mqttConnectOptions options
     * @return MessageHandler
     */
    @Bean
    @ServiceActivator(inputChannel = "mqttOutboundChannel")
    public MessageHandler mqttOutbound(@Autowired MqttConnectOptions mqttConnectOptions) {
        MqttPahoMessageHandler messageHandler = new MqttPahoMessageHandler(mqttOutboundProperties.getClientId(), mqttClientFactory(mqttConnectOptions));
        messageHandler.setAsync(true);
        messageHandler.setDefaultTopic(mqttOutboundProperties.getTopics());
        // 设置转换器，发送bytes
        DefaultPahoMessageConverter converter = new DefaultPahoMessageConverter();
        converter.setPayloadAsBytes(true);
        messageHandler.setConverter(converter);
        return messageHandler;
    }

    /**
     * 发送通道（消息生产者）
     *
     * @return MessageChannel
     */
    @Bean
    public MessageChannel mqttOutboundChannel() {
        return new DirectChannel();
    }

    /**
     * 接收通道（消息消费者）
     *
     * @return MessageChannel
     */
    @Bean
    public MessageChannel mqttInputChannel() {
        return new DirectChannel();
    }

    /**
     * 配置消息适配器，配置订阅客户端
     *
     * @param mqttConnectOptions options
     * @return MessageProducer
     */
    @Bean
    public MessageProducer inbound(@Autowired MqttConnectOptions mqttConnectOptions) {
        adapter = new MqttPahoMessageDrivenChannelAdapter(mqttInboundProperties.getClientId(), mqttClientFactory(mqttConnectOptions), mqttInboundProperties.getTopics());
        // 设置转换器，接收bytes
        DefaultPahoMessageConverter converter = new DefaultPahoMessageConverter();
        converter.setPayloadAsBytes(true);
        adapter.setConverter(converter);
        adapter.setQos(1);
        adapter.setOutputChannel(mqttInputChannel());
        return adapter;
    }


    /**
     * 接收消息处理器（订阅）
     *
     * @return MessageHandler
     */
    @Bean
    @ServiceActivator(inputChannel = "mqttInputChannel")
    public MessageHandler handler() {
//        if(ObjectUtils.isEmpty(messageHandler)){
        return defaultMessageHandler;
//        }
//        return messageHandler;
    }

    /**
     * 动态添加Topic
     *
     * @param topicNames topicNames
     */
    public void addTopics(String... topicNames) {
        if (!ObjectUtils.isEmpty(topicNames) && topicNames.length > 0) {
            for (String topic : topicNames) {
                adapter.addTopic(topic);
            }
        }
    }

    /**
     * 动态删除Topic
     *
     * @param topicNames topicNames
     */
    public void removeTopic(String... topicNames) {
        if (!ObjectUtils.isEmpty(topicNames) && topicNames.length > 0) {
            for (String topicName : topicNames) {
                adapter.removeTopic(topicName);
            }
        }
    }

}
