package me.youm.core.rocket.config;

import com.aliyun.openservices.ons.api.PropertyKeyConst;
import com.aliyun.openservices.ons.api.PropertyValueConst;
import com.aliyun.openservices.ons.api.bean.OrderConsumerBean;
import com.aliyun.openservices.ons.api.bean.OrderProducerBean;
import com.aliyun.openservices.ons.api.bean.Subscription;
import com.aliyun.openservices.ons.api.order.MessageOrderListener;
import lombok.extern.slf4j.Slf4j;
import me.youm.core.rocket.props.RocketProps;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * @author youta
 */
@Slf4j
public class RocketConfig {

    @Resource
    public RocketProps rocketProps;


    /**
     * 创建生产者
     *
     * @return 生产者bean
     */
    protected OrderProducerBean createProducer(String groupId) {
        OrderProducerBean producerBean = new OrderProducerBean();
        Properties properties = this.getProperties();
        properties.put(PropertyKeyConst.GROUP_ID, groupId);
        producerBean.setProperties(properties);
        return producerBean;
    }

    /**
     * 创建集群订阅消费者
     *
     * @param groupId groupId
     * @return 消费者bean
     */
    protected OrderConsumerBean createConsumer(String groupId, Map<Subscription, MessageOrderListener> subscriptionTable) {
        OrderConsumerBean consumerBean = new OrderConsumerBean();
        Properties properties = this.createConsumerProperties(groupId);
        consumerBean.setProperties(properties);
        consumerBean.setSubscriptionTable(subscriptionTable);
        return consumerBean;
    }

    /**
     * 创建广播模式消费者
     *
     * @param groupId           消费者id
     * @param subscriptionTable 消费监听Map
     * @return 消费者bean
     */
    protected OrderConsumerBean createBbRoadCastConsumer(String groupId, Map<Subscription, MessageOrderListener> subscriptionTable) {
        OrderConsumerBean consumerBean = new OrderConsumerBean();
        Properties properties = this.createConsumerProperties(groupId, true);
        consumerBean.setProperties(properties);
        consumerBean.setSubscriptionTable(subscriptionTable);
        return consumerBean;
    }


    /**
     * 创建消费者属性参数
     * 默认 集群订阅
     *
     * @param groupId groupId
     * @return 消费者属性参数
     */
    private Properties createConsumerProperties(String groupId) {
        return this.createConsumerProperties(groupId, false);
    }


    /**
     * 创建消费者属性参数
     *
     * @param groupId      groupId
     * @param isBbRoadCast 是否是广播订阅队列
     * @return 消费者属性参数
     * 集群订阅：默认模式 ，相同消费者id所有消费者平均分摊消费消息。
     * 例如某个 Topic 有 9 条消息，一个 Consumer ID 有 3 个 Consumer 实例，
     * 那么在集群消费模式下每个实例平均分摊，只消费其中的 3 条消息。
     * <p>
     * 广播订阅：相同消费者id 所标识的所有消费者都会各自消费某条消息一次。
     * 例如某个 Topic 有 9 条消息，一个 Consumer ID 有 3 个 Consumer 实例，
     * 那么在广播消费模式下每个实例都会各自消费 9 条消息。
     */
    private Properties createConsumerProperties(String groupId, boolean isBbRoadCast) {
        Properties properties = this.getProperties();
        properties.put(PropertyKeyConst.GROUP_ID, groupId);
        properties.put(PropertyKeyConst.ConsumeThreadNums, (Runtime.getRuntime().availableProcessors() * 2 + 1) + "");
        if (isBbRoadCast) {
            properties.put(PropertyKeyConst.MessageModel, PropertyValueConst.BROADCASTING);
        }
        return properties;
    }

    /**
     * 创建消费者监听Map
     * key:订阅相关类 (Subscription )
     * value: 消费者监听(MessageListener)
     *
     * @param topic     消息主题
     * @param listeners 监听队列，可设置多个
     * @return 消费者监听Map
     */
//    public Map<Subscription, MessageOrderListener> createSubscriptionTable(String topic, MessageOrderListener listeners) {
//        //expression即Tag，可以设置成具体的Tag，如 taga||tagb||tagc，也可设置成*。
//        // *仅代表订阅所有Tag，不支持通配
//        return this.createSubscriptionTable(topic, STAR_FLOWER, listeners);
//    }

    /**
     * 创建消费者监听Map
     * key:订阅相关类 (Subscription )
     * value: 消费者监听(MessageListener)
     *
     * @param topic     消息主题
     * @param tag       消息标签
     * @param listeners 监听队列，可设置多个
     * @return 消费者监听Map
     */
    public Map<Subscription, MessageOrderListener> createSubscriptionTable(String topic, String tag, MessageOrderListener listeners) {
        //创建监听
        Subscription subscription = new Subscription();
        //主题
        subscription.setTopic(topic);
        //标签
        subscription.setExpression(tag);
        Map<Subscription, MessageOrderListener> subscriptionTable = new HashMap<>(2);
        if (!ObjectUtils.isEmpty(listeners)) {
            subscriptionTable.put(subscription, listeners);
        }
        return subscriptionTable;
    }

    public Map<Subscription, MessageOrderListener> createSubscriptionTable(String tag, MessageOrderListener listeners) {
        //创建监听
        Subscription subscription = new Subscription();
        //主题
        subscription.setTopic(rocketProps.getTopic());
        //标签
        subscription.setExpression(tag);
        Map<Subscription, MessageOrderListener> subscriptionTable = new HashMap<>(2);
        if (!ObjectUtils.isEmpty(listeners)) {
            subscriptionTable.put(subscription, listeners);
        }
        return subscriptionTable;
    }


    private Properties getProperties() {
        Properties properties = new Properties();
        properties.put(PropertyKeyConst.AccessKey, rocketProps.getAccessKey());
        properties.put(PropertyKeyConst.SecretKey, rocketProps.getSecretKey());
        properties.put(PropertyKeyConst.NAMESRV_ADDR, rocketProps.getNameAddr());
        return properties;

    }

}
