package me.youm.core.webflux.handler;

import cn.hutool.core.util.StrUtil;
import io.netty.channel.ConnectTimeoutException;
import lombok.extern.slf4j.Slf4j;
import me.youm.core.common.enums.CodeEnum;
import me.youm.core.common.exception.BusinessException;
import me.youm.core.common.exception.PayException;
import me.youm.core.common.exception.RocketMqException;
import me.youm.core.common.exception.TokenException;
import me.youm.core.common.model.Result;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.server.ResponseStatusException;

/**
 * 异常处理通知
 *
 * @author youta
 */
@Slf4j
@RestControllerAdvice
public class ExceptionHandlerAdvice {

    @ExceptionHandler(value = {ResponseStatusException.class})
    public Result handle(ResponseStatusException ex) {
        log.error("response status exception", ex);
        if (StrUtil.contains(ex.getMessage(), HttpStatus.NOT_FOUND.toString())) {
            return Result.error(CodeEnum.NOT_FOUND.getCode(), ex.getMessage());
        } else {
            return Result.error();
        }
    }

    @ExceptionHandler(value = {ConnectTimeoutException.class})
    public Result handle(ConnectTimeoutException ex) {
        log.error("connect timeout exception", ex);
        return Result.error();
    }


    @ExceptionHandler(value = {BusinessException.class})
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public Result handle(BusinessException ex) {
        log.error("business exception", ex);
        return Result.error(ex.getCode(),ex.getMsg());
    }

    @ExceptionHandler(value = {PayException.class})
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public Result handle(PayException ex) {
        log.error("pay exception", ex);
        return Result.error(ex.getCode(),ex.getMsg());
    }

    @ExceptionHandler(value = {RocketMqException.class})
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public Result handle(RocketMqException ex) {
        log.error("rocket exception", ex);
        return Result.error(ex.getCode(),ex.getMsg());
    }

    @ExceptionHandler(value = {RuntimeException.class})
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public Result handle(RuntimeException ex) {
        log.error("runtime exception", ex);
        return Result.error();
    }

    @ExceptionHandler(value = {TokenException.class})
    @ResponseStatus(HttpStatus.UNAUTHORIZED)
    public Result handle(TokenException ex) {
        log.error("token exception");
        return Result.error(HttpStatus.UNAUTHORIZED.value(), ex.getMessage());
    }

    @ExceptionHandler(value = {Exception.class})
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public Result handle(Exception ex) {
        log.error("exception", ex);
        return Result.error();
    }

    @ExceptionHandler(value = {Throwable.class})
    public Result handle(Throwable throwable) {
        Result result = Result.error();
        if (throwable instanceof ResponseStatusException) {
            result = handle((ResponseStatusException) throwable);
        } else if (throwable instanceof ConnectTimeoutException) {
            result = handle((ConnectTimeoutException) throwable);
        } else if (throwable instanceof TokenException) {
            result = handle((TokenException) throwable);
        } else if (throwable instanceof RuntimeException) {
            result = handle((RuntimeException) throwable);
        } else if (throwable instanceof Exception) {
            result = handle((Exception) throwable);
        }
        return result;
    }

}
