package me.youm.frame.common.model;

import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.ToString;
import me.youm.frame.common.constants.ResultConstant;
import me.youm.frame.common.enums.EnumInterface;
import me.youm.frame.common.exception.BusinessException;
import org.springframework.lang.NonNull;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.util.Objects;
import java.util.Optional;

/**
 * @author youta
 */
@ToString
@ApiModel(description = "响应实体")
public class Result<T> implements Serializable {

    private static final long serialVersionUID = 129486345465L;

    @ApiModelProperty(value = "是否成功", required = true)
    private Boolean success;
    @ApiModelProperty(value = "响应编码", required = true)
    private Integer code;
    @ApiModelProperty(value = "响应信息", required = true)
    private String message;
    @ApiModelProperty(value = "承载数据")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private T data;

    public Result() {
    }

    public Result<T> assertSuccess() {
        Result<T> result = Optional.of(this).get();
        if (!result.success) {
            throw new BusinessException(result.getCode(), !StringUtils.hasText(result.getMessage()) ? ResultConstant.ERROR_MSG : result.getMessage());
        }
        return result;
    }

    /**
     * 判断返回是否为成功
     *
     * @param result Result
     * @return 是否成功
     */
    public static boolean isSuccess(@NonNull Result<?> result) {
        return Objects.requireNonNull(result).isSuccess();
    }

    /**
     * 判断返回是否为成功
     *
     * @param result Result
     * @return 是否成功
     */
    public static boolean isNotSuccess( Result<?> result) {
        return !Result.isSuccess(result);
    }

    public Result(Boolean success, String message, Integer code, T data) {
        this.success = success;
        this.message = message;
        this.code = code;
        this.data = data;
    }

    public static <T> Result<T> success() {
        Result<T> responseData = new Result<T>();
        responseData.setSuccess(true);
        responseData.setMessage(ResultConstant.SUCCESS_MSG);
        responseData.setCode(ResultConstant.SUCCESS_CODE);
        return responseData;
    }

    public static <T> Result<T> success(T data) {
        Result<T> responseData = new Result<T>();
        responseData.setSuccess(true);
        responseData.setMessage(ResultConstant.SUCCESS_MSG);
        responseData.setCode(ResultConstant.SUCCESS_CODE);
        responseData.setData(data);
        return responseData;
    }

    public static Result<?> error() {
        Result<?> responseData = new Result<>();
        responseData.setSuccess(false);
        responseData.setMessage(ResultConstant.ERROR_MSG);
        responseData.setCode(ResultConstant.ERROR_CODE);
        return responseData;
    }

    public static Result<?> error(String msg) {
        Result<?> responseData = new Result<>();
        responseData.setSuccess(false);
        responseData.setMessage(msg);
        responseData.setCode(ResultConstant.ERROR_CODE);
        return responseData;
    }

    public static Result<?> error(Integer code, String msg) {
        Result<?> responseData = new Result<>();
        responseData.setSuccess(false);
        responseData.setMessage(msg);
        responseData.setCode(code);
        return responseData;
    }

    public static Result<?> error(EnumInterface failure) {
        Result<?> responseData = new Result<>();
        responseData.setSuccess(false);
        responseData.setCode(failure.getCode());
        responseData.setMessage(failure.getMsg());
        return responseData;
    }


    public Boolean isSuccess() {
        return success;
    }

    public String getMessage() {
        return message;
    }

    public Integer getCode() {
        return code;
    }

    public T getData() {
        return data;
    }

    public Result<T> setSuccess(Boolean success) {
        this.success = success;
        return this;
    }

    public Result<T> setMessage(String message) {
        this.message = message;
        return this;
    }

    public Result<T> setCode(Integer code) {
        this.code = code;
        return this;
    }

    public Result<T> setData(T data) {
        this.data = data;
        return this;
    }
}

