package me.youm.frame.jpa.base.controller;

import cn.hutool.core.util.StrUtil;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import me.youm.frame.common.enums.CodeEnum;
import me.youm.frame.common.enums.SearchTypeEnum;
import me.youm.frame.common.model.IPage;
import me.youm.frame.common.model.Result;
import me.youm.frame.jpa.base.entity.BaseEntity;
import me.youm.frame.jpa.base.entity.BaseSearchField;
import me.youm.frame.jpa.base.service.IBaseService;
import me.youm.frame.jpa.utils.UpdatePoUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.time.LocalDateTime;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;


/**
 * 基础控制器基类，包含常见的基本的CRUD的请求的处理，其他特殊的方法通过子类继承实现。
 * T1,是操作的实体类，T2是对应的service接口类继承IBaseService,ID是主键的类型
 *
 * @author youta
 */
@SuppressWarnings("all")
@Slf4j
public class BaseController<T1 extends BaseEntity, T2 extends IBaseService, ID> {

    /**
     * T1实体对应的Service,在子类控制器则不需要再次注入了
     */
    @Autowired
    public T2 entityService;

    private Class<T1> clazz = null;

    /**
     * Description:无参构造函数，获得T1的clazz对象
     */
    public BaseController() {
        Class clz = this.getClass();
        ParameterizedType type = (ParameterizedType) clz.getGenericSuperclass();
        Type[] types = type.getActualTypeArguments();
        clazz = (Class<T1>) types[0];
    }

    /**
     * Description:实体的分页查询，包括排序等,使用SpringData自己的对象接收分页参数
     *
     * @param page 分页参数对象
     * @return R
     */
    public Result<IPage> page(Pageable page) {
        Page<T1> entities = entityService.findAll(page);
        IPage IPage = new IPage(entities);
        return Result.ok(IPage);
    }

    /**
     * Description: 高级查询加分页功能
     *
     * @param page   分页参数对象
     * @param entity 包含高级查询条件的实体
     * @return R
     */
    public Result<IPage> advancedPageQuery(Pageable page, T1 entity) {
        Page<T1> entities = entityService.advancedQuery(entity, page);
        IPage IPage = new IPage(entities);
        return Result.ok(IPage);
    }

    /**
     * Description: 高级查询不分页
     *
     * @param entity 包含高级查询条件的实体
     * @return R
     */
    public Result<?> advancedQuery(T1 entity) {
        return Result.ok(entityService.advancedQuery(entity));
    }

    /**
     * Description: 同一个值,在多个字段中模糊查询,不分页
     *
     * @param q      模糊查询的值
     * @param fields 例如:"name,address,desc",对这三个字段进行模糊匹配
     * @return R
     */
    public Result<?> queryMultiField(String q, String fields) {
        if (StringUtils.isEmpty(fields)) {
            return Result.error(CodeEnum.SELECT_PROPERTY_NAME_NOT_EMPTY);
        }
        // 构造高级查询条件
        T1 be = buildQueryConditions(q, fields);
        return Result.ok(entityService.advancedQuery(be));
    }

    /**
     * Description:同一个值,在多个字段中模糊查询,分页
     *
     * @param q      模糊查询的值
     * @param fields 例如:"name,address,desc",对这三个字段进行模糊匹配
     * @param page   分页参数对象
     * @return R
     */
    public Result<?> queryMultiField(String q, String fields, Pageable page) {
        if (StringUtils.isEmpty(fields)) {
            return Result.error(CodeEnum.SELECT_PROPERTY_NAME_NOT_EMPTY);
        }
        // 构造高级查询条件
        T1 be = buildQueryConditions(q, fields);
        Page<T1> entities = entityService.advancedQuery(be, page);
        IPage iPage = new IPage(entities);
        return Result.ok(iPage);
    }

    /**
     * Description:根据id获取一个实体的信息
     *
     * @param id 主键
     * @return R
     */
    public Result<?> getById(ID id) {
        if (null == id) {
            return Result.error(CodeEnum.KEY_NOT_NULL);
        }

        Optional entity = entityService.findById(id);

        if (entity.isPresent()) {
            return Result.ok(entity.get());
        } else {
            return Result.error(CodeEnum.SELECT_NON_EXISTENT);
        }

    }

    /**
     * Description:保存一个实体，保存之前会做检查
     *
     * @param entity 要保存的实体对象
     * @return R
     */
    public Result<T1> save(T1 entity) {
        entityService.save(entity);
        return Result.ok(entity);

    }

    /**
     * 使用id来更新,如果属性空值,则不更新现有的值
     *
     * @param entity
     * @return R
     */
    public Result<?> updateById(T1 entity) {
        // 检查数据记录是否已经被删除了，被删除了，则不允许更新
        Optional<T1> entityOptional = entityService.findById(entity.getId());
        if (!entityOptional.isPresent()) {
            return Result.error(CodeEnum.UPDATE_NON_EXISTENT);
        } else {
            // 检查更新时间戳，避免用旧的数据更新数据库里的新数据
            LocalDateTime updateTime = entity.getUpdateTime();
            LocalDateTime dbUpdateTime = entityOptional.get().getUpdateTime();
            if (updateTime != null && updateTime.compareTo(dbUpdateTime) != 0) {
                return Result.error(CodeEnum.UPDATE_NEW_BY_OLD_NOT_ALLOWED);
            }
        }
        //检查业务key的存在性，不应该存在重复的业务key,此处不知道业务key是什么属性，可以在在service层实现，重写方法即可！

        if (null != entity.getId()) {
            updateEntity(entityOptional.get(), entity, false, "createTime", "updateTime");
        }
        return Result.ok(entity);

    }


    /**
     * Description:使用id删除指定的实体
     *
     * @param id 使用主键id
     * @return R
     */
    public Result<?> deleteById(ID id) {
        if (null == id) {
            return Result.error(CodeEnum.KEY_NOT_NULL);
        }

        //进行关联性检查,调用对应的方法
        // 在删除前用id到数据库查询一次,不执行空删除，不检查就可能会在数据库层面报错，尽量不让用户见到看不懂的信息
        Optional entity = entityService.findById(id);
        if (!entity.isPresent()) {
            return Result.error(CodeEnum.DELETE_NON_EXISTENT);
        }

        try {
            this.entityService.deleteById(id); // 关联关系可以在service层重写实现
        } catch (Exception e) {
            return Result.error(e.getMessage());
        }

        return Result.ok(id);
    }

    /**
     * Description:使用id的Set集合来删除指定的实体，不使用数组防止存在重复的数据
     *
     * @param entityIds 使用主键Set集合
     * @return R
     */
    public Result deleteByIds(Set<ID> entityIds) {
        if (CollectionUtils.isEmpty(entityIds)) {
            return Result.ok();
        }
        try {
            this.entityService.deleteAllById(entityIds); // 关联关系可以在service层重写实现
        } catch (Exception e) {
            return Result.error(e.getMessage());
        }

        return Result.ok();
    }


    /**
     * Description:检查操作的idd对应实体是否存在，因为多人操作，可能被其他人删除了！
     *
     * @param entityId 实体主键id
     * @return Boolean true存在，false 不存在
     */
    public Boolean chkEntityIdExist(ID entityId) {
        return null != entityId && entityService.existsById(entityId);
    }


    /**
     * 获取某个属性集合,去除重复,通常是前端选择需要,支持模糊匹配
     * 非法属性自动过滤掉
     *
     * @param property 驼峰式的属性
     * @param value    模糊查询的value值
     * @return R
     */
    public Result getPropertySet(String property, String value) {
        // 属性名不允许为空
        if (StringUtils.isEmpty(property)) {
            return Result.error(CodeEnum.SELECT_PROPERTY_NOT_EMPTY);
        }
        return Result.ok(entityService.advanceSearchProperty(property, value));
    }

    /**
     * Description:是否使用前端的数据实体的空值属性更新数据库中的
     * true，则用空置更新;fasle则不用空值更新,还允许部分更新
     *
     * @param dbEntity         后端查询数据库的实体
     * @param entity           前端传来的实体
     * @param useNull          是否使用前端的空置更新数据库的有值属性
     * @param ignoreProperties 忽略的属性,就是前端需要用空值更新后端的属性,比如常见的:更新时间,更新时间由框架或者数据库自己维护
     * @return T1
     */
    private T1 updateEntity(T1 dbEntity, T1 entity, Boolean useNull, String... ignoreProperties) {
        if (!useNull) {
            // 将前端来的数据空值用数据库中的值补上
            UpdatePoUtil.copyNullProperties(dbEntity, entity, ignoreProperties);
        }
        return (T1) entityService.update(entity);
    }

    /**
     * Description:根据查询值及多字段,来构建高级查询条件
     *
     * @param q      查询额值
     * @param fields 需要模糊匹配的字段，支持的分隔符：中英文的逗号分号，和中文的顿号！
     * @return T1 当前的泛型实体, 包含高级查询参数
     */
    @SneakyThrows
    private T1 buildQueryConditions(String q, String fields) {
        // 如果q不为空,则构造高级查询条件
        T1 entity = clazz.newInstance();
        if (!StrUtil.isAllEmpty(q, fields)) {
            Set<BaseSearchField> conditions = new HashSet<>();
            // 判断filds是一个字段还是多个字段,若是多个字段则进行切分
            // 切分属性值为集合，支持的分隔符：中英文的逗号分号，和中文的顿号！
            String[] rawFields = fields.split(",|;|、|，|；");
            for (String c : rawFields) {
                // 构建默认OR的多字段模糊查询
                BaseSearchField condition = new BaseSearchField();
                condition.setName(c);
                condition.setSearchType(SearchTypeEnum.FUZZY.getValue());
                condition.setVal(q);
                conditions.add(condition);
            }
            entity.setConditions(conditions);
            return entity;
        }
        return null;
    }

}
