package me.youm.frame.lock.aspect;

import lombok.extern.slf4j.Slf4j;
import me.youm.frame.common.context.ReactiveRequestContextHolder;
import me.youm.frame.common.exception.IdempotentException;
import me.youm.frame.common.utils.SecurityUtil;
import me.youm.frame.lock.annotation.ShoreLock;
import me.youm.frame.redis.reactive.service.ReactiveRedisService;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;

import javax.annotation.Resource;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.net.URI;
import java.time.Duration;


/**
 * @author youta
 */
@Aspect
@Slf4j
public class LockAspect {

    public static final String REQUEST_KEY = "RequestKey:";

    @Resource
    private ReactiveRedisService reactiveRedisService;

    @Pointcut("@annotation(me.youm.frame.lock.annotation.ShoreLock)")
    public void myPoint() {
    }

    @Around(value = "myPoint()")
    public Mono<?> apiIdempotentCheck(ProceedingJoinPoint pjp) throws Throwable {
        Signature signature = pjp.getSignature();
        MethodSignature methodSignature = (MethodSignature) signature;
        Method method = methodSignature.getMethod();
        Mono<?> mono = (Mono<?>) pjp.proceed();
        //无注解直接放行
        ShoreLock annotation = method.getAnnotation(ShoreLock.class);
        if (annotation == null) {
            return mono;
        }
        return ReactiveRequestContextHolder.getExchange()
                .flatMap(serverWebExchange -> check(serverWebExchange, pjp.getArgs().length == 0 ? null : pjp.getArgs()[0], method, mono)
                );
    }

    private Mono<?> check(ServerWebExchange serverWebExchange, Object arg, Method method, Mono<?> mono) {
        URI uri = serverWebExchange.getRequest().getURI();
        String token  = SecurityUtil.getHeaderToken(serverWebExchange);
        //获取请求参数
        String key;
        //接口方法名称
        String apiMethodName = method.getName();
        Parameter[] parameters = method.getParameters();
        //获取接口间隔时间
        Annotation[] declaredAnnotations = method.getDeclaredAnnotations();
        String apiName = getRequestName(uri, declaredAnnotations);
        String parameterName;
        if (parameters.length > 0) {
            Parameter parameter = parameters[0];
            //接口参数名称
            parameterName = parameter.getName();
            if (StringUtils.hasText(token)) {
                parameterName = parameterName + token;
            }
        } else {
            parameterName = "";
            if (StringUtils.hasText(token)) {
                parameterName = token;
            }
        }
        ShoreLock annotation = method.getAnnotation(ShoreLock.class);
        long l = annotation.expireTime();
        if (StringUtils.hasText(annotation.key())) {
            key = annotation.key();
        } else {
            key = REQUEST_KEY + ":" + apiName + ":" + apiMethodName + ":" + parameterName;
        }
        //如果当前key存在直接抛出异常
        return reactiveRedisService.hasKey(key).flatMap(aBoolean -> {
            if (aBoolean) {
                return Mono.error(new IdempotentException(annotation.info()));
            }
            if (arg != null && !"".equals(arg)) {
                return reactiveRedisService.set(key, arg.toString(), Duration.ofSeconds(l));
            } else {
                return reactiveRedisService.set(key, "1", Duration.ofSeconds(l));
            }
        }).flatMap(x -> {
            if (annotation.delKey()) {
                return reactiveRedisService.del(key);
            }
            return mono;
        });

    }

    private String getRequestName(URI requestURI, Annotation[] declaredAnnotations) {
        //获取接口请求类型以及接口api
        String apiName = null;
        for (Annotation declaredAnnotation : declaredAnnotations) {
            if (declaredAnnotation instanceof RequestMapping) {
                RequestMapping requestMapping = (RequestMapping) declaredAnnotation;
                String requestType = requestMapping.method()[0].name();
                apiName = requestType + ":" + requestURI;
            }
            if (declaredAnnotation instanceof DeleteMapping) {
                apiName = "DELETE:" + requestURI;
            }
            if (declaredAnnotation instanceof GetMapping) {
                apiName = "GET:" + requestURI;
            }
            if (declaredAnnotation instanceof PutMapping) {
                apiName = "PUT:" + requestURI;
            }
            if (declaredAnnotation instanceof PostMapping) {
                apiName = "POST:" + requestURI;
            }
        }
        return apiName;
    }

}
