package me.youm.frame.oss.util;

import cn.hutool.core.bean.OptionalBean;
import cn.hutool.core.util.IdUtil;
import com.aliyun.oss.ClientException;
import com.aliyun.oss.OSS;
import com.aliyun.oss.OSSException;
import com.aliyun.oss.model.Bucket;
import com.aliyun.oss.model.ObjectMetadata;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import me.youm.frame.common.enums.CodeEnum;
import me.youm.frame.common.exception.AliOssException;
import me.youm.frame.oss.props.OssProperties;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.stream.Collectors;


/**
 * @author liuqiu
 */
@Component
@Slf4j
@AllArgsConstructor
public class OssUtil {

    private final OSS ossClient;
    private final OssProperties ossProperties;
    private final static long FILE_LIMIT_SIZE = 1024 * 1024 * 20;
    private final static String BMP = ".bmp";
    private final static String GIF = ".gif";
    private final static String JPEG = ".jpeg";
    private final static String JPG = ".jpg";
    private final static String PNG = ".png";
    private final static String HTML = ".html";
    private final static String TXT = ".txt";
    private final static String VSD = ".vsd";
    private final static String PPT = ".ppt";
    private final static String PPTX = ".pptx";
    private final static String DOCX = ".docx";
    private final static String DOC = ".doc";
    private final static String XML = ".xml";

    /**
     * 上传图片
     *
     * @param file 文件
     * @return 路径
     */
    public String singleFileUpload(MultipartFile file, String bucketName, String fileDir) {
        if (file == null) {
            throw new AliOssException(HttpStatus.BAD_REQUEST.value(), "文件是空的");
        }
        if (file.getSize() < FILE_LIMIT_SIZE) {
            throw new AliOssException(CodeEnum.FILE_SIZE_TOO_LONG_ERROR.getCode(), CodeEnum.FILE_SIZE_TOO_LONG_ERROR.getMsg());
        }

        if (ossClient.doesBucketExist(bucketName)) {
            throw new AliOssException(CodeEnum.BUCKET_NOT_EXIST.getCode(), CodeEnum.BUCKET_NOT_EXIST.getMsg());
        }

        fileDir = StringUtils.hasText(fileDir) ?  fileDir + "/" : "";
        String originalFilename = file.getOriginalFilename();
        originalFilename = OptionalBean.ofNullable(originalFilename).get();
        String substring = originalFilename.substring(originalFilename.lastIndexOf(".")).toLowerCase();
        String name = IdUtil.simpleUUID() + substring;
        try {
            InputStream inputStream = file.getInputStream();
            this.uploadFile2Oss(inputStream, name, bucketName, fileDir);
            String endpoint = ossProperties.getEndpoint();
            String str = OptionalBean.ofNullable(endpoint)
                    .get().split("//")[1] + "/";
            return "https://" + bucketName + "." + str + fileDir + name;
        } catch (Exception e) {
            throw new AliOssException(CodeEnum.FILE_UPLOAD_ERROR.getCode(), CodeEnum.FILE_UPLOAD_ERROR.getMsg());
        }
    }

    /**
     * 上传图片获取fileUrl
     *
     * @param inputStream 流文件
     * @param fileName    文件名
     */
    private void uploadFile2Oss(InputStream inputStream, String fileName, String bucketName, String fileDir) {
        try {
            //创建上传Object的Metadata
            ObjectMetadata objectMetadata = new ObjectMetadata();
            objectMetadata.setContentLength(inputStream.available());
            objectMetadata.setCacheControl("no-cache");
            objectMetadata.setHeader("Pragma", "no-cache");
            objectMetadata.setContentType(getContentType(fileName.substring(fileName.lastIndexOf("."))));
            objectMetadata.setContentDisposition("inline;filename=" + fileName);
            //上传文件
            ossClient.putObject(bucketName, fileDir + fileName, inputStream, objectMetadata);
        } catch (OSSException ossException) {
            log.error("upload file failed by ossException", ossException);
            String errorCode = ossException.getErrorCode();
            throw new AliOssException(CodeEnum.ALI_YUN_OSS_ERROR.getCode(), errorCode);
        } catch (ClientException clientException) {
            log.error("upload file failed by clientException", clientException);
            String errorCode = clientException.getErrorCode();
            throw new AliOssException(CodeEnum.ALI_YUN_OSS_ERROR.getCode(), errorCode);
        } catch (IOException e) {
            log.error(e.getMessage(), e);
        } finally {
            try {
                if (inputStream != null) {
                    inputStream.close();
                }
            } catch (IOException e) {
                log.error("io error", e);
            }
        }
    }

    public static String getContentType(String FileNameExtension) {
        if (BMP.equalsIgnoreCase(FileNameExtension)) {
            return "image/bmp";
        }
        if (GIF.equalsIgnoreCase(FileNameExtension)) {
            return "image/gif";
        }
        if (JPEG.equalsIgnoreCase(FileNameExtension) ||
                JPG.equalsIgnoreCase(FileNameExtension) ||
                PNG.equalsIgnoreCase(FileNameExtension)) {
            return "image/jpeg";
        }
        if (HTML.equalsIgnoreCase(FileNameExtension)) {
            return "text/html";
        }
        if (TXT.equalsIgnoreCase(FileNameExtension)) {
            return "text/plain";
        }
        if (VSD.equalsIgnoreCase(FileNameExtension)) {
            return "application/vnd.visio";
        }
        if (PPTX.equalsIgnoreCase(FileNameExtension) ||
                PPT.equalsIgnoreCase(FileNameExtension)) {
            return "application/vnd.ms-powerpoint";
        }
        if (DOCX.equalsIgnoreCase(FileNameExtension) ||
                DOC.equalsIgnoreCase(FileNameExtension)) {
            return "application/msword";
        }
        if (XML.equalsIgnoreCase(FileNameExtension)) {
            return "text/xml";
        }
        return "image/jpeg";
    }


    /**
     * 多图片上传
     *
     * @param fileList fileList
     * @return string
     */
    public String batchFileUpload(List<MultipartFile> fileList, String bucketName, String fileDir) {
        String fileUrl;
        fileDir = StringUtils.isEmpty(fileDir) ? "" : fileDir + "/";
        StringBuilder photoUrl = new StringBuilder();
        for (int i = 0; i < fileList.size(); i++) {
            fileUrl = singleFileUpload(fileList.get(i), bucketName, fileDir);
            if (i == 0) {
                photoUrl = new StringBuilder(fileUrl);
            } else {
                photoUrl.append(",").append(fileUrl);
            }
        }
        return photoUrl.toString().trim();
    }


    public List<String> getAllBuckets() {
        List<Bucket> buckets = ossClient.listBuckets();
        buckets = OptionalBean.ofNullable(buckets).get();
        return buckets.stream().map(Bucket::getName).collect(Collectors.toList());
    }

    public void addBucket(String bucketName) {
        try {
            ossClient.createBucket(bucketName);
        } catch (OSSException ossException) {
            log.error("create bucket failed by ossException", ossException);
            String errorCode = ossException.getErrorCode();
            throw new AliOssException(CodeEnum.ALI_YUN_OSS_ERROR.getCode(), errorCode);
        } catch (ClientException clientException) {
            log.error("create bucket failed by clientException", clientException);
            String errorCode = clientException.getErrorCode();
            throw new AliOssException(CodeEnum.ALI_YUN_OSS_ERROR.getCode(), errorCode);
        }
    }
}
