package me.youm.frame.security.filter;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import me.youm.frame.common.constants.Oauth2Constant;
import me.youm.frame.common.constants.ShoreConstant;
import me.youm.frame.common.utils.ResponseUtil;
import me.youm.frame.common.utils.TokenUtil;
import me.youm.frame.redis.reactive.service.ReactiveRedisService;
import me.youm.frame.security.props.TokenProperties;
import org.springframework.core.Ordered;
import org.springframework.http.HttpStatus;
import org.springframework.http.server.reactive.ServerHttpResponse;
import org.springframework.stereotype.Component;
import org.springframework.web.server.ServerWebExchange;
import org.springframework.web.server.WebFilter;
import org.springframework.web.server.WebFilterChain;
import reactor.core.publisher.Mono;
import reactor.util.annotation.NonNull;

/**
 * @author youta
 */
@Slf4j
@Component
@AllArgsConstructor
public class SecurityFilter implements WebFilter, Ordered {
    private final TokenProperties properties;
    private final ReactiveRedisService reactiveRedisService;

    @Override
    @NonNull
    public Mono<Void> filter(@NonNull ServerWebExchange exchange, @NonNull WebFilterChain chain) {
        if (!properties.getEnable()) {
            return chain.filter(exchange);
        }
        //　如果在忽略的url里，则跳过
        String path = exchange.getRequest().getURI().getPath();
        String requestUrl = exchange.getRequest().getURI().getRawPath();
        if (ignore(path) || ignore(requestUrl)) {
            return chain.filter(exchange);
        }
        // 验证token是否有效
        ServerHttpResponse resp = exchange.getResponse();
        String headerToken = exchange.getRequest().getHeaders().getFirst(Oauth2Constant.HEADER_TOKEN);
        if (headerToken == null) {
            return unauthorized(resp, "没有携带Token信息");
        }
        String token = TokenUtil.getToken(headerToken);
        return reactiveRedisService.hasKey(Oauth2Constant.HEADER_TOKEN + ":login:token:" + token).flatMap(aBoolean -> {
            if (!aBoolean) {
                return unauthorized(resp, "登录超时，请重新登录");
            } else {
                return chain.filter(exchange);
            }
        });

    }

    private Mono<Void> unauthorized(ServerHttpResponse resp, String msg) {
        return ResponseUtil.webFluxResponseWriter(resp, ShoreConstant.JSON_UTF8, HttpStatus.UNAUTHORIZED, msg);
    }

    private boolean ignore(String path) {
        return properties.getIgnoreUrl().stream()
                .map(url -> url.replace("/**", ""))
                .anyMatch(path::startsWith);
    }

    @Override
    public int getOrder() {
        return -200;
    }

}
