package me.youm.frame.security.aspect;

import cn.dev33.satoken.exception.NotLoginException;
import cn.dev33.satoken.session.SaSession;
import cn.dev33.satoken.stp.StpUtil;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import me.youm.frame.common.constants.AuthConstant;
import me.youm.frame.common.entity.TokenInfo;
import me.youm.frame.common.enums.CodeEnum;
import me.youm.frame.common.exception.AuthorityException;
import me.youm.frame.common.exception.TokenException;
import me.youm.frame.common.utils.StringPool;
import me.youm.frame.security.annotation.PreAuth;
import me.youm.frame.security.enums.SuperAdmin;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;
import org.springframework.util.PatternMatchUtils;
import org.springframework.util.StringUtils;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collection;
import java.util.stream.Collectors;

/**
 * @author youta
 */
@Aspect
@Component
@AllArgsConstructor
@Slf4j
public class PreAuthAspect {

    private static final String ALL_PERMISSION = "*:*:*";

    @Around("@annotation(me.youm.frame.security.annotation.PreAuth)")
    public Object around(ProceedingJoinPoint point) throws Throwable {
        Signature signature = point.getSignature();
        MethodSignature methodSignature = (MethodSignature) signature;
        Method method = methodSignature.getMethod();
        PreAuth preAuth = method.getAnnotation(PreAuth.class);
        if (ObjectUtils.isEmpty(preAuth)) {
            return point.proceed();
        }
        if (hasPerm(preAuth.value())) {
            return point.proceed();
        } else {
            throw new AuthorityException(CodeEnum.FORBIDDEN.getMsg());
        }
    }


    /**
     * 验证用户是否具备某权限
     *
     * @param permission 权限字符串
     * @return 用户是否具备某权限
     */
    public boolean hasPerm(String permission) {
        SaSession tokenSession;
        try {
            tokenSession = StpUtil.getTokenSession();
        }catch (NotLoginException ex){
            throw new TokenException(ex.getMessage());
        }
        TokenInfo currentUser = tokenSession.getModel(AuthConstant.USER_SESSION_ID, TokenInfo.class);
        if (currentUser ==null) {
            return false;
        }
        if (!StringUtils.hasText(permission)) {
            return true;
        }
        // 如果用户是超级管理员，则直接跳过权限验证
        if (currentUser.getUserName().equalsIgnoreCase(SuperAdmin.ADMIN.getValue())) {
            return true;
        }
        return hasPermissions(Arrays.stream(String.valueOf(currentUser.getAuthorities()).split(StringPool.COMMA)).collect(Collectors.toList()), permission);
    }

    /**
     * 判断是否包含权限
     *
     * @param authorities 权限列表
     * @param permission  权限字符串
     * @return 用户是否具备某权限
     */
    private boolean hasPermissions(Collection<String> authorities, String permission) {
        return authorities.stream().filter(StringUtils::hasText)
                .anyMatch(x -> ALL_PERMISSION.contains(x) || PatternMatchUtils.simpleMatch(permission, x));
    }

}
