package com.zing.zalo.zbrowser.cache;

import android.util.Log;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Type;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

/**
 * Created by hoangdv4 on 25/03/2021.
 */

public class MiniProgramDataStorage {

    private String appId;
    private DiskLruCache diskCache;
    private int cacheExpireTime;
    private int diskCacheSize;
    private int expireTime = -1;
    private Map<String, InfoItem> itemMap;
    private ArrayList<ErrorListener> errorListeners;
    private boolean cacheMode = true;

    private static final String LOG_TAG ="hdv4_MiniApCache";

    public static final int CACHE_OUT_OF_MEMORY = 1000;
    public static final int CACHE_EXPIRED = 1001;
    public static final int CACHE_CREATE_FAILED = 1002;

    public static final int KEY_NOT_FOUND = 2000;
    public static final int KEY_EXPIRED = 2001;
    public static final int INVALID_VALUE = 3000;

    public MiniProgramDataStorage(String diskStorageDirectory, String appId, int diskStorageSize, int storageExpireTimeSeconds) {
        try {
            this.cacheExpireTime = storageExpireTimeSeconds;
            this.appId = appId;
            this.diskCacheSize = diskStorageSize;
            this.errorListeners = new ArrayList<>();
            this.diskCache = DiskLruCache.open(new File(diskStorageDirectory), 1, 1, diskStorageSize + 100, null, true);
            this.itemMap = buildMapFromDiskCache();
        } catch (IOException e) {
            Log.w(LOG_TAG, "Cannot create disk cache", e);
            sendErrorCode(CACHE_CREATE_FAILED, e.getMessage());
        }
    }

    public String get(String key){
        if(itemMap == null){
            itemMap = buildMapFromDiskCache();
        }
        if(itemMap != null && key != null && itemMap.containsKey(key)){
            InfoItem infoItem = itemMap.get(key);
            if(infoItem.expTime < System.currentTimeMillis()/1000){
                remove(key);
                sendErrorCode(KEY_EXPIRED, "KEY_EXPIRED");
                return null;
            }
            return infoItem.getValue();
        }
        sendErrorCode(KEY_NOT_FOUND, "KEY_NOT_FOUND");
        return null;
    }

    protected boolean put(String key, String value, long itemExpireTimeSeconds, boolean isPutToDisk){
        if(itemMap == null){
            itemMap = buildMapFromDiskCache();
        }
        if(itemMap != null && key != null){
            InfoItem infoItem = itemMap.get(key);
            if(infoItem == null){
                infoItem = new InfoItem(key, value, (int) (System.currentTimeMillis() / 1000 + itemExpireTimeSeconds));
            }else{
                infoItem.setValue(value);
            }
            itemMap.put(key, infoItem);
            if(getSizeFromMap(itemMap) >= diskCacheSize){
                if(cacheMode) {
                    removeValueNoUse();
                }else{
                    itemMap.remove(key);
                    sendErrorCode(CACHE_OUT_OF_MEMORY, "CACHE_OUT_OF_MEMORY");
                }
            }
            if(isPutToDisk) {
                String jsonData = getJsonFromMap(itemMap);
                return internalPut(jsonData);
            }else{
                return false;
            }
        }
        sendErrorCode(INVALID_VALUE, "INVALID_VALUE");
        return false;
    }

    public boolean put (String key, String value){
        return put(key, value, cacheExpireTime, true);
    }

    public boolean put (String key, String value, long expTime){
        return put(key, value, expTime, true);
    }

    protected boolean put (String key, String value, boolean isPutToDisk){
        return put(key, value, cacheExpireTime, isPutToDisk);
    }

    protected void putDataToDiskCache(){
        String jsonData = getJsonFromMap(itemMap);
        internalPut(jsonData);
    }

    public void remove(String key){
        remove(key, true);
    }

    protected void remove(String key, boolean isPutToDiskCache){
        if(itemMap == null){
            itemMap = buildMapFromDiskCache();
        }
        if(itemMap != null && key != null){
            if(itemMap.containsKey(key)){
                itemMap.remove(key);
            }
            if(isPutToDiskCache) {
                String jsonData = getJsonFromMap(itemMap);
                internalPut(jsonData);
            }
        }
    }

    public void clear(){
        if(itemMap == null){
            itemMap.clear();
        }
        removeFromDiskCache(appId);
    }

    public String getAll(){
        return getJsonFromMap(itemMap);
    }

    private void removeValueNoUse() {
        ArrayList<InfoItem> infoItems = new ArrayList<>();

        for(Map.Entry<String, InfoItem> e :itemMap.entrySet()){
            infoItems.add(e.getValue());
        }

        Collections.sort(infoItems, new Comparator<InfoItem>() {
            // Ưu tiên phần tử có lastAccess nhỏ nhất và createtime nhỏ nhất
            @Override
            public int compare(InfoItem o1, InfoItem o2) {
                if(o1.lastAccess > o2.lastAccess){
                    return 1;
                }else if(o1.lastAccess < o2.lastAccess){
                    return -1;
                }
                else if(o1.createTime > o2.createTime){
                    return 1;
                }
                else if(o1.createTime < o2.createTime) {
                    return -1;
                }
                else return 0;
            }
        });
        Iterator<InfoItem> intertator = infoItems.iterator();
        while (getSizeFromMap(itemMap) >= diskCacheSize && intertator.hasNext()){
            String removeKey = intertator.next().key;
            itemMap.remove(removeKey);
        }
    }

    private int getSizeFromMap(Map<String, InfoItem> map){
        String stringData = getJsonFromMap(map);
        return  stringData.getBytes().length;
    }


    private String getJsonFromMap(Map<String, InfoItem> map){
        Type mapType = new TypeToken<Map<String, InfoItem>>(){}.getType();
        return new Gson().toJson(map, mapType);
    }


    private Map<String, InfoItem> getMapFromJson(String JsonData){
        Type mapType = new TypeToken<Map<String, InfoItem>>(){}.getType();
        return new Gson().fromJson(JsonData, mapType);
    }

    private Map<String, InfoItem> buildMapFromDiskCache(){
        String dataJson = internalGet();
        if(dataJson != null){
            return getMapFromJson(dataJson);
        }
        return new HashMap<>();
    }

    private String internalGet(){
        CacheItem cacheItem = null;
        if (diskCache != null) {
            cacheItem = getFromDiskCache(appId);
        }
        if (cacheItem != null) {
            if(cacheItem.expireTime + cacheExpireTime > System.currentTimeMillis()/1000) {
                expireTime = cacheItem.expireTime;
                return new String(cacheItem.data);
            }
            removeFromDiskCache(appId);

        }
        sendErrorCode(CACHE_EXPIRED,"CACHE_EXPIRED");
        return null;
    }

    private boolean internalPut(String value){
        if(expireTime < 0){
            expireTime = (int) (System.currentTimeMillis() / 1000);
        }
        CacheItem cacheItem = new CacheItem(expireTime, "", value.getBytes(), "");
        if (diskCache != null) {
            return putToDiskCache(appId, cacheItem);
        }
        return false;
    }

    private void removeFromDiskCache(String key) {
        DiskLruCache cache = diskCache;
        if (cache == null || cache.isClosed()) {
            return;
        }
        try {
            String keyShorter = makeShortKey(key);
            cache.remove(keyShorter);
        } catch (Exception e) {
            Log.w(LOG_TAG, "Remove cache file from disk cache error", e);
            return;
        }
    }

    private CacheItem getFromDiskCache(String key) {
        DiskLruCache cache = diskCache;
        if (cache == null || cache.isClosed()) {
            return null;
        }
        InputStream inputStream = null;
        try {
            String keyShorter = makeShortKey(key);
            DiskLruCache.Snapshot snapshot = cache.get(keyShorter);
            if (snapshot == null) {
                return null;
            }
            ByteArrayOutputStream bufferData = new ByteArrayOutputStream((int) snapshot.getLength(0));
            inputStream = cache.isGzip() ? new GZIPInputStream(snapshot.getInputStream(0)) : snapshot.getInputStream(0);
            int byteRead;
            byte[] buffer = new byte[64 * 1024];
            while ((byteRead = inputStream.read(buffer)) != -1) {
                bufferData.write(buffer, 0, byteRead);
            }

            CacheItem result = CacheItem.deserialize(ByteBuffer.wrap(bufferData.toByteArray()));
            return result;
        } catch (Exception e) {
            Log.w(LOG_TAG, "Get cache file from disk cache error", e);
            return null;
        } finally {
            try {
                if (inputStream != null) {
                    inputStream.close();
                }
            } catch (Exception e) {
            }
        }
    }

    private boolean putToDiskCache(String key, CacheItem value) {
        DiskLruCache cache = diskCache;
        if (cache == null || cache.isClosed() || key == null || value == null) {
            return false;
        }
        try {
            String keyShorter = makeShortKey(key);
            DiskLruCache.Editor edit = cache.edit(keyShorter);
            OutputStream editorOut = cache.isGzip() ? new GZIPOutputStream(edit.newOutputStream(0)) : edit.newOutputStream(0);
            editorOut.write(value.serialize().array());
            editorOut.close();
            edit.commit();
            return true;
        } catch (Exception e) {
            Log.w(LOG_TAG, "Put data to disk cache error", e);
            return false;
        }
    }

    private static String makeShortKey(String key) {
        //we use md5 hash to make key shorter
        String var1 = "";
        try {
            MessageDigest var2 = MessageDigest.getInstance("MD5");
            var2.reset();
            var2.update(key.getBytes());
            byte[] var3 = var2.digest();
            String var4 = "";

            for (int var5 = 0; var5 < var3.length; ++var5) {
                var4 = Integer.toHexString(255 & var3[var5]);
                if (var4.length() == 1) {
                    var1 = var1 + "0" + var4;
                } else {
                    var1 = var1 + var4;
                }
            }
        } catch (Exception var6) {
            //ensure we still have a unique key
            var1 = String.valueOf(key.hashCode());
        }
        return var1;
    }

    public int getSize() {
        return getSizeFromMap(itemMap);
    }

    private class InfoItem{
        public String key;
        private String value;
        public int lastAccess;
        public int expTime;
        public int createTime;

        public InfoItem(String key, String value, int expTime) {
            this.key = key;
            this.value = value;
            this.expTime = expTime;
            this.lastAccess = (int) (System.currentTimeMillis() / 1000);
            this.createTime = (int) (System.currentTimeMillis() / 1000);
        }

        public String getValue(){
            this.lastAccess = (int) (System.currentTimeMillis() / 1000);
            return value;
        }

        public void setValue(String value){
            this.lastAccess = (int) (System.currentTimeMillis() / 1000);
            this.value = value;
        }
    }

    public void setOnErrorListener(ErrorListener errorListener){
        if(errorListeners != null && errorListener != null){
            errorListeners.add(errorListener);
        }
    }

    public void removeErrorListener(ErrorListener errorListener){
        if(errorListener != null && errorListener != null && errorListeners.contains(errorListener)) {
            errorListeners.remove(errorListener);
        }
    }

    void sendErrorCode(int errorCode, String errorMessage){
        for (ErrorListener errorListener : errorListeners){
            errorListener.onError(errorCode, errorMessage);
        }
    }

    interface ErrorListener{
        void onError(int errorCode, String errorMessage);
    }



}
