package com.zing.zalo.zbrowser.cache;

import android.util.Log;
import android.util.Pair;

import com.zing.zalo.leveldb.Iterator;
import com.zing.zalo.leveldb.LevelDB;

import org.rocksdb.Options;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksIterator;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Scanner;

/**
 * Created by hoangdv4 on 25/03/2021.
 */

public class MiniProgramDataStorage {

    private String appId;
    private DiskLruCache diskCache;
    private int cacheExpireTime;
    private long diskStorageSize;
    private int expireTime = -1;
    ArrayList<CPair> accessQueue;
    private ArrayList<ErrorListener> errorListeners;
    private boolean cacheMode = true;
    private String diskStorageDirectory;
    private long currentSize;

    private static final String LOG_TAG ="hdv4_MiniApCache";

    public static final int CACHE_OUT_OF_MEMORY = 1000;
    public static final int CACHE_EXPIRED = 1001;
    public static final int CACHE_CREATE_FAILED = 1002;
    public static final int KEY_NOT_FOUND = 2000;
    public static final int KEY_EXPIRED = 2001;
    public static final int INVALID_VALUE = 3000;


    private long DEFAULT_MEMSIZE = 5 << 20; //5MB
    LevelDB levelDB;

    public MiniProgramDataStorage(String diskStorageDirectory, String appId, long diskStorageSizeBytes, int storageExpireTimeSeconds) {
        try {
            Long createTime = getDBAccessimeDB();
            if(createTime == null){
                setDBAccessTime();
            }else{
                Long dbAccessTime = getDBAccessimeDB();
                if(dbAccessTime != null && (dbAccessTime+ storageExpireTimeSeconds*1000) > System.currentTimeMillis()){
                    clear();
                }
            }

            this.cacheExpireTime = storageExpireTimeSeconds;
            this.appId = appId;
            this.diskStorageSize = diskStorageSizeBytes;
            this.errorListeners = new ArrayList<>();
            this.diskStorageDirectory = diskStorageDirectory;
            this.accessQueue = new ArrayList<>();
//            Options options = new Options();
//            options.setCreateIfMissing(true);
//            options.setWriteBufferSize(DEFAULT_MEMSIZE);
//            options.setSstFileManager(new SstFileManager()) // -> quản lý file size
            this.levelDB = LevelDB.open((diskStorageDirectory+"/"+appId), LevelDB.configure().createIfMissing(true));

            Iterator iterator = levelDB.getIterator();
            for (iterator.seekToFirst(); iterator.isValid(); iterator.next()) {
                Pair<Long, String> p = parseValue(iterator.getValue().getBytes());
                if(p!= null){
                    String skey = iterator.getKey();//new String(iterator.key(), StandardCharsets.UTF_8);
                    long sizeValue = p.second.length();
                    currentSize += skey.length() + sizeValue;
                    accessQueue.add(new CPair(p.first, skey));
                }
            }
            iterator.close();

        } catch (Exception e) {
            Log.w(LOG_TAG, "Cannot create disk cache", e);
            sendErrorCode(CACHE_CREATE_FAILED, e.getMessage());
        }
    }

    public String get(String key){
        if(levelDB != null && key != null){
            try {
                return internalGet(key);
            } catch (Exception e) {
                e.printStackTrace();
                return null;
            }
        }
        return null;
    }

    public boolean put (String key, String value){
        if(levelDB != null && key != null && value != null) {
            long datasize = 0;
            int charSize = 2;
            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.N) {
                charSize = Character.SIZE;
            }
            datasize = (key.length() + value.length()) * charSize;

            Log.d("hoangdv4", "nope loop put: d:"+datasize + " c:"+ currentSize+" ss:"+diskStorageSize);
            while (datasize + currentSize > diskStorageSize){
                Log.d("hoangdv4", "put: d:"+datasize + " c:"+ currentSize+" ss:"+diskStorageSize);
                boolean isSuccess = internalDeleteLast();
                if (!isSuccess){
                    if(datasize + currentSize > diskStorageSize){
                        return false;
                    }
                    break;
                }
            }
            currentSize += datasize;
            return internalPut(key, value, System.currentTimeMillis());
        }
        return false;
    }

    public boolean delete(String key){
        if(levelDB != null) {
            return internalDelete(key);
        }
        return false;
    }

    public void clear(){
        if(accessQueue != null){
            accessQueue.clear();
        }
        try {
            levelDB.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
        deleteRecursive(new File(diskStorageDirectory, appId));
    }

    public long getSize() {
        return currentSize;
    }

    public void close(){
        setDBAccessTime();
        if(accessQueue != null)
            accessQueue.clear();
        if(levelDB != null) {
            try {
                levelDB.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    public void reloadBD(){
        Iterator iterator = levelDB.getIterator();
        if(accessQueue != null){
            accessQueue.clear();
        }else{
            accessQueue = new ArrayList<>();
        }
        for (iterator.seekToFirst(); iterator.isValid(); iterator.next()) {
            long time = parseOnlyTimeAccess(iterator.getValue().getBytes());
            String skey = iterator.getKey();//new String(iterator.getKeyByteArray(), StandardCharsets.UTF_8);
            accessQueue.add(new CPair(time, skey));
        }
        iterator.close();
    }

    public Map<String, String> getAll() {
        Map<String, String> has = new HashMap<>();
        try {
            Iterator iterator = levelDB.getIterator();
            for (iterator.seekToFirst(); iterator.isValid(); iterator.next()) {
                Pair<Long, String> p = parseValue(iterator.getValue().getBytes());
                if(p!= null){
                    has.put(iterator.getKey()/*new String(iterator.getKeyByteArray(), StandardCharsets.UTF_8)*/, p.second);
                }
            }
            iterator.close();
            return has;
        }catch (Exception e){
            return new HashMap<>();
        }
    }

    class CPair implements Serializable, Comparable<CPair> {
        long keyAccess;
        String key;

        public CPair(long keyAccess, String key) {
            this.keyAccess = keyAccess;
            this.key = key;
        }

        @Override
        public int compareTo(CPair o) {
            if (this.keyAccess < o.keyAccess) {
                return 1;
            } else if (this.keyAccess > o.keyAccess) {
                return -1;
            } else {
                return 0;
            }
        }
    }

    public void setOnErrorListener(ErrorListener errorListener){
        if(errorListeners != null && errorListener != null){
            errorListeners.add(errorListener);
        }
    }

    public void removeErrorListener(ErrorListener errorListener){
        if(errorListener != null && errorListener != null && errorListeners.contains(errorListener)) {
            errorListeners.remove(errorListener);
        }
    }

    void sendErrorCode(int errorCode, String errorMessage){
        for (ErrorListener errorListener : errorListeners){
            errorListener.onError(errorCode, errorMessage);
        }
    }

    interface ErrorListener{
        void onError(int errorCode, String errorMessage);
    }

    private void deleteRecursive(File fileOrDirectory) {
        if (fileOrDirectory.isDirectory())
            for (File child : fileOrDirectory.listFiles())
                deleteRecursive(child);

        fileOrDirectory.delete();
    }


    private boolean internalPut(String key, String value, long currentTime){
        byte[] btime = longToBytes(currentTime);
        byte[] bValue = value.getBytes();
        byte[] result = new byte[bValue.length + btime.length];
        try {
            System.arraycopy(btime, 0, result, 0, btime.length);
            System.arraycopy(bValue, 0, result, btime.length, bValue.length);
            Pair<Long, String> p = parseValue(levelDB.get(key.getBytes()));
            if(p!= null) {
                int index = Collections.binarySearch(accessQueue, new CPair(p.first, key));
                if(index >= 0) {
                    accessQueue.remove(index);
                }
            }
            levelDB.put(key, result);
            accessQueue.add(new CPair(currentTime, key));
            return true;
        }catch (Exception ex){
            return false;
        }
    }

    private String internalGet(String key) {
        Pair<Long, String> p = null;
        try {
            p = parseValue(levelDB.get(key.getBytes()));
            internalPut(key, p.second, System.currentTimeMillis());
            return p.second;
        } catch (Exception e) {
            return null;
        }

    }

    private boolean internalDelete(String key) {
        try {
            Pair<Long, String> p = parseValue(levelDB.get(key.getBytes()));
            long itemSize =  0;
            if(p!= null) {
                itemSize = key.getBytes().length+ p.second.getBytes().length;
                levelDB.delete(key);
                int index = Collections.binarySearch(accessQueue, new CPair(p.first, key));
                accessQueue.remove(index);
                currentSize -= itemSize;
                return true;
            }
            return false;
        }catch (Exception e){
            return false;
        }
    }

    private boolean internalDeleteLast() {
        try {
            String key = accessQueue.get(0).key;
            Pair<Long, String> p = parseValue(levelDB.get(key.getBytes()));
            long itemSize =  0;
            if(p!= null){
                itemSize = key.getBytes().length+ p.second.getBytes().length;
            }else{
                return false;
            }
            levelDB.delete(key);
            accessQueue.remove(0);
            currentSize -= itemSize;
            return true;
        }catch (Exception e){
            return false;
        }

    }

    private Pair<Long, String> parseValue(byte[] rawvalue) {
        if(rawvalue != null) {
            int longByte = 8;
            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.N) {
                longByte = Long.BYTES;
            }

            int valueSize = rawvalue.length - longByte;
            byte[] btime = new byte[longByte];
            byte[] bValue = new byte[valueSize];

            System.arraycopy(rawvalue, 0, btime, 0, longByte);
            System.arraycopy(rawvalue, longByte, bValue, 0, valueSize);

            return new Pair<>(bytesToLong(btime), new String(bValue, StandardCharsets.UTF_8));
        }
        return null;
    }

    private long parseOnlyTimeAccess(byte[] rawvalue) {
        int longByte = 8;
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.N) {
            longByte = Long.BYTES;
        }
        byte[] btime = new byte[longByte];
        System.arraycopy(rawvalue, 0, btime, 0, longByte);
        return bytesToLong(btime);
    }


    private static byte[] longToBytes(long l) {
        int longByte = 8;
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.N) {
            longByte = Long.BYTES;
        }

        byte[] result = new byte[longByte];
        for (int i = longByte - 1; i >= 0; i--) {
            result[i] = (byte)(l & 0xFF);
            l >>= Byte.SIZE;
        }
        return result;
    }

    private static long bytesToLong(final byte[] b) {
        int longByte = 8;
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.N) {
            longByte = Long.BYTES;
        }

        long result = 0;
        for (int i = 0; i < longByte; i++) {
            result <<= Byte.SIZE;
            result |= (b[i] & 0xFF);
        }
        return result;
    }


    private boolean setDBAccessTime() {
        try {
            FileWriter myWriter = new FileWriter(diskStorageDirectory+"/"+appId+"/properties");
            myWriter.write(String.valueOf(System.currentTimeMillis()));
            myWriter.close();
            return true;
        } catch (IOException e) {
            System.out.println("An error occurred.");
            e.printStackTrace();
            return false;
        }
    }


    private Long getDBAccessimeDB(){
        try {
            File myObj = new File(diskStorageDirectory+"/"+appId+"/properties");
            Scanner myReader = new Scanner(myObj);
            while (myReader.hasNextLine()) {
                String data = myReader.nextLine();
                long createTime = Long.parseLong(data);
                return createTime;
            }
            myReader.close();
        } catch (Exception e) {
            System.out.println("An error occurred.");
            e.printStackTrace();
            return null;
        }
        return null;
    }




}
