/*
 * Copyright (c) 2012-2018 by Zalo Group.
 * All Rights Reserved.
 */
package com.zing.zalo.zbrowser.downloader;

import com.zing.zalo.zbrowser.ZBrowserCore;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 *
 * @author datbt
 */
public final class HttpRequest {

	public static final String USER_AGENT = String.format("%s ZBrowser/%s Zalo android/%d", System.getProperty("http.agent"), ZBrowserCore.VERSION, 500);

	private static final Map<String, String> DEFAULT_REQUEST_HEADERS;

	static {
		DEFAULT_REQUEST_HEADERS = new ConcurrentHashMap<>();
		DEFAULT_REQUEST_HEADERS.put("User-Agent", USER_AGENT);
		DEFAULT_REQUEST_HEADERS.put("Accept-Encoding", "gzip");
		DEFAULT_REQUEST_HEADERS.put("zbrowser", "1");
	}

	public final HttpUrl url;

	private final HttpHeader requestHeaders;

	private final ByteBuffer requestData;

	private State state;

	private String errorMessage;

	private HttpResponse response;

	public final HttpResponse.Callback responseCallback;

	public final int redirectCount;

	public final Map<String, String> userRequestHeaders;

	private ZSSLConnection sslConnection;

	public HttpRequest(HttpUrl url, Map<String, String> requestHeaders, HttpResponse.Callback responseCallback, int redirectCount) {
		this.url = url;
		if (url == null) {
			throw new IllegalArgumentException("Http url is null");
		}
		this.requestHeaders = new HttpHeader(DEFAULT_REQUEST_HEADERS);
		this.requestHeaders.setHeaders(requestHeaders);
		this.requestHeaders.setHeader("Host", url.domain);

		this.requestData = ByteBuffer.wrap(buildRawHttpRequest().getBytes());
		this.state = State.INIT;
		this.errorMessage = null;
		this.response = null;
		this.responseCallback = responseCallback;
		this.redirectCount = redirectCount;
		this.userRequestHeaders = requestHeaders;

		this.sslConnection = null;
	}

	public boolean isSecure() {
		return url.scheme.equals(HttpUrl.SCHEME_HTTPS);
	}

	public State processSSLData() {
		if (!isSecure()) {
			//not https request - wrong call
			return State.ERROR;
		}
		try {
			sslConnection.checkForData();
		} catch (IOException e) {
			e.printStackTrace();
			if (state != State.DONE) {
				setErrorState(e.getMessage());
			}
		}
		return state;
	}

	public boolean isSSLConnectionInitialized() {
		return this.sslConnection != null;
	}

	public void setSSLConnection(ZSSLConnection sslConnection) {
		if (this.sslConnection == null) {
			this.sslConnection = sslConnection;
		}
	}

	public ZSSLConnection getSLLConnection() {
		return this.sslConnection;
	}

	public State getState() {
		return state;
	}

	public State setState(State state) {
		if (state == null) {
			return null;
		}
		this.state = state;
		return state;
	}

	public State setErrorState(String errorMessage) {
		this.state = State.ERROR;
		this.errorMessage = errorMessage;
		return this.state;
	}

	public String getErrorMessage() {
		return errorMessage;
	}

	public HttpResponse getResponse() {
		if (response == null) {
			response = new HttpResponse(this);
		}
		return response;
	}

	public boolean isSendRequestDataDone() {
		return requestData.remaining() == 0;
	}

	public int send(SocketChannel socket) throws IOException {
		if (isSecure()) {
			//https request, sslconnection should handle
			return -1;
		}
		if (isSendRequestDataDone()) {
			return 0;
		}
		System.err.println("header to send: " + System.currentTimeMillis() + "\n" + new String(requestData.array()));
		return socket.write(requestData);
	}

//	private void setHeader(String name, String value) {
//		if (name == null || name.isEmpty() || value == null || value.isEmpty()) {
//			return;
//		}
//		this.requestHeaders.put(name.trim().toLowerCase(), value.trim());
//	}
//
//	public List<String> getListHeaderNames() {
//		List<String> listHeaders = new ArrayList<>();
//		for (Map.Entry<String, String> entry : this.requestHeaders.entrySet()) {
//			String headerName = entry.getKey();
//			listHeaders.add(headerName);
//		}
//		return listHeaders;
//	}
//
//	public String getHeader(String name) {
//		return requestHeaders.get(name);
//	}

	public String buildRawHttpRequest() {
		StringBuilder reqBuilder = new StringBuilder();
		reqBuilder.append("GET ").append(url.uri).append(" ").append(HTTP.VERSION_1_1).append(HTTP.CRLF);
		List<String> listHeaderNames = this.requestHeaders.getListHeaderNames();
		for  (String headerName : listHeaderNames) {
			String value = this.requestHeaders.getHeader(headerName);
			reqBuilder.append(headerName).append(": ").append(value).append(HTTP.CRLF);
		}
		reqBuilder.append(HTTP.CRLF);

		return reqBuilder.toString();
	}

	public static enum State {
		INIT, SENDING_REQ, RECV_HEADER, RECV_DATA, DONE, ERROR
	}

}
