package com.zing.zalo.zbrowser.cache;

import android.util.Log;
import android.util.Pair;

import com.zing.zalo.leveldb.Iterator;
import com.zing.zalo.leveldb.LevelDB;

import java.io.IOException;
import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by hoangdv4 on 25/03/2021.
 */

public class MiniProgramDataStorage {

    private String appId;
    private long diskStorageSize;
    ArrayList<CPair> accessQueue;
    private ArrayList<ErrorListener> errorListeners;
    private String diskStorageDirectory;
    private long currentSize;
    private static final String LOG_TAG ="dungnn";

    public static final int CACHE_OUT_OF_MEMORY = 1000;
    public static final int CACHE_EXPIRED = 1001;
    public static final int CACHE_CREATE_FAILED = 1002;
    public static final int KEY_NOT_FOUND = 2000;
    public static final int KEY_EXPIRED = 2001;
    public static final int INVALID_VALUE = 3000;

    private long DEFAULT_MEMSIZE = 5 << 20; //5MB
    private LevelDB levelDB;
    private int longByte = 8;

    public MiniProgramDataStorage(String diskStorageDirectory, String appId, long diskStorageSizeBytes) {
        this.appId = appId;
        this.errorListeners = new ArrayList<>();
        this.diskStorageDirectory = diskStorageDirectory;
        this.accessQueue = new ArrayList<>();
        if(diskStorageSizeBytes <= 0){
            this.diskStorageSize = DEFAULT_MEMSIZE;
        }else {
            this.diskStorageSize = diskStorageSizeBytes;
        }
        createDB();
    }

    public String get(String key){
        if(levelDB != null && key != null){
            try {
                return internalGet(key);
            } catch (Exception e) {
                e.printStackTrace();
                return null;
            }
        }
        return null;
    }

    public boolean put (String key, String value){
        if(levelDB != null && key != null && value != null) {
            return internalPut(key, value, System.currentTimeMillis());
        }
        return false;
    }

    public boolean delete(String key){
        if(levelDB != null) {
            return internalDelete(key);
        }
        return false;
    }

    public void clear(){
        if(accessQueue != null){
            accessQueue.clear();
        }
        try {
            if(levelDB!= null) {
                levelDB.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        currentSize = 0;
        createDB();
    }

    public long getSize() {
        return currentSize;
    }

    public void close(){
        if(accessQueue != null) {
            accessQueue.clear();
        }
        if(errorListeners != null){
            errorListeners.clear();
        }
        if(levelDB != null) {
            try {
                levelDB.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    public Map<String, String> getAll() {
        Map<String, String> has = new HashMap<>();
        try {
            Iterator iterator = levelDB.getIterator();
            for (iterator.seekToFirst(); iterator.isValid(); iterator.next()) {
                Pair<Long, String> p = /*parseValue(levelDB.get(iterator.getKeyByteArray()))*/parseValue(iterator.getValueByteArray()); //TODO:: Chuyển
                if(p!= null){
                    has.put(iterator.getKey(), p.second);
                }
            }
            iterator.close();
            return has;
        }catch (Exception e){
            return new HashMap<>();
        }
    }

    class CPair implements Serializable, Comparable<CPair> {
        long keyAccess;
        String valuseKey;

        public CPair(long keyAccess, String key) {
            this.keyAccess = keyAccess;
            this.valuseKey = key;
        }

        @Override
        public int compareTo(CPair o) {
            if (this.keyAccess < o.keyAccess) {
                return -1;
            } else if (this.keyAccess > o.keyAccess) {
                return 1;
            } else {
                return 0;
            }
        }
    }

    public void setOnErrorListener(ErrorListener errorListener){
        if(errorListeners != null && errorListener != null){
            errorListeners.add(errorListener);
        }
    }

    public void removeErrorListener(ErrorListener errorListener){
        if(errorListener != null && errorListener != null && errorListeners.contains(errorListener)) {
            errorListeners.remove(errorListener);
        }
    }

    private void sendErrorCode(int errorCode, String errorMessage){
        for (ErrorListener errorListener : errorListeners){
            errorListener.onError(errorCode, errorMessage);
        }
    }

    interface ErrorListener{
        void onError(int errorCode, String errorMessage);
    }

    private boolean createDB(){
        try{
            LevelDB.Configuration sConfiguration = LevelDB.configure();
            sConfiguration.createIfMissing(true);
            sConfiguration.setCompressionType(LevelDB.Configuration.CompressionType.kNoCompression);

            this.levelDB = LevelDB.open((diskStorageDirectory+"/"+appId), sConfiguration);
            Iterator iterator = levelDB.getIterator();
            for (iterator.seekToFirst(); iterator.isValid(); iterator.next()) {
                byte[]key = iterator.getKeyByteArray();
                Pair<Long, String> p = /*parseValue(levelDB.get(iterator.getKeyByteArray()));*/parseValue(iterator.getValueByteArray()); //TODO:: Chuyeenr
                if(p!= null){
                    String skey = iterator.getKey();
                    currentSize += skey.getBytes().length + p.second.getBytes().length;
                    accessQueue.add(new CPair(p.first, skey));
                }
            }

            iterator.close();
            Collections.sort(accessQueue, CPair::compareTo);
            return true;
        } catch (Exception e) {
            Log.w(LOG_TAG, "Cannot create disk cache", e);
            sendErrorCode(CACHE_CREATE_FAILED, e.getMessage());
            return false;
        }
    }


    private boolean internalPut(String key, String value, long currentTime){
        byte[] btime = longToBytes(currentTime);
        byte[] bValue = value.getBytes();
        byte[] result = new byte[bValue.length + btime.length];

        try {
            System.arraycopy(btime, 0, result, 0, btime.length);
            System.arraycopy(bValue, 0, result, btime.length, bValue.length);
            Pair<Long, String> p = parseValue(levelDB.get(key.getBytes()));
            if(p!= null) {
                int index = Collections.binarySearch(accessQueue, new CPair(p.first, key), CPair::compareTo);
                if(index >= 0) {
                    accessQueue.remove(index);
                }
                levelDB.delete(key);
                long deleteDataSize = key.getBytes().length + value.getBytes().length;
                currentSize -= deleteDataSize;
            }

            long datasize;
            datasize = key.getBytes().length + value.getBytes().length;
            while (datasize + currentSize > diskStorageSize){
                boolean isSuccess = internalDeleteLast();
                if (!isSuccess){
                    if(datasize + currentSize > diskStorageSize){
                        return false;
                    }
                    break;
                }
            }

            currentSize += datasize;
            levelDB.put(key, result);
            accessQueue.add(new CPair(currentTime, key));
            return true;
        }catch (Exception ex){
            return false;
        }
    }

    private String internalGet(String key) {
        Pair<Long, String> p;
        try {
            p = parseValue(levelDB.get(key.getBytes()));
            if(p!= null) {
                internalPut(key, p.second, System.currentTimeMillis());
                return p.second;
            }
            return null;
        } catch (Exception e) {
            return null;
        }

    }

    private boolean internalDelete(String key) {
        try {
            Pair<Long, String> p = parseValue(levelDB.get(key.getBytes()));
            long itemSize =  0;
            if(p!= null) {
                itemSize = key.getBytes().length+ p.second.getBytes().length;
                levelDB.delete(key);
                int index = Collections.binarySearch(accessQueue, new CPair(p.first, key));
                if(index >= 0) {
                    accessQueue.remove(index);
                }
                currentSize -= itemSize;
                return true;
            }
            return false;
        }catch (Exception e){
            return false;
        }
    }

    private boolean internalDeleteLast() {
        try {
            String key = accessQueue.get(0).valuseKey;
            Pair<Long, String> p = parseValue(levelDB.get(key.getBytes()));
            long itemSize =  0;
            if(p!= null){
                itemSize = key.getBytes().length + p.second.getBytes().length;
            }else{
                return false;
            }
            levelDB.delete(key);
            accessQueue.remove(0);
            currentSize -= itemSize;
            return true;
        }catch (Exception e){
            return false;
        }
    }

    private Pair<Long, String> parseValue(byte[] rawvalue) {
        if(rawvalue != null) {
            int valueSize = rawvalue.length - longByte;
            byte[] btime = new byte[longByte];
            byte[] bValue = new byte[valueSize];

            System.arraycopy(rawvalue, 0, btime, 0, longByte);
            System.arraycopy(rawvalue, longByte, bValue, 0, valueSize);

            return new Pair<>(bytesToLong(btime), new String(bValue, StandardCharsets.UTF_8));
        }
        return null;
    }

    private byte[] longToBytes(long l) {
        byte[] result = new byte[longByte];
        for (int i = longByte - 1; i >= 0; i--) {
            result[i] = (byte)(l & 0xFF);
            l >>= Byte.SIZE;
        }
        return result;
    }

    private Long bytesToLong(final byte[] b) {
        long result = 0;
        for (int i = 0; i < longByte; i++) {
            result <<= Byte.SIZE;
            result |= (b[i] & 0xFF);
        }
        return result;
    }

}