package com.zing.zalo.zbrowser;

import android.content.Context;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.text.TextUtils;
import android.util.SparseArray;
import android.webkit.WebResourceRequest;
import android.webkit.WebResourceResponse;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.collection.LruCache;

import com.zing.zalo.zbrowser.cache.ConfigCache;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import timber.log.Timber;

/**
 * Created by lamhx on 2/26/19.
 */
public class ZBrowserPreload {
    private static final String TAG = ZBrowserPreload.class.getSimpleName();

//    private LinkJumpSocketPreloadManager<String> linkJumpSocketManager;
    private ZBrowser zBrowser = null;

    private int MAX_SIZE_LINKJUMP_MAP = 10;

    private AtomicBoolean initialized;
    private final Object ZBROWSER_LOCKER = new Object();

    private int TIME_DELAY_REQUEST = 1000;
    private static final int MSG_REQUEST_JUMPLINK = 1;
    private static final int MSG_REQUEST_QUEUE = 2;

    private HandlerThread mHandlerThread;
    private Handler mHandler;
    private SparseArray<String> mWebViewRequestQueue;
    private LruCache<String, String> mUrlLinkMap;

    private String mDirectoryCache;
    private ZBrowserCore.InvestigateListener investigateListener;
    @NonNull
    private ZBrowserPreloadCallback zBrowserCallback;

    private Hashtable<String, ArrayList<ZBrowserCore.PreloadCallback>> mappingPreloadCallback;

    private final Object mappingCallbackLocker = new Object();


    public ZBrowserPreload(@NonNull ZBrowserPreloadCallback zBrowserCallback, String directory, final Context context) {
        initialized = new AtomicBoolean(false);

        mappingPreloadCallback = new Hashtable<>();
        this.zBrowserCallback = zBrowserCallback;
        this.mDirectoryCache = directory;
        mUrlLinkMap = new LruCache<>(MAX_SIZE_LINKJUMP_MAP);
        mWebViewRequestQueue = new SparseArray<>();
        mHandlerThread = new HandlerThread(TAG);
        mHandlerThread.start();
        mHandler = new Handler(mHandlerThread.getLooper()) {
            @Override
            public void handleMessage(Message msg) {
                switch (msg.what) {
                    case MSG_REQUEST_QUEUE:
                        if (msg.obj instanceof PreloadRequestBundle) {
                            final PreloadRequestBundle request = (PreloadRequestBundle) msg.obj;
                            if (mWebViewRequestQueue.get(request.hashCode()) == null) {
                                mWebViewRequestQueue.put(request.hashCode(), request.keyRequest);
                            }
                            if (!mHandler.hasMessages(MSG_REQUEST_JUMPLINK)) {
                                mHandler.sendEmptyMessageDelayed(MSG_REQUEST_JUMPLINK, TIME_DELAY_REQUEST);
                            }
                        }
                        break;
                    case MSG_REQUEST_JUMPLINK:
                        int size = mWebViewRequestQueue.size();
                        if (size > 0) {
                            int i = 0;
                            List<String> linkJumpList = new ArrayList<>();
                            for (i = 0; i < size; i++) {
                                final String request = mWebViewRequestQueue.valueAt(i);
                                if (request != null) {
                                    linkJumpList.add(request);
                                    mUrlLinkMap.put(request, "");
                                    Timber.tag(TAG).d("request jumpLink: %s", request);
                                }
                            }
                            ZBrowser zbrowser = getZBrowser();
                            if (zbrowser != null) {
                                zbrowser.load(linkJumpList, zBrowserDownloadCallback, null, context);
                            }
                        }
                        mWebViewRequestQueue.clear();
                        break;
                }
            }
        };
        init(context);
    }

    public void executeTask(Runnable task) {
        if (mHandler != null) {
            mHandler.post(task);
        }
    }

    private void init(final Context context) {
        try {

            mUrlLinkMap.evictAll();
            ConfigCache configCache = null;

            String config = zBrowserCallback.getCurrentCacheConfigAtLocal();
            if (TextUtils.isEmpty(config)) {
                //fake request to get zbrowser config only
                executeTask(new Runnable() {
                    @Override
                    public void run() {
                        synchronized (ZBROWSER_LOCKER) {
                            requestNewConfig(context);
                        }
                    }
                });

                return;
            }
            configCache = ConfigCache.getConfigFromJson(config);

            if (configCache == null) {
                configCache = ConfigCache.DEFAULT_CONFIG;
            }

            zBrowser = new ZBrowser(configCache, mDirectoryCache, zBrowserCallback, context);
            zBrowser.setInvestigateListener(investigateListener);
            initialized.set(true);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void requestNewConfig(final Context context){
        //fake request to get zbrowser config only
        String fakeReq = "https://media.zalo.me/";
        zBrowserCallback.getConfig(fakeReq, new ZBrowserPreloadCallback.AsyncCallback<JSONObject>() {
            @Override
            public void onError(int code, String exception) {
                Timber.tag(TAG).e("request jump preload fail: %d - %s", code, exception);
            }
            /*
            Chuyển callback từ json về string, trên app đã parse trước và trả về qua callback.
             */
            @Override
            public void onSuccess(JSONObject zbrowserConfig) {
                try {
//                    JSONObject data = jsonObject.getJSONObject("data");
//                    update zbrowser config if exist in jump response
//                    JSONObject zbrowserConfig = data.optJSONObject("zbrowser_config");
                    if (zbrowserConfig != null) {
                        zBrowserCallback.updateCacheConfigIntoLocal(zbrowserConfig.toString());
                    }
                    ConfigCache config = ConfigCache.getConfigFromJson(zbrowserConfig);
                    zBrowser = new ZBrowser(config, mDirectoryCache, zBrowserCallback, context);
                    zBrowser.setInvestigateListener(investigateListener);
                    initialized.set(true);
                } catch (Exception e) {

                }
            }
        });
    }

    public void setInvestigateListener(final ZBrowserCore.InvestigateListener listener) {
        this.investigateListener = listener;
        executeTask(new Runnable() {
            @Override
            public void run() {
                synchronized (ZBROWSER_LOCKER) {
                    if (zBrowser != null) {
                        zBrowser.setInvestigateListener(listener);
                    }
                }
            }
        });
    }

    @Nullable
    public ZBrowser getZBrowser() {
        return zBrowser;
    }

    @Nullable
    public String handleJsBridge(String jsExtraData) {
        ZBrowser zbrowser = getZBrowser();
        if (zbrowser != null) {
            return zbrowser.handleJsBridge(jsExtraData).toString();
        }
        return null;
    }

    private void load(final String url, final ZBrowser.DownloadCallback callback, final PreloadParam preloadParam, final Context context) {
        executeTask(new Runnable() {
            @Override
            public void run() {
                if (isNeedToPreload(url)) {
                    Timber.tag(TAG).d("request load : %s", url);
                    ZBrowser zbrowser = getZBrowser();
                    if (zbrowser != null) {
                        zbrowser.load(url, callback, preloadParam, context);
                    }
                }
            }
        });
    }

    public void clearCache() {
        executeTask(new Runnable() {
            @Override
            public void run() {
                ZBrowser zbrowser = getZBrowser();
                if (zbrowser != null) {
                    zbrowser.clearCache();
                }
            }
        });
    }

    public void reset(final Context context) {
        executeTask(new Runnable() {
            @Override
            public void run() {
                synchronized (ZBROWSER_LOCKER) {
                    if (zBrowser != null) {
                        zBrowser.cleanAllDataInCache(context);
                        zBrowser.destroy();
                    }
                    zBrowser = null;
                }
            }
        });
    }

    @RequiresApi(api = 21)
    public WebResourceResponse getWebResourceResponse(WebResourceRequest request, Context context) {
        ZBrowser zbrowser = getZBrowser();
        if (zbrowser != null) {
            return zbrowser.getWebResourceResponse(request, context);
        }
        return null;
    }

    public WebResourceResponse getWebResourceResponse(String url, Context context) {
        ZBrowser zbrowser = getZBrowser();
        if (zbrowser != null) {
            return zbrowser.getWebResourceResponse(url, context);
        }
        return null;
    }

    public boolean isNeedToPreload(String url) {
        try {
            ZBrowser zbrowser = getZBrowser();
            if (zbrowser != null) {
                ConfigCache cacheConfig = zbrowser.getCacheConfig();
                return cacheConfig.isNeedToPreload(url);
            }
        } catch (Exception ignored) {
        }
        return false;
    }

    public boolean isLoaded(String url) {
//        return false;
        return (mUrlLinkMap.get(url) != null);
    }

    public void preloadUrl(final ZBrowserCore.PreloadCallback preloadCallback, final String url) {
        if (initialized.get() && !isNeedToPreload(url)) {
            return;
        }
        executeTask(new Runnable() {
            @Override
            public void run() {
                try {
                    if (isNeedToPreload(url)) {
                        if(isLoaded(url)) {
                            if(preloadCallback != null && !TextUtils.isEmpty(mUrlLinkMap.get(url))) {
                                preloadCallback.onDone(url);
                            }
                            return;
                        }
                        mHandler.sendMessage(mHandler.obtainMessage(MSG_REQUEST_QUEUE, new PreloadRequestBundle(url, preloadCallback)));
                    }
                } catch (Exception e) {
                }
            }
        });
    }

    public void preloadUrl(@NonNull final String request) {
        if (initialized.get() && !isNeedToPreload(request)) {
            return;
        }
        executeTask(new Runnable() {
            @Override
            public void run() {
                try {
                    if (true ) {
                        mHandler.sendMessage(mHandler.obtainMessage(MSG_REQUEST_QUEUE, request));
                    }
                } catch (Exception e) {
                }
            }
        });
    }

    private ZBrowser.DownloadCallback zBrowserDownloadCallback = new ZBrowser.DownloadCallback() {
        @Override
        public void onDataDownloading(String url, int percent, boolean isDone) {
            if(percent >= 100 || isDone) {
                synchronized (mappingCallbackLocker) {
                    ArrayList<ZBrowserCore.PreloadCallback> callbackList = mappingPreloadCallback.remove(url);
                    if (callbackList == null) return;
                    for (ZBrowserCore.PreloadCallback preloadCallback : callbackList) {
                        if (preloadCallback != null) {
                            preloadCallback.onDone(url);
                        }
                    }
                }
            }
        }

        @Override
        public void onStart(String url) {

        }

        @Override
        public void onError(String url) {
            mUrlLinkMap.remove(url);
            synchronized (mappingCallbackLocker) {
                ArrayList<ZBrowserCore.PreloadCallback> callbackList = mappingPreloadCallback.remove(url);
                if (callbackList == null) return;
                for (ZBrowserCore.PreloadCallback preloadCallback : callbackList) {
                    if (preloadCallback != null) {
                        preloadCallback.onFailed(url);
                    }
                }
            }
        }
    };

    @Override
    protected void finalize() throws Throwable {
        if (mHandlerThread != null) {
            mHandlerThread.quit();
        }
        super.finalize();
    }

    public void setCacheConfig(JSONObject zbrowserConfig, Context context) {
        if(zBrowser!= null)
            zBrowser.setCacheConfig(zbrowserConfig, context);
    }

    public interface ZBrowserPreloadCallback {
        void getConfig(
                String requests,
                @NonNull AsyncCallback<JSONObject> callback);

        String getCurrentCacheConfigAtLocal();

        void updateCacheConfigIntoLocal(String config);

        interface AsyncCallback<Result> {
            void onError(int code, String exception);

            void onSuccess(Result result);
        }
    }
}
