package com.zing.zalo.zbrowser.cache;

import androidx.annotation.NonNull;
import android.text.TextUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.URI;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

public class ConfigCache {

    private static final int HTML_MEM_SIZE = 524288; // 1024 * 512;
    private static final int STATIC_MEM_SIZE = HTML_MEM_SIZE;
    private static final int PHOTO_MEM_SIZE = -1;
    private static final int HTML_DISK_SIZE = 5242880; //5 * 1024 * 1024;
    private static final int STATIC_DISK_SIZE = HTML_DISK_SIZE;
    private static final int PHOTO_DISK_SIZE = 10485760; // 10 * 1024 * 1024;

    private static final int JUMP_MEM_SIZE = 32768; // 1024 * 32;
    private static final int JUMP_DISK_SIZE = 1048576; // 1024 * 1024;

    // H5 config
    private static final int H5_STATIC_MEM_SIZE = STATIC_MEM_SIZE;
    private static final int H5_STATIC_DISK_SIZE = STATIC_DISK_SIZE;
    private static final long H5_STORAGE_SIZE = 5<<20; //5MB
    private static final long H5_STORAGE_EXPIRE_TIME = 7889400; // 3Month

    public static final ConfigCache DEFAULT_CONFIG;

    static {
        DEFAULT_CONFIG = new ConfigCache(
                3600,
                1800,
                7200,
                1,
                false,
                new LinkedList<String>(),
                new LinkedList<String>(),
                new LinkedList<String>(),
                new LinkedList<String>(),
                false,
                HTML_MEM_SIZE,
                STATIC_MEM_SIZE,
                HTML_DISK_SIZE,
                STATIC_DISK_SIZE,
                PHOTO_MEM_SIZE,
                PHOTO_DISK_SIZE,
                JUMP_MEM_SIZE,
                JUMP_DISK_SIZE,
                86400,
                -1,
                false,
                "",
                H5_STATIC_DISK_SIZE,
                H5_STATIC_MEM_SIZE,
                7200,
                new LinkedList<String>(),
                H5_STORAGE_SIZE,
                H5_STORAGE_EXPIRE_TIME,
                new LinkedList<String>()
                );
    }

    public final int expire;
    public final int htmlExpire;
    public final int staticExpire;
    public final int maxDownloadConnection;
    @Deprecated
    public final boolean preloadCover;
    public final List<String> preloadDomains;
    public final List<String> preloadStatic;
    public final List<String> cacheDomainsStatic;
    public final List<String> cacheDomainsPhoto;
    public final boolean downloadOverWifiOnly;

    //************ CACHE SIZE ***************//
    public final int htmlMemSize;
    public final int staticMemSize;
    public final int htmlDiskSize;
    public final int staticDiskSize;
    public final int photoMemSize;
    public final int photoDiskSize;

    // H5 config
    public final int h5StaticMemSize;
    public final int h5StaticDiskSize;
    public final int h5StaticExpire;
    public final List<String> h5CacheDomainsStatic;


    //jump preload config
    public final int jumpMemSize;
    public final int jumpDiskSize;
    public final int jumpExpire;

    public final long h5StorageSize;
    public final long h5StorageExpireTime;
    public final List<String> ignoreParamsCacheDomains;

    //version
    public int version;

    //debug
    public final boolean debug;

    //script url to inject to page loaded by zbrowser
    public final String zbrowserScriptUrl;

    private ConfigCache(
            int expire,
            int htmlExpire,
            int staticExpire,
            int maxDownloadConnection,
            boolean preloadCover,
            List<String> preloadDomains,
            List<String> preloadStatic,
            List<String> cacheDomainsStatic,
            List<String> cacheDomainsPhoto,
            boolean downloadWifiOnly,
            int htmlMemSize,
            int staticMemSize,
            int htmlDiskSize,
            int staticDiskSize,
            int photoMemSize,
            int photoDiskSize,
            int jumpMemSize,
            int jumpDiskSize,
            int jumpExpire,
            int version,
            boolean debug,
            String zbrowserScriptUrl,

            int h5StaticDiskSize,
            int h5StaticMemSize,
            int h5StaticExpire,
            List<String> h5CacheDomainsStatic,
            long h5StorageSize,
            long h5StorageExpireTime,
            List<String> ignoreParamsCacheDomains) {
        this.expire = expire;
        this.htmlExpire = htmlExpire;
        this.staticExpire = staticExpire;
        this.maxDownloadConnection = maxDownloadConnection;
        this.preloadCover = preloadCover;
        this.preloadDomains = preloadDomains;
        this.preloadStatic = preloadStatic;
        this.cacheDomainsStatic = cacheDomainsStatic;
        this.cacheDomainsPhoto = cacheDomainsPhoto;
        this.downloadOverWifiOnly = downloadWifiOnly;
        this.htmlMemSize = htmlMemSize;
        this.staticMemSize = staticMemSize;
        this.htmlDiskSize = htmlDiskSize;
        this.staticDiskSize = staticDiskSize;
        this.photoMemSize = photoMemSize;
        this.photoDiskSize = photoDiskSize;

        this.jumpMemSize = jumpMemSize;
        this.jumpDiskSize = jumpDiskSize;
        this.jumpExpire = jumpExpire;

        // H5 config
        this.h5StaticDiskSize = h5StaticDiskSize;
        this.h5StaticMemSize  = h5StaticMemSize;
        this.h5StaticExpire   = h5StaticExpire;
        this.h5CacheDomainsStatic = h5CacheDomainsStatic;

        this.h5StorageSize = h5StorageSize;
        this.h5StorageExpireTime = h5StorageExpireTime;
        this.ignoreParamsCacheDomains = ignoreParamsCacheDomains;

        this.version = version < 0 ? 0 : version;
        this.debug = debug;

        this.zbrowserScriptUrl = zbrowserScriptUrl;
    }

    public static ConfigCache getConfigFromJson(String jsonString) {
        if (jsonString == null) {
            return DEFAULT_CONFIG;
        }
        try {
            JSONObject jsonConfig = new JSONObject(jsonString);
            return getConfigFromJson(jsonConfig);
        } catch (JSONException ignored) {
        }
        return DEFAULT_CONFIG;
    }

    public static ConfigCache getConfigFromJson(@NonNull JSONObject jsonConfig) {
        try {

            int expire = getJsonInt(jsonConfig, "expire_time", DEFAULT_CONFIG.expire);
            int expireHtml = getJsonInt(jsonConfig, "html_expire_time", DEFAULT_CONFIG.htmlExpire);
            int expireStc = getJsonInt(jsonConfig, "static_expire_time", DEFAULT_CONFIG.staticExpire);
            int maxConnection = getJsonInt(jsonConfig, "max_download_connection", DEFAULT_CONFIG.maxDownloadConnection);
            boolean preloadCover = jsonConfig.optBoolean("preload_cover_image");
            List<String> preloadDomains = getJsonStringArray(jsonConfig, "preload_domains", DEFAULT_CONFIG.preloadDomains);
            List<String> preloadFiles = getJsonStringArray(jsonConfig, "preload_files", DEFAULT_CONFIG.preloadDomains);
            List<String> cacheDomainsStatic = getJsonStringArray(jsonConfig, "cache_domains", DEFAULT_CONFIG.cacheDomainsStatic);
            List<String> cacheDomainsPhoto = getJsonStringArray(jsonConfig, "cache_domains_photo", DEFAULT_CONFIG.cacheDomainsPhoto);
            boolean downloadWifiOnly = !("all".equalsIgnoreCase(jsonConfig.optString("preload_network")));

            int htmlMemSize = jsonConfig.optInt("html_mem_size", HTML_MEM_SIZE);
            int htmlDiskSize = jsonConfig.optInt("html_disk_size", HTML_DISK_SIZE);
            int staticMemSize = jsonConfig.optInt("static_mem_size", STATIC_MEM_SIZE);
            int staticDiskSize = jsonConfig.optInt("static_disk_size", STATIC_DISK_SIZE);
            int photoMemSize = jsonConfig.optInt("photo_mem_size", PHOTO_MEM_SIZE);
            int photoDiskSize = jsonConfig.optInt("photo_disk_size", PHOTO_DISK_SIZE);

            // H5 config
            int h5StaticMemSize = jsonConfig.optInt("h5_static_mem_size", H5_STATIC_MEM_SIZE);
            int h5StaticDiskSize = jsonConfig.optInt("h5_static_disk_size", H5_STATIC_DISK_SIZE);
            int h5ExpireStc = jsonConfig.optInt("h5_static_expire_time", DEFAULT_CONFIG.staticExpire);
            List<String> h5CacheDomains = getJsonStringArray(jsonConfig,"h5_cache_domains",  DEFAULT_CONFIG.h5CacheDomainsStatic);

            long h5StorageSize = jsonConfig.optInt("h5_storage_size", H5_STATIC_MEM_SIZE);
            long h5StorageExpireTime = jsonConfig.optInt("h5_storage_expire_time", H5_STATIC_DISK_SIZE);
            List<String> ignoreParamsCacheDomains = getJsonStringArray(jsonConfig,"ignore_params_cache_domains",  DEFAULT_CONFIG.h5CacheDomainsStatic);

            int jumpMemSize = jsonConfig.optInt("jump_mem_size", JUMP_MEM_SIZE);
            int jumpDiskSize = jsonConfig.optInt("jump_disk_size", JUMP_DISK_SIZE);
            int jumpExpire = jsonConfig.optInt("jump_expire", DEFAULT_CONFIG.jumpExpire);

            int version = jsonConfig.optInt("version", 0);
            boolean debug = jsonConfig.optBoolean("debug", false);

            String zbrowserScriptUrl = jsonConfig.optString("script_url", "");

            return new ConfigCache(
                    expire,
                    expireHtml,
                    expireStc,
                    maxConnection,
                    preloadCover,
                    preloadDomains,
                    preloadFiles,
                    cacheDomainsStatic,
                    cacheDomainsPhoto,
                    downloadWifiOnly,
                    htmlMemSize,
                    staticMemSize,
                    htmlDiskSize,
                    staticDiskSize,
                    photoMemSize,
                    photoDiskSize,
                    jumpMemSize,
                    jumpDiskSize,
                    jumpExpire,
                    version,
                    debug,
                    zbrowserScriptUrl,

                    h5StaticDiskSize,
                    h5StaticMemSize,
                    h5ExpireStc,
                    h5CacheDomains,
                    h5StorageSize,
                    h5StorageExpireTime,
                    ignoreParamsCacheDomains
            );
        } catch (Exception e) {
            return DEFAULT_CONFIG;
        }
    }

    private static int getJsonInt(JSONObject obj, String key, int defaultVal) {
        try {
            return Integer.parseInt(obj.get(key).toString());
        } catch (Exception e) {
            return defaultVal;
        }
    }

    private static List<String> getJsonStringArray(JSONObject obj, String key, List<String> defaultVal) {
        try {
            JSONArray arr = (JSONArray) obj.get(key);
            List<String> ret = new LinkedList<>();
            int size = arr.length();
            int i;
            for (i = 0; i < size; i++) {
                Object o = arr.get(i);
                ret.add(o.toString());
            }
            return ret;
        } catch (Exception e) {
            return defaultVal;
        }
    }

    public boolean isNeedToPreload(String url) {
        try {
            if (preloadDomains != null && preloadDomains.size() > 0) {
                URI uri = new URI(url);
                String host = uri.getHost();
                if (TextUtils.isEmpty(host)) return false;
                for (String str : preloadDomains) {
                    int length = str.length();
                    if (host.length() >= length) {
                        String endStr = host.substring(host.length() - length);
                        if (endStr.equalsIgnoreCase(str)) {
                            return true;
                        }
                    }
                }

                return false;
            }
        } catch (Exception ignored) {
        }
        return false;
    }

    public boolean checkWhiteList(String url) {
        try {
            if (ignoreParamsCacheDomains != null && ignoreParamsCacheDomains.size() > 0) {
                URI uri = new URI(url);
                String host = uri.getHost();
                if (TextUtils.isEmpty(host)) return false;
                for (String str : ignoreParamsCacheDomains) {
                    int length = str.length();
                    if (host.length() >= length) {
                        String endStr = host.substring(host.length() - length);
                        if (endStr.equalsIgnoreCase(str)) {
                            return true;
                        }
                    }
                }

                return false;
            }
        } catch (Exception ignored) {
        }
        return false;
    }

    public boolean isNeedToCache(String url) {
        try {
            URI uri = new URI(url);
            String host = uri.getHost();
            if (TextUtils.isEmpty(host)) return false;
            return ((preloadDomains != null && preloadDomains.contains(host))
                    || (cacheDomainsStatic != null && cacheDomainsStatic.contains(host))
                    || cacheDomainsPhoto != null && cacheDomainsPhoto.contains(host));

        } catch (Exception ignored) {
        }
        return false;
    }

    @Override
    public int hashCode() {
        int result = 1;
        result = (result << 5) - result + expire;
        result = (result << 5) - result + htmlExpire;
        result = (result << 5) - result + staticExpire;
        result = (result << 5) - result + maxDownloadConnection;
        result = (result << 5) - result + (preloadCover ? 1 : 0);
        result = (result << 5) - result + (downloadOverWifiOnly ? 1 : 0);
        result = (result << 5) - result + htmlMemSize;
        result = (result << 5) - result + staticMemSize;
        result = (result << 5) - result + htmlDiskSize;
        result = (result << 5) - result + staticDiskSize;
        result = (result << 5) - result + photoMemSize;
        result = (result << 5) - result + photoDiskSize;
        result = (result << 5) - result + jumpMemSize;
        result = (result << 5) - result + jumpDiskSize;
        result = (result << 5) - result + jumpExpire;

        result = (result << 5) - result + h5StaticDiskSize;
        result = (result << 5) - result + h5StaticMemSize;
        result = (result << 5) - result + h5StaticExpire;
        result = (int) ((result << 5) - result + h5StorageSize);
        result = (int) ((result << 5) - result + h5StorageExpireTime);
        for (String element : h5CacheDomainsStatic) {
            result = (result << 5) - result + (element == null ? 0 : element.hashCode());
        }


        for (String element : preloadDomains) {
            result = (result << 5) - result + (element == null ? 0 : element.hashCode());
        }
        for (String element : preloadStatic) {
            result = (result << 5) - result + (element == null ? 0 : element.hashCode());
        }
        for (String element : cacheDomainsStatic) {
            result = (result << 5) - result + (element == null ? 0 : element.hashCode());
        }
        for (String element : cacheDomainsPhoto) {
            result = (result << 5) - result + (element == null ? 0 : element.hashCode());
        }
        for (String element : ignoreParamsCacheDomains) {
            result = (result << 5) - result + (element == null ? 0 : element.hashCode());
        }

        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof ConfigCache)) return false;
        ConfigCache that = (ConfigCache) obj;

        if (
                this.expire != that.expire
                        || this.htmlExpire != that.htmlExpire
                        || this.staticExpire != that.staticExpire
                        || this.maxDownloadConnection != that.maxDownloadConnection
                        || this.preloadCover != that.preloadCover
                        || this.downloadOverWifiOnly != that.downloadOverWifiOnly
                        || this.htmlMemSize != that.htmlMemSize
                        || this.staticMemSize != that.staticMemSize
                        || this.htmlDiskSize != that.htmlDiskSize
                        || this.staticDiskSize != that.staticDiskSize
                        || this.photoMemSize != that.photoMemSize
                        || this.photoDiskSize != that.photoDiskSize
                        || this.jumpMemSize != that.jumpMemSize
                        || this.jumpDiskSize != that.jumpDiskSize
                        || this.jumpExpire != that.jumpExpire

                        || this.h5StaticDiskSize != that.h5StaticDiskSize
                        || this.h5StaticMemSize != that.h5StaticMemSize
                        || this.h5StaticExpire != that.h5StaticExpire
                        || this.h5StorageExpireTime != that.h5StorageExpireTime
                        || this.h5StorageSize != that.h5StorageSize

        ) {
            return false;
        }

        if (!equalLists(that.h5CacheDomainsStatic, this.h5CacheDomainsStatic)) {
            return false;
        }

        if (!equalLists(that.preloadDomains, this.preloadDomains)) {
            return false;
        }

        if (!equalLists(that.preloadStatic, this.preloadStatic)) {
            return false;
        }

        if (!equalLists(that.cacheDomainsStatic, this.cacheDomainsStatic)) {
            return false;
        }

        if (!equalLists(that.cacheDomainsPhoto, this.cacheDomainsPhoto)) {
            return false;
        }

        if (!equalLists(that.ignoreParamsCacheDomains, this.ignoreParamsCacheDomains)) {
            return false;
        }

        return true;
    }

    public static boolean equalLists(List<String> a, List<String> b) {
        // Check for sizes and nulls

        if (a == null && b == null) return true;


        if ((a == null && b != null) || (a != null && b == null) || (a.size() != b.size())) {
            return false;
        }

        // Sort and compare the two lists
        Collections.sort(a);
        Collections.sort(b);
        return a.equals(b);
    }
}
