package com.zing.zalo.zbrowser.cache;

import android.util.Log;

import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import com.google.gson.reflect.TypeToken;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;

/*
    Created by hoangdv4
    Document: https://docs.zalo.services/display/ZAP/Mini+Program+Data+Storage
 */
public class MiniProgramDataStorageManager {

    private static MiniProgramDataStorageManager miniProgramDataStorageManager;
    private static Map<String, MiniProgramDataStorage> storageMap;
    private String diskStorageDirectory;
    private int STORAGE_SIZE = 5<<20; // 5MB.
    private int EXPIRE_TIME = 7889400; // 3 Months.
    private Map<String, InfoItem> infoItems;

    private MiniProgramDataStorageManager(String diskStorageDirectory){
        if(diskStorageDirectory != null){
            this.diskStorageDirectory = diskStorageDirectory;
        }
        checkExp();
        storageMap = new HashMap<>();
        loadLibrary();
    }

    public static MiniProgramDataStorageManager getInstance(String diskStorageDirectory){
        if(miniProgramDataStorageManager == null){
            miniProgramDataStorageManager = new MiniProgramDataStorageManager(diskStorageDirectory);
        }
        return miniProgramDataStorageManager;
    }

    public synchronized List<String> put (String appId, Map<String, String> data){
        MiniProgramDataStorage storage = getStorage(appId);
        ArrayList<String> errorlist = new ArrayList<>();
        for(Map.Entry<String, String> entry : data.entrySet()){
            boolean isSuccess = storage.put(entry.getKey(), entry.getValue());
            if(!isSuccess){
                errorlist.add(entry.getKey());
            }
        }
        return errorlist;
    }

    public synchronized boolean put (String appId, String key, String value){
        return getStorage(appId).put(key, value);
    }

    public synchronized Map<String, String> get(String appId, List<String> listKey){
        Map<String, String> respond = new HashMap<>();
        MiniProgramDataStorage storage = getStorage(appId);
        for(String key : listKey){
            String value = storage.get(key);
            respond.put(key, value);
        }
        return respond;
    }

    public synchronized List<String> remove(String appId, List<String> listKey){
        MiniProgramDataStorage storage = getStorage(appId);
        ArrayList<String> errorlist = new ArrayList<>();
        for(String key : listKey){
            boolean isSuccess = storage.delete(key);
            if(!isSuccess){
                errorlist.add(key);
            }
        }
        return errorlist;
    }

    public synchronized boolean remove(String appId, String key){
        if(appId != null && key != null) {
            return getStorage(appId).delete(key);
        }
        return false;
    }

    public synchronized void deleteStorage(String appId){
        if(appId != null) {
            getStorage(appId).close();
            deleteRecursive(new File(diskStorageDirectory, appId));
        }
    }

    public synchronized String get(String appId, String key){
        MiniProgramDataStorage storage = getStorage(appId);
        if(storage != null){
            return getStorage(appId).get(key);
        }
        return null;
    }

    public synchronized long getSize(String appId){
        if(appId != null) {
            return getStorage(appId).getSize();
        }
        return -1;
    }

    public synchronized void clearStorage(String appId){
        if(appId != null) {
            deleteRecursive(new File(diskStorageDirectory, appId));
            getStorage(appId).clear();
        }
    }

    public synchronized void closeAllStorage(){
        for(Map.Entry<String, MiniProgramDataStorage> entry : storageMap.entrySet()) {
            entry.getValue().close();
        }
        setPropertiesFile();
    }

    public synchronized void closeStorage(String appId){
        if(appId != null) {
            getStorage(appId).close();
            storageMap.remove(appId);
        }
        setPropertiesFile();
    }

//    public Map<String, String> getAll(String appId){
//        if(appId != null) {
//            return getStorage(appId).getAll();
//        }
//        return new HashMap<>();
//    }


    private MiniProgramDataStorage getStorage(String appId){
        if(storageMap == null){
            storageMap = new HashMap<>();
        }
        updateAccessttime(appId);
        if( storageMap.containsKey(appId)){
            return storageMap.get(appId);
        }else{
            long starttime = System.currentTimeMillis();
            MiniProgramDataStorage miniProgramDataStorage = new MiniProgramDataStorage(
                    diskStorageDirectory,
                    appId,
                    STORAGE_SIZE);
            storageMap.put(appId, miniProgramDataStorage);
            Log.d("hoangdv4", "init: " + (System.currentTimeMillis() - starttime));
            return miniProgramDataStorage;
        }
    }

    private void updateAccessttime(String appId){
        if(infoItems == null){
            infoItems = new HashMap<>();
        }
        if(infoItems.containsKey(appId) && infoItems.get(appId) != null){
            infoItems.get(appId).lastAccess = System.currentTimeMillis();
        } else{
            infoItems.put(appId, new InfoItem(System.currentTimeMillis(), EXPIRE_TIME));
        }
        setPropertiesFile();
    }

    private void setPropertiesFile() {
        try {
            Log.d("hoangdv4", "setPropertiesFile: innnnnnnn");
            File file = checkAndCreatePropertiesFile();
            if(file != null) {
                String data = getJsonFromList(infoItems);
                FileOutputStream stream = new FileOutputStream(file);
                stream.write(data.getBytes());
                stream.close();
            }
        }
        catch (IOException e) {
            Log.e("Exception", "File write failed: " + e.toString());
        }
    }

    private String getPropertiesFile(){
        try {
            File myObj = checkAndCreatePropertiesFile();
            if(myObj != null) {
                Scanner myReader = new Scanner(myObj);
                StringBuilder data = new StringBuilder();
                while (myReader.hasNextLine()) {
                    data.append(myReader.nextLine());
                }
                myReader.close();
                return data.toString();
            }
        } catch (Exception e) {
            System.out.println("An error occurred.");
            e.printStackTrace();
            return null;
        }
        return null;
    }

    private File checkAndCreatePropertiesFile(){
        try {
            File dir = new File(diskStorageDirectory);
            if (!dir.exists()) dir.mkdirs();
            File file = new File(diskStorageDirectory, "properties.o");
            file.createNewFile();
            return file;
        }catch(IOException e){
            e.printStackTrace();
            return null;
        }
    }

    public static void loadLibrary(){
        System.loadLibrary("leveldb-jni");
    }

    private void deleteRecursive(File fileOrDirectory) {
        if (fileOrDirectory.isDirectory())
            for (File child : fileOrDirectory.listFiles())
                deleteRecursive(child);
        fileOrDirectory.delete();
    }

    private void checkExp(){
        if(infoItems == null) {
            String rawdata = getPropertiesFile();
            if (rawdata != null) {
                infoItems = getListFromJson(rawdata);
                if(infoItems == null){
                    infoItems = new HashMap<>();
                    setPropertiesFile();
                }
                boolean haveUpdate = false;
                for(Map.Entry<String, InfoItem> entry : infoItems.entrySet()){
                    InfoItem infoItem = entry.getValue();
                    if (infoItem != null && ((infoItem.lastAccess / 1000L) + infoItem.expTime) < (System.currentTimeMillis()/1000L)) {
                        deleteRecursive(new File(diskStorageDirectory, entry.getKey()));
                        infoItems.remove(entry.getKey());
                        haveUpdate = true;
                    }
                }
                if(haveUpdate){
                    setPropertiesFile();
                }
            }
        }
    }

    private String getJsonFromList(Map<String, InfoItem> infoItemMap){
        Type mapType = new TypeToken<Map<String, InfoItem>>(){}.getType();
        return new Gson().toJson(infoItemMap, mapType);
    }

    private Map<String, InfoItem> getListFromJson(String JsonData){
        try {
            Type mapType = new TypeToken<Map<String, InfoItem>>(){}.getType();
            return new Gson().fromJson(JsonData, mapType);
        }catch (Exception ex){
            ex.printStackTrace();
            return null;
        }

    }

    private class InfoItem{
        public long lastAccess;
        public long expTime;

        public InfoItem(long lastAccess, long expTime) {
            this.lastAccess =  lastAccess;
            this.expTime = expTime;
        }
    }
}
