package com.zing.zalo.zbrowser;

import android.content.Context;
import android.graphics.Bitmap;
import android.util.Log;
import android.webkit.ValueCallback;
import android.webkit.WebChromeClient;
import android.webkit.WebResourceRequest;
import android.webkit.WebResourceResponse;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.collection.LruCache;

import com.zing.zalo.zbrowser.cache.CacheItem;
import com.zing.zalo.zbrowser.cache.ConfigCache;
import com.zing.zalo.zbrowser.cache.DiskLruCache;
import com.zing.zalo.zbrowser.cache.MemoryLruCache;
import com.zing.zalo.zbrowser.downloader.HttpHeader;
import com.zing.zalo.zbrowser.downloader.HttpResponse;
import com.zing.zalo.zbrowser.downloader.HttpUrl;
import com.zing.zalo.zbrowser.downloader.ZNIOHttpClient;
import com.zing.zalo.zbrowser.util.ZBrowserUtil;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

public class ZBrowserCore {

    public static final String VERSION = "19.09.09";

    protected static final String LOG_TAG = "ZBrowser";

    private final ZNIOHttpClient nioHttpClient = new ZNIOHttpClient();

    private MemoryLruCache memoryHtmlCache = null;

    private MemoryLruCache memoryStaticCache = null;

    private MemoryLruCache memoryPhotoCache = null;

    private MemoryLruCache h5MemoryStaticCache = null;

    private DiskLruCache h5DiskStaticCache = null;

    private DiskLruCache diskHtmlCache = null;

    private DiskLruCache diskStaticCache = null;

    private DiskLruCache diskPhotoCache = null;

    private MemoryLruCache memoryHtmlRedirectCache = null;

    private DiskLruCache diskHtmlRedirectCache = null;

    private final Map<Integer, MemoryLruCache> mapUrlCacheMemory = new ConcurrentHashMap<>();

    private final Map<Integer, DiskLruCache> mapUrlCacheDisk = new ConcurrentHashMap<>();

    private final Map<String, Boolean> mapDownloading = new ConcurrentHashMap<>();

    private final Map<String, Map<String, String>> cookiesManager = new ConcurrentHashMap<>();

    private CacheCallback cacheCallback;

    protected final Map<String, String> requestHeaders = new HashMap<>();

    protected ConfigCache cacheConfig;

    protected Stats statistic = Stats.createEmptyStats();

    private Stats tempStat = Stats.createEmptyStats();

    protected InvestigateListener investigateListener;

    protected ZBrowserPreload.ZBrowserPreloadCallback zbrowserCallback;

    private Toast debugToast = null;

    private static final boolean DEBUG_TOAST = false;

    /*count number of urls have been preloaded*/
    private final AtomicInteger preloadedUrlCount = new AtomicInteger();

    public final WebViewClient zbrowserWebviewListener = new WebViewClient() {

        @Override
        public boolean shouldOverrideUrlLoading(WebView view, String url) {
            //user open a link from web page
            //consider this action is call webview load url
            onCallWebviewLoadUrl(url);
            if (cacheConfig.debug) {
                cancelDebugToast();
            }

            return super.shouldOverrideUrlLoading(view, url);
        }

        @Override
        public void onPageStarted(WebView view, String url, Bitmap favicon) {
            statistic.pageStart = System.currentTimeMillis();
            view.setTag(/* already show toast */Boolean.FALSE);
        }

        @Override
        public void onPageCommitVisible(WebView view, String url) {
            statistic.pageVisible = System.currentTimeMillis();
        }


        //App chủ động gọionCallWebviewLoadUrl(String url) để set statistic.currentBaseUrl nếu cần.
        @Nullable
        @Override
        public WebResourceResponse shouldInterceptRequest(WebView view, WebResourceRequest request) {
            if (!cacheConfig.isNeedToPreload(statistic.currentBaseUrl)) return null;
            return getWebResourceResponse(request, view.getContext());
        }

        @Nullable
        @Override
        public WebResourceResponse shouldInterceptRequest(WebView view, String url) {
            if (!cacheConfig.isNeedToPreload(statistic.currentBaseUrl)) return null;
            return getWebResourceResponse(url, view.getContext());
        }
    };

    public final WebChromeClient zbrowserWebChromeClientListener = new WebChromeClient() {
        @Override
        public void onProgressChanged(WebView view, int newProgress) {
            if (cacheConfig.isNeedToPreload(view.getUrl())) {
                if (newProgress >= 100) {
                    statistic.pageFinish = System.currentTimeMillis();

                    //inject zbrowser script to page for tracking...
                    if (cacheConfig.zbrowserScriptUrl != null && cacheConfig.zbrowserScriptUrl.startsWith("http")) {
                        String jsSrc = cacheConfig.zbrowserScriptUrl;
                        String script = String.format("(function(){var o=document.createElement('script');o.defer='true';o.src='%s';document.body.appendChild(o);})();", jsSrc);

                        view.evaluateJavascript(script, new ValueCallback<String>() {
                            @Override
                            public void onReceiveValue(String s) {

                            }
                        });
                    }

                    //show debug info
                    if (DEBUG_TOAST && cacheConfig.debug) {
                        Object alreadyShowToast = view.getTag();
                        if (alreadyShowToast instanceof Boolean && !(Boolean) alreadyShowToast) {
                            view.setTag(Boolean.TRUE);
                            String debugText = statistic.toDebugString();
//                            debugToast = Toast.makeText(CoreUtility.getAppContext(), debugText, Toast.LENGTH_SHORT);
//                            debugToast.show();
                        }
                    }
                }
            }
        }

        @Override
        public void onCloseWindow(WebView window) {
            if (cacheConfig.debug) {
                cancelDebugToast();
            }
        }
    };

    // H5 config Không được apply ở Constructor này
    @Deprecated
    public ZBrowserCore(int htmlMemSize,
                        int staticMemSize,
                        int photoMemSize,
                        String diskCacheDirectory,
                        int htmlDiskSize,
                        int staticDiskSize,
                        int photoDiskSize,
                        ConfigCache cacheConfig,
                        Context context
    ) {
        setCacheConfig(cacheConfig, context);
        if (htmlMemSize > 0) {
            memoryHtmlCache = new MemoryLruCache(htmlMemSize, mapUrlCacheMemory);
        }
        if (staticMemSize > 0) {
            memoryStaticCache = new MemoryLruCache(staticMemSize, mapUrlCacheMemory);
        }
        if (photoMemSize > 0) {
            memoryPhotoCache = new MemoryLruCache(photoMemSize, mapUrlCacheMemory);
        }
        if (htmlDiskSize > 0) {
            try {
                diskHtmlCache = DiskLruCache.open(new File(diskCacheDirectory, "html"), 1, 1, htmlDiskSize, mapUrlCacheDisk, true);
            } catch (Exception e) {
                Log.w(LOG_TAG, "Cannot init html disk cache", e);
            }
        }
        if (staticDiskSize > 0) {
            try {
                diskStaticCache = DiskLruCache.open(new File(diskCacheDirectory, "static"), 1, 1, staticDiskSize, mapUrlCacheDisk, true);
            } catch (Exception e) {
                Log.w(LOG_TAG, "Cannot static html disk cache", e);
            }
        }
        if (photoDiskSize > 0) {
            try {
                diskPhotoCache = DiskLruCache.open(new File(diskCacheDirectory, "photo"), 1, 1, photoDiskSize, mapUrlCacheDisk, true);
            } catch (Exception e) {
                Log.w(LOG_TAG, "Cannot init photo disk cache", e);
            }
        }
        memoryHtmlRedirectCache = new MemoryLruCache(htmlMemSize > 0 ? htmlMemSize : ConfigCache.DEFAULT_CONFIG.htmlMemSize, null);
        try {
            diskHtmlRedirectCache = DiskLruCache.open(new File(diskCacheDirectory, "redirect"), 1, 1, htmlDiskSize > 0 ? htmlDiskSize : ConfigCache.DEFAULT_CONFIG.htmlDiskSize, null, false);
        } catch (Exception e) {
            Log.w(LOG_TAG, "Cannot init html redirect disk cache", e);
        }
    }

    public ZBrowserCore(@NonNull ConfigCache configCache, @NonNull String diskCacheDirectory, Context context) {
        setCacheConfig(configCache, context);
        if (configCache.htmlMemSize > 0) {
            memoryHtmlCache = new MemoryLruCache(configCache.htmlMemSize, mapUrlCacheMemory);
        }
        if (configCache.staticMemSize > 0) {
            memoryStaticCache = new MemoryLruCache(configCache.staticMemSize, mapUrlCacheMemory);
        }
        if (configCache.h5StaticMemSize > 0) {
            h5MemoryStaticCache = new MemoryLruCache(configCache.h5StaticMemSize, mapUrlCacheMemory);
        }
        if (configCache.photoMemSize > 0) {
            memoryPhotoCache = new MemoryLruCache(configCache.photoMemSize, mapUrlCacheMemory);
        }
        if (configCache.htmlDiskSize > 0) {
            try {
                diskHtmlCache = DiskLruCache.open(new File(diskCacheDirectory, "html"), 1, 1, configCache.htmlDiskSize, mapUrlCacheDisk, true);
            } catch (Exception e) {
                Log.w(LOG_TAG, "Cannot init html disk cache", e);
            }
        }
        if (configCache.h5StaticDiskSize > 0) {
            try {
                h5DiskStaticCache = DiskLruCache.open(new File(diskCacheDirectory, "h5static"), 1, 1, configCache.h5StaticDiskSize, mapUrlCacheDisk, true);
            } catch (Exception e) {
                Log.w(LOG_TAG, "Cannot static h5 static disk cache", e);
            }
        }
        if (configCache.staticDiskSize > 0) {
            try {
                diskStaticCache = DiskLruCache.open(new File(diskCacheDirectory, "static"), 1, 1, configCache.staticDiskSize, mapUrlCacheDisk, true);
            } catch (Exception e) {
                Log.w(LOG_TAG, "Cannot static static disk cache", e);
            }
        }
        if (configCache.photoDiskSize > 0) {
            try {
                diskPhotoCache = DiskLruCache.open(new File(diskCacheDirectory, "photo"), 1, 1, configCache.photoDiskSize, mapUrlCacheDisk, false);
            } catch (Exception e) {
                Log.w(LOG_TAG, "Cannot init photo disk cache", e);
            }
        }

        memoryHtmlRedirectCache = new MemoryLruCache(configCache.htmlMemSize > 0 ? configCache.htmlMemSize : ConfigCache.DEFAULT_CONFIG.htmlMemSize, null);
        try {
            diskHtmlRedirectCache = DiskLruCache.open(new File(diskCacheDirectory, "redirect"), 1, 1, configCache.htmlDiskSize > 0 ? configCache.htmlDiskSize : ConfigCache.DEFAULT_CONFIG.htmlDiskSize, null, false);
        } catch (Exception e) {
            Log.w(LOG_TAG, "Cannot init html redirect disk cache", e);
        }
    }

    public void cancelDebugToast() {
        if (debugToast != null) {
            debugToast.cancel();
            debugToast = null;
        }
    }

    public void addCookie(String domain, Map<String, String> cookies) {
        if (domain == null || domain.isEmpty() || cookies == null || cookies.isEmpty()) {
            return;
        }
        Map<String, String> currentCookies = cookiesManager.get(domain);
        if (currentCookies == null) {
            currentCookies = cookies;
        } else {
            for (Map.Entry<String, String> entry : cookies.entrySet()) {
                String cookieName = entry.getKey();
                String cookieValue = entry.getValue();
                if (cookieName == null || cookieName.isEmpty() || cookieValue == null || cookieValue.isEmpty()) {
                    continue;
                }
                currentCookies.put(cookieName, cookieValue);
            }
        }
        cookiesManager.put(domain, currentCookies);
    }

    private String buildCookieForDomain(String domain) {
        if (domain == null || domain.isEmpty()) {
            return null;
        }
        String baseDomain = String.format(".%s", domain);
        StringBuilder cookieBuilder = new StringBuilder();
        for (Map.Entry<String, Map<String, String>> entry : cookiesManager.entrySet()) {
            String currentDomain = entry.getKey();
            Map<String, String> currentCookies = entry.getValue();
            if (baseDomain.endsWith(currentDomain)) {
                for (Map.Entry<String, String> entryCookie : currentCookies.entrySet()) {
                    String cookieName = entryCookie.getKey();
                    String cookieValue = entryCookie.getValue();
                    cookieBuilder.append(cookieName).append("=").append(cookieValue).append("; ");
                }
            }
        }
        return cookieBuilder.toString();
    }

    private Map<String, String> prepareCookies(String url) {
        Map<String, String> result = cloneRequestHeader();
        try {
            result.remove("cookie");
            String domain = HttpUrl.parseUrl(url).domain;
            String cookies = buildCookieForDomain(domain);
            if (!cookies.isEmpty()) {
                result.put("cookie", cookies);
            }
        } catch (Exception ignore) {

        }
        return result;
    }

    public void setZbrowserPreloadCallback(ZBrowserPreload.ZBrowserPreloadCallback callback) {
        this.zbrowserCallback = callback;
    }

    public void setCacheCallback(CacheCallback callback) {
        this.cacheCallback = callback;
    }

    public void setInvestigateListener(InvestigateListener listener) {
        this.investigateListener = listener;
    }
//    private static String makeBase64Key(String key) {
//        return Base64.encodeToString(key.getBytes(), Base64.URL_SAFE | Base64.NO_PADDING | Base64.NO_WRAP);
//    }

    private static String makeShortKey(String key) {
        //we use md5 hash to make key shorter
        String var1 = "";
        try {
            MessageDigest var2 = MessageDigest.getInstance("MD5");
            var2.reset();
            var2.update(key.getBytes());
            byte[] var3 = var2.digest();
            String var4 = "";

            for (int var5 = 0; var5 < var3.length; ++var5) {
                var4 = Integer.toHexString(255 & var3[var5]);
                if (var4.length() == 1) {
                    var1 = var1 + "0" + var4;
                } else {
                    var1 = var1 + var4;
                }
            }
        } catch (Exception var6) {
            //ensure we still have a unique key
            var1 = String.valueOf(key.hashCode());
        }
        return var1;
    }

    public CacheItem getFromDiskCache(DiskLruCache cache, String key) {
        if (cache == null || cache.isClosed()) {
            return null;
        }
        InputStream inputStream = null;
        try {
            String keyShorter = makeShortKey(key);
            DiskLruCache.Snapshot snapshot = cache.get(keyShorter);
            if (snapshot == null) {
                return null;
            }
            ByteArrayOutputStream bufferData = new ByteArrayOutputStream((int) snapshot.getLength(0));
            inputStream = cache.isGzip() ? new GZIPInputStream(snapshot.getInputStream(0)) : snapshot.getInputStream(0);
            int byteRead;
            byte[] buffer = new byte[64 * 1024];
            while ((byteRead = inputStream.read(buffer)) != -1) {
                bufferData.write(buffer, 0, byteRead);
            }

            CacheItem result = CacheItem.deserialize(ByteBuffer.wrap(bufferData.toByteArray()));
            return result;
        } catch (Exception e) {
            Log.w(LOG_TAG, "Get cache file from disk cache error", e);
            return null;
        } finally {
            try {
                if (inputStream != null) {
                    inputStream.close();
                }
            } catch (Exception e) {
            }
        }
    }

    protected void putToDiskCache(DiskLruCache cache, String key, CacheItem value) {
        if (cache == null || cache.isClosed() || key == null || value == null) {
            return;
        }
        try {
            String keyShorter = makeShortKey(key);
            DiskLruCache.Editor edit = cache.edit(keyShorter);
            OutputStream editorOut = cache.isGzip() ? new GZIPOutputStream(edit.newOutputStream(0)) : edit.newOutputStream(0);
            editorOut.write(value.serialize().array());
            editorOut.close();
            edit.commit();
            mapUrlCacheDisk.put(keyShorter.hashCode(), cache);
        } catch (Exception e) {
            Log.w(LOG_TAG, "Put data to disk cache error", e);
        }
    }

    private MemoryLruCache mapDiskCacheToMemCache(DiskLruCache diskCache) {
        if (diskCache == diskHtmlCache) {
            return memoryHtmlCache;
        }
        if (diskCache == diskStaticCache) {
            return memoryStaticCache;
        }
        if (diskCache == h5DiskStaticCache) {
            return h5MemoryStaticCache;
        }
        if (diskCache == diskPhotoCache) {
            return memoryPhotoCache;
        }
        return null;
    }

    private int getExpireTimeOfResource(String url) {
        String domain = "";
        try {
            domain = HttpUrl.parseUrl(url).domain;
        } catch (Exception e) {
        }
        if (cacheConfig.preloadDomains.contains(domain)) {
            return cacheConfig.htmlExpire;
        }
        if (cacheConfig.cacheDomainsStatic.contains(domain) || cacheConfig.cacheDomainsPhoto.contains(domain)) {
            return cacheConfig.staticExpire;
        }
        return 0;
    }

    /**
     * Set custom http request header such as cookie...
     *
     * @param headers map header name and value
     */
    public void setRequestHeaders(Map<String, String> headers) {
        if (headers != null) {
            for (Map.Entry<String, String> entry : headers.entrySet()) {
                this.requestHeaders.put(entry.getKey(), entry.getValue());
            }
        } else {
            this.requestHeaders.clear();
        }
    }

    /**
     * Set cache config to zbrowser
     * Every time this function is called, zbrowser will preload
     * static files within {@code cacheConfig.preloadStatic}
     *
     * @param cacheConfig new cache config
     */
    public boolean setCacheConfig(ConfigCache cacheConfig, Context context) {
        boolean result = false;
        if (cacheConfig == null) {
            cacheConfig = ConfigCache.DEFAULT_CONFIG;
        }
        if (!cacheConfig.equals(this.cacheConfig)) {
            this.cacheConfig = cacheConfig;
            //preload static files
            for (String staticFilesUrl : this.cacheConfig.preloadStatic) {
                loadStaticFile(staticFilesUrl, context);
            }
            result = true;
        }

        nioHttpClient.setMaxConnection(cacheConfig.maxDownloadConnection);

        return result;
    }

    /**
     * Download an html resource from {@code url}, and store content
     * to both memory cache and disk cache.
     *
     * @param inputUrl     http url of resource
     * @param callback     download callback function raise when downloading article.
     * @param preloadParam information about {@code url} in order to decide to preload it or not
     */
    public void load(final String inputUrl, final DownloadCallback callback, PreloadParam preloadParam, final Context context) {
        final String url = ZBrowserUtil.removeTrackingParam(inputUrl, getCacheConfig());

        if (memoryHtmlCache == null && diskHtmlCache == null) {
            //no cache instance available, do nothing
            if (investigateListener != null) {
                investigateListener.submitLog(String.format("PRELOAD - IGNORE (env): %s", url));
            }
            return;
        }

        if (!getCacheConfig().isNeedToPreload(url)) {
            if (investigateListener != null) {
                investigateListener.submitLog(String.format("PRELOAD - IGNORE (config): %s", url));
            }
            return;
        }

        if (mapDownloading.containsKey(url)) {
            //already downloading this url, just return
            if (investigateListener != null) {
                investigateListener.submitLog(String.format("PRELOAD - DONE (downloading): %s", url));
            }
            return;
        }
        if (mapUrlCacheMemory.containsKey(url.hashCode()) || mapUrlCacheDisk.containsKey(makeShortKey(url).hashCode())) {
            //already cached this url, just return
            if (investigateListener != null) {
                investigateListener.submitLog(String.format("PRELOAD - DONE (cached): %s", url));
            }
            if (callback != null) {
                callback.onDataDownloading(inputUrl, 100, true);
            }
            return;
        }
        String urlRedirect = ZBrowserUtil.removeTrackingParam(checkUrlRedirect(url), getCacheConfig());
        if (!urlRedirect.equals(url)) {
            if (mapUrlCacheMemory.containsKey(urlRedirect.hashCode()) || mapUrlCacheDisk.containsKey(makeShortKey(urlRedirect).hashCode())) {
                //url is redirected and already cached, just return
                if (investigateListener != null) {
                    investigateListener.submitLog(String.format("PRELOAD - DONE (cached - redirected): %s", url));
                }
                if (callback != null) {
                    callback.onDataDownloading(inputUrl, 100, true);
                }
                return;
            }
        }

        if (investigateListener != null) {
            investigateListener.submitLog(String.format("Start download: %s", url));
        }
        if (callback != null) {
            callback.onStart(inputUrl);
        }
        final ByteArrayOutputStream downloadBuffer = new ByteArrayOutputStream();
        try {
            mapDownloading.put(url, true);
            Map<String, String> requestHeader = prepareCookies(url);
            nioHttpClient.downloadAsync(url, requestHeader, new HttpResponse.Callback() {
                public void onResponseHeader(String requestUrl, int statusCode, HttpHeader responseHeaders) {
                    if (300 <= statusCode && statusCode < 400 && responseHeaders.hasHeader("location")) {
                        //cache redirect to replace url when jump
                        String destUrl = responseHeaders.getHeader("location");
                        CacheItem cacheItem = new CacheItem((int) (System.currentTimeMillis() / 1000 + cacheConfig.htmlExpire), "", destUrl.getBytes(), "");

                        String redirectDomain = "NA";
                        String rawDomain = "";
                        try {
                            redirectDomain = HttpUrl.parseUrl(destUrl).domain;
                            rawDomain = HttpUrl.parseUrl(requestUrl).domain;
                        } catch (MalformedURLException malformedURLException) {
                            malformedURLException.printStackTrace();
                        }
                        if(redirectDomain.equals(rawDomain)){
                            String key = ZBrowserUtil.removeTrackingParam(requestUrl, getCacheConfig());
                            memoryHtmlRedirectCache._put(key, cacheItem);
                            putToDiskCache(diskHtmlRedirectCache, key, cacheItem);
                        } else if (callback != null) {
                            callback.onError(inputUrl);
                        }

                    }
                    //fail
                    //try to remove redirect cache
                    if (statusCode >= 400) {
                        try {
                            String memKey = url;
                            String diskKey = makeShortKey(memKey);

                            memoryHtmlRedirectCache.remove(memKey);
                            diskHtmlRedirectCache.remove(diskKey);
                        } catch (Exception e) {

                        }
                    }
                }

                @Override
                public void onResponseBodyData(String requestUrl, byte[] data, HttpHeader responseHeaders) {
                    try {
                        downloadBuffer.write(data);
                        if (callback != null) {
                            long contentLength = 100000;
                            if (responseHeaders.hasHeader("content-length")) {
                                String contentLengthStr = responseHeaders.getHeader("content-length");
                                try {
                                    contentLength = Long.parseLong(contentLengthStr);
                                } catch (Exception e) {

                                }
                            }
                            int currentPercent = (int) (downloadBuffer.size() * 100f / contentLength);
                            if (currentPercent > 94) {
                                currentPercent = 94;
                            }

                            callback.onDataDownloading(requestUrl, currentPercent, false);
                        }
                    } catch (Exception ignore) {

                    }
                }

                @Override
                public void onResponseDone(String requestUrl, int statusCode, HttpHeader responseHeaders) {
                    preloadedUrlCount.incrementAndGet();

                    mapDownloading.remove(url);
                    mapDownloading.remove(requestUrl);

                    try {
                        byte[] rawData = downloadSuccessHandle(requestUrl, statusCode, downloadBuffer, responseHeaders);
                        int expiredTime = (int) (System.currentTimeMillis() / 1000 + cacheConfig.htmlExpire);
                        String contentType = "text/html";
                        String etag = responseHeaders.getHeader("etag");
                        if (etag == null) {
                            etag = "";
                        }

                        //preload a list of url resource along with this html
                        String preloadUrls = responseHeaders.getHeader("preload");
                        if (preloadUrls != null) {
                            if (investigateListener != null) {
                                investigateListener.submitLog(String.format("Preload resources of html: %s", preloadUrls));
                            }
                            String[] listUrls = preloadUrls.split(";");
                            for (String url : listUrls) {
                                url = url.trim();
                                if (url.startsWith("http")) {
                                    loadStaticFile(url, context);
                                }
                            }
                        }

                        CacheItem cacheItem = new CacheItem(expiredTime, contentType, rawData, etag);
                        if (memoryHtmlCache != null) {
                            memoryHtmlCache._put(requestUrl, cacheItem);
                        }
                        if (diskHtmlCache != null) {
                            putToDiskCache(diskHtmlCache, requestUrl, cacheItem);
                        }
                        if (callback != null) {
                            callback.onDataDownloading(inputUrl, 100, true);
                        }
                        if (investigateListener != null) {
                            investigateListener.submitLog(String.format("PRELOAD - DONE (downloaded): %s", requestUrl));
                        }
                    } catch (Exception e) {
                        //error
                        this.onResponseError(requestUrl, e.getMessage());
                    }
                }

                @Override
                public void onResponseError(String requestUrl, String errorMessage) {
                    mapDownloading.remove(url);
                    mapDownloading.remove(requestUrl);
                    if (callback != null) {
                        callback.onError(requestUrl);
                    }
                    if (investigateListener != null) {
                        investigateListener.submitLog(String.format("PRELOAD - ERROR (onResponseError): %s \n Msg: %s", requestUrl, errorMessage));
                    }
                }
            });
        } catch (Exception e) {
            mapDownloading.remove(url);
            if (callback != null) {
                callback.onError(url);
            }
            if (investigateListener != null) {
                investigateListener.submitLog(String.format("PRELOAD - ERROR: %s \n Msg: %s", url, e.toString()));
            }
        }
    }

    // hoangdv4
    private boolean loadStaticFile(final String inputUrl, Context context) {
        final String url = ZBrowserUtil.removeTrackingParam(inputUrl, getCacheConfig());
        final MemoryLruCache memCache;
        final DiskLruCache diskCache;
        String urlDomain = "";
        try {
            urlDomain = HttpUrl.parseUrl(url).domain;
        } catch (Exception e) {
        }
        int expireTime = cacheConfig.staticExpire;
        if (cacheConfig.cacheDomainsStatic.contains(urlDomain)) {
            if(cacheConfig.h5CacheDomainsStatic.contains(urlDomain)){
                memCache = h5MemoryStaticCache;
                diskCache = h5DiskStaticCache;
            }else {
                memCache = memoryStaticCache;
                diskCache = diskStaticCache;
            }
        } else if (cacheConfig.cacheDomainsPhoto.contains(urlDomain)) {
            memCache = memoryPhotoCache;
            diskCache = diskPhotoCache;
        } else if (cacheConfig.preloadDomains.contains(urlDomain)) {
            memCache = memoryHtmlCache;
            diskCache = diskHtmlCache;
            expireTime = cacheConfig.htmlExpire;
        } else {
            // this url is not intended for caching, just return
            return false;
        }
        if (memCache == null && diskCache == null) {
            // no cache instance available, do nothing
            return false;
        }

        if (mapDownloading.containsKey(url)) {
            //already downloading this url, just return
            return false;
        }
        if (mapUrlCacheMemory.containsKey(url.hashCode()) || mapUrlCacheDisk.containsKey(makeShortKey(url).hashCode())) {
            //already cached this url, no need to download again
            return false;
        }
        if (cacheConfig.downloadOverWifiOnly && !ZBrowserUtil.isConnectedWifi(context)) {
            //only download over wifi
            return false;
        }

        downloadFile(url, memCache, diskCache, null, expireTime);
        return true;
    }

    private Map<String, String> cloneRequestHeader() {
        Map<String, String> result = new HashMap<>();
        if (requestHeaders != null) {
            for (Map.Entry<String, String> entry : requestHeaders.entrySet()) {
                result.put(entry.getKey(), entry.getValue());
            }
        }
        return result;
    }

    private void downloadFile(final String url, final MemoryLruCache memCache, final DiskLruCache diskCache, String etag, final int expireTime) {
        if (memCache == null && diskCache == null) {
            //there is no cache to store downloaded content, just return
            return;
        }
        try {
            //download and put to cache
            mapDownloading.put(url, true);

            Map<String, String> requestHeader = prepareCookies(url);
            if (etag != null && !etag.isEmpty()) {
                requestHeader.put("if-none-match", etag);
            }

            final ByteArrayOutputStream downloadBuffer = new ByteArrayOutputStream();
            final ArrayList<String> differenceDomain = new ArrayList<>();
            nioHttpClient.downloadAsync(url, requestHeader, new HttpResponse.Callback() {
                @Override
                public void onResponseHeader(String requestUrl, int statusCode, HttpHeader responseHeaders) {
                    //only care about request html
                    if (memCache != memoryHtmlCache && diskCache != diskHtmlCache) {
                        return;
                    }
                    if (statusCode != HttpURLConnection.HTTP_NOT_MODIFIED && 300 <= statusCode && statusCode < 400 && responseHeaders.hasHeader("location")) {
                        //cache redirect to replace url when jump
                        String destUrl = responseHeaders.getHeader("location");
                        CacheItem cacheItem = new CacheItem((int) (System.currentTimeMillis() / 1000 + cacheConfig.htmlExpire), "", destUrl.getBytes(), "");
                        String key = ZBrowserUtil.removeTrackingParam(requestUrl, getCacheConfig());

                        String redirectDomain = "NA";
                        String rawDomain = "";
                        try {
                            redirectDomain = HttpUrl.parseUrl(destUrl).domain;
                            rawDomain = HttpUrl.parseUrl(requestUrl).domain;
                        } catch (MalformedURLException malformedURLException) {
                            malformedURLException.printStackTrace();
                        }
                        if(redirectDomain.equals(rawDomain)){
                            differenceDomain.add(redirectDomain);
                            memoryHtmlRedirectCache._put(key, cacheItem);
                            putToDiskCache(diskHtmlRedirectCache, key, cacheItem);
                        } else {
                            log(String.format("Download file error: difference domain - Url: %s - Redirect Url: %s", requestUrl, destUrl));
                        }
                    }
                    //fail
                    //try to remove redirect cache
                    if (statusCode >= 400) {
                        try {
                            String memKey = url;
                            String diskKey = makeShortKey(memKey);

                            memoryHtmlRedirectCache.remove(memKey);
                            diskHtmlRedirectCache.remove(diskKey);
                        } catch (Exception e) {

                        }
                    }
                }

                @Override
                public void onResponseBodyData(String requestUrl, byte[] data, HttpHeader responseHeaders) {
                    try {
                        downloadBuffer.write(data);
                    } catch (Exception ignore) {
                        log(String.format("Download file error: downloadBuffer write data - Url: %s", requestUrl));
                    }
                }

                @Override
                public void onResponseDone(String requestUrl, int statusCode, HttpHeader responseHeaders) {
                    mapDownloading.remove(url);

                    if(differenceDomain.size() == 0) {
                        if (statusCode == HttpURLConnection.HTTP_NOT_MODIFIED) {
                            if (investigateListener != null) {
                                investigateListener.submitLog(String.format("Not modified: %s", requestUrl));
                            }
                            //not modified
                            CacheItem cacheItem = null;
                            if (memCache != null) {
                                cacheItem = memCache.get(requestUrl);
                                if (cacheItem != null) {
                                    cacheItem.expireTime = (int) (System.currentTimeMillis() / 1000 + expireTime);
                                }
                            }
                            if (diskCache != null) {
                                if (cacheItem == null) {
                                    cacheItem = getFromDiskCache(diskCache, requestUrl);
                                }
                                if (cacheItem != null) {
                                    cacheItem.expireTime = (int) (System.currentTimeMillis() / 1000 + expireTime);
                                    if (memCache != null) {
                                        memCache._put(requestUrl, cacheItem);
                                    }
                                    putToDiskCache(diskCache, requestUrl, cacheItem);
                                }
                            }
                            return;
                        }

                        try {
                            byte[] rawData = downloadSuccessHandle(url, statusCode, downloadBuffer, responseHeaders);
                            int expiredTime = (int) (System.currentTimeMillis() / 1000 + expireTime);
                            String contentType = responseHeaders.getHeader("content-type");
                            if (contentType == null || contentType.isEmpty()) {
                                contentType = "text/plain";
                            } else {
                                contentType = contentType.split(";")[0];
                            }
                            String etag = responseHeaders.getHeader("etag");
                            if (etag == null) {
                                etag = "";
                            }
                            CacheItem cacheItem = new CacheItem(expiredTime, contentType, rawData, etag);
                            if (memCache != null) {
                                memCache._put(requestUrl, cacheItem);
                            }
                            if (diskCache != null) {
                                putToDiskCache(diskCache, requestUrl, cacheItem);
                            }
                            mapDownloading.remove(requestUrl);

                            log(String.format("Download file done - Url: %s", requestUrl));
                        } catch (Exception e) {
                            e.printStackTrace();
                            this.onResponseError(requestUrl, e.getMessage());
                        }
                    } else {
                        log(String.format("Download file error: difference domain - Url: %s - Redirect Url: %s", requestUrl, differenceDomain.get(0)));
                    }
                }

                @Override
                public void onResponseError(String requestUrl, String errorMessage) {
                    mapDownloading.remove(url);
                    if (investigateListener != null) {
                        investigateListener.submitLog(String.format("Download file error:  %s - Url: %s", errorMessage, requestUrl));
                    }
                }
            });

        } catch (Exception e) {
            mapDownloading.remove(url);
            e.printStackTrace();
        }
    }

    private byte[] downloadSuccessHandle(String url, int statusCode, ByteArrayOutputStream downloadBuffer, HttpHeader responseHeaders) throws IOException {
        if (statusCode != 200) {
            throw new IOException("Status code != 200: " + statusCode + "! Url: " + url);
        }
        byte[] rawData = downloadBuffer.toByteArray();
        GZIPInputStream gzipInputStream = null;
        try {
            if ("gzip".equalsIgnoreCase(responseHeaders.getHeader("content-encoding"))) {
                ByteArrayOutputStream out = new ByteArrayOutputStream();
                gzipInputStream = new GZIPInputStream(new ByteArrayInputStream(rawData));
                byte[] buf = new byte[65536];
                int read;
                while ((read = gzipInputStream.read(buf)) != -1) {
                    out.write(buf, 0, read);
                }

                rawData = out.toByteArray();
            }

            return rawData;
        } catch (Exception e) {
            if (investigateListener != null) {
                investigateListener.submitLog(String.format("Error while handling downloaded data: %s - Exception: %s", url, e.toString()));
            }
            throw new IOException(e);
        } finally {
            if (gzipInputStream != null) {
                gzipInputStream.close();
            }
        }
    }

    private static final Map<String, String> STATIC_RESP_HEADERS_FOR_CACHE_ITEM = new HashMap<>();

    static {
        STATIC_RESP_HEADERS_FOR_CACHE_ITEM.put("Access-Control-Allow-Origin", "*");
    }

    /**
     * Get WebResourceResponse represent for content of {@code url}.
     * Override WebViewClient.shouldInterceptRequest and return this function.
     * This function is now deprecated for external call.
     * Use {@code getWebResourceResponse(WebResourceRequest)} instead
     *
     * @param inputUrl http url of resource
     * @return WebResourceResponse represent for content of resource if it is cached
     * otherwise null
     */
    public WebResourceResponse getWebResourceResponse(String inputUrl, Context context) {
        try {
//            if (investigateListener != null && cacheConfig.isNeedToPreload(inputUrl)) {
//                investigateListener.submitLog("GET - REQUEST: " + inputUrl);
//            }
            boolean isUrlNeedCache = cacheConfig.isNeedToCache(inputUrl);
            String url = ZBrowserUtil.removeTrackingParam(inputUrl, getCacheConfig());
            CacheItem cacheItem = null;
            MemoryLruCache memoryLruCache = null;
            DiskLruCache diskLruCache = null;
            if (mapUrlCacheMemory.containsKey(url.hashCode())) {
                memoryLruCache = mapUrlCacheMemory.get(url.hashCode());
                cacheItem = memoryLruCache.get(url);
            }
            if (cacheItem != null) {
                ++statistic.memCacheHit;
            } else {
                mapUrlCacheMemory.remove(url.hashCode());
                if (isUrlNeedCache) {
                    ++statistic.memCacheMiss;
                }

                String urlKey = makeShortKey(url);
                if (mapUrlCacheDisk.containsKey(urlKey.hashCode())) {
                    diskLruCache = mapUrlCacheDisk.get(urlKey.hashCode());
                    cacheItem = getFromDiskCache(diskLruCache, url);
                    if (cacheItem != null) {
                        ++statistic.diskCacheHit;
                        if (memoryLruCache == null) {
                            memoryLruCache = mapDiskCacheToMemCache(diskLruCache);
                        }
                        if (memoryLruCache != null) {
                            memoryLruCache._put(url, cacheItem);
                        }
                    } else {
                        mapUrlCacheDisk.remove(urlKey.hashCode());
                        if (isUrlNeedCache) {
                            ++statistic.diskCacheMiss;
                        }
                    }
                } else {
                    if (isUrlNeedCache) {
                        ++statistic.diskCacheMiss;
                    }
                }

            }

            if (cacheItem != null) {
                if (cacheItem.expireTime < System.currentTimeMillis() / 1000) {
                    int expireTimeOfResource = getExpireTimeOfResource(url);
                    if (expireTimeOfResource > 0) {
                        //cache out of date - revalidate
                        if (investigateListener != null) {
                            investigateListener.submitLog("Revalidate: " + url + " - " + cacheItem.expireTime);
                        }
                        downloadFile(url, memoryLruCache, diskLruCache, cacheItem.etag, expireTimeOfResource);
                    } else {
                        //resource is not for cached anymore, delete lazy
                        mapUrlCacheMemory.remove(url.hashCode());
                        mapUrlCacheDisk.remove(makeShortKey(url).hashCode());
                    }
                }

                if (cacheCallback != null) {
                    cacheCallback.onCacheHit(url);
                }

                /*if (investigateListener != null) {
                    investigateListener.submitLog("Cache hit: " + url);
                }*/

                if (!statistic.baseUrlHit && inputUrl.equals(statistic.currentBaseUrl)) {
                    statistic.baseUrlHit = true;
                }
                statistic.totalByteFromCache.addAndGet(cacheItem.data.length);

                WebResourceResponse result = new WebResourceResponse(cacheItem.mimeType, "UTF-8", new ByteArrayInputStream(cacheItem.data));
                Map<String, String> respHeaders = result.getResponseHeaders();
                if (respHeaders == null) {
                    respHeaders = new HashMap<>();
                }
                respHeaders.putAll(STATIC_RESP_HEADERS_FOR_CACHE_ITEM);
                result.setResponseHeaders(respHeaders);
                if (investigateListener != null) {
                    investigateListener.submitLog("GET - DONE: " + inputUrl);
                }
                return result;
            }
            //this url is not cached yet
            //try to download and cache for later use
            boolean isDownloading = loadStaticFile(url, context);

            if (cacheCallback != null) {
                cacheCallback.onCacheMiss(url, isDownloading);
            }

            if (investigateListener != null && isDownloading) {
                if (isDownloading) {
                    investigateListener.submitLog("GET - FALSE (downloading):  " + url);
                } else {
                    investigateListener.submitLog("GET - FALSE: " + url);
                }
            }
        } catch (Exception e) {
            if (investigateListener != null)
                investigateListener.submitLog("GET - FALSE (exception): " + inputUrl);
        }
        return null;
    }


    /**
     * Get WebResourceResponse represent for content of {@code request}
     * Override WebViewClient.shouldInterceptRequest and return this function
     *
     * @param request Object containing the details of the request
     * @return WebResourceResponse represent for content of resource if it is cached
     * otherwise null
     */
    @RequiresApi(api = 21)
    public WebResourceResponse getWebResourceResponse(WebResourceRequest request, Context context) {
        //care about GET request only
        if ("GET".equalsIgnoreCase(request.getMethod())) {
            return getWebResourceResponse(request.getUrl().toString(), context);
        }
        if (investigateListener != null) {
            investigateListener.submitLog("IGNORE - WebResourceResponse: " + request.getUrl());
        }
        return null;
    }

    /**
     * Force remove a resource from both memory and disk cache
     *
     * @param inputUrl url of resource that need removed
     */
    public void removeCacheItem(String inputUrl) {
        try {
            String memKey = ZBrowserUtil.removeTrackingParam(inputUrl, getCacheConfig());
            String diskKey = makeShortKey(memKey);

            MemoryLruCache memoryLruCache = mapUrlCacheMemory.get(memKey.hashCode());
            DiskLruCache diskLruCache = mapUrlCacheDisk.get(diskKey.hashCode());

            mapUrlCacheMemory.remove(memKey.hashCode());
            mapUrlCacheDisk.remove(diskKey.hashCode());

            if (memoryLruCache != null) {
                memoryLruCache.remove(memKey);
            }
            if (diskLruCache != null) {
                diskLruCache.remove(diskKey);
            }
        } catch (Exception e) {

        }
    }

    public void removeFromDiskCache(DiskLruCache cache, String key) {
        if (cache == null || cache.isClosed()) {
            return;
        }
        try {
            String keyShorter = makeShortKey(key);
            cache.remove(keyShorter);
            mapUrlCacheDisk.remove(keyShorter.hashCode());
        } catch (Exception e) {
            Log.w(LOG_TAG, "Remove cache file from disk cache error", e);
            return;
        }
    }

    public boolean checkCacheItem(String inputUrl) {
        try {
            String memKey = ZBrowserUtil.removeTrackingParam(inputUrl, getCacheConfig());
            String diskKey = makeShortKey(memKey);

            MemoryLruCache memoryLruCache = mapUrlCacheMemory.get(memKey.hashCode());
            DiskLruCache diskLruCache = mapUrlCacheDisk.get(diskKey.hashCode());

            if (memoryLruCache != null && diskLruCache != null) {
                CacheItem item = memoryLruCache.get(memKey);
                if(item == null){
                    item = getFromDiskCache(diskLruCache, memKey);
                    if(item == null) {
                        return false;
                    } else {
                        if (item.expireTime < System.currentTimeMillis() / 1000) {
                            removeCacheItem(inputUrl);
                            return false;
                        }
                        memoryLruCache._put(memKey, item);
                        return true;
                    }
                }
            } else {
                return false;
            }
        } catch (Exception e) {

        }
        return false;
    }

    public Stats getStatistic() {
        long currentMemCacheSize = 0;
        if (memoryHtmlCache != null) {
            currentMemCacheSize += memoryHtmlCache.getSize();
        }
        if (memoryStaticCache != null) {
            currentMemCacheSize += memoryStaticCache.getSize();
        }
        if (h5MemoryStaticCache != null) {
            currentMemCacheSize += h5MemoryStaticCache.getSize();
        }
        if (memoryPhotoCache != null) {
            currentMemCacheSize += memoryPhotoCache.getSize();
        }
        long currentDiskCacheSize = 0;
        if (diskHtmlCache != null) {
            currentDiskCacheSize += diskHtmlCache.size();
        }
        if (diskStaticCache != null) {
            currentDiskCacheSize += diskStaticCache.size();
        }
        if (h5DiskStaticCache != null) {
            currentDiskCacheSize += h5DiskStaticCache.size();
        }
        if (diskPhotoCache != null) {
            currentDiskCacheSize += diskPhotoCache.size();
        }
        statistic.currentMemCacheSize = currentMemCacheSize;
        statistic.currentDiskCacheSize = currentDiskCacheSize;

        return statistic;
    }

    /*for debugging only*/
    public long getHtmlMemCacheSize() {
        long result = 0;
        if (memoryHtmlCache != null) {
            result = memoryHtmlCache.getSize();
        }
        return result;
    }

    public long getStaticMemCacheSize() {
        long result = 0;
        if (memoryStaticCache != null) {
            result += memoryStaticCache.getSize();
        }
        if (h5MemoryStaticCache != null) {
            result += h5MemoryStaticCache.getSize();
        }
        if (memoryPhotoCache != null) {
            result += memoryPhotoCache.getSize();
        }
        return result;
    }

    public void clearCache() {
        try {
            if (memoryHtmlCache != null) {
                memoryHtmlCache.evictAll();
            }
            if (memoryStaticCache != null) {
                memoryStaticCache.evictAll();
            }
            if (h5MemoryStaticCache != null) {
                h5MemoryStaticCache.evictAll();
            }
            if (memoryPhotoCache != null) {
                memoryPhotoCache.evictAll();
            }
        } catch (Exception e) {

        }
    }
    /********************/

    /**
     * Destroy all instance inside zbrowser
     * Http client will be shutdown
     * All cached content in memory cache will be evicted
     * Cached content in disk cache will not be deleted
     */
    public void destroy() {
        try {
            nioHttpClient.shutdown();
            if (memoryHtmlCache != null) {
                memoryHtmlCache.evictAll();
                memoryHtmlCache = null;
            }
            if (memoryStaticCache != null) {
                memoryStaticCache.evictAll();
                memoryStaticCache = null;
            }
            if (h5MemoryStaticCache != null) {
                h5MemoryStaticCache.evictAll();
                h5MemoryStaticCache = null;
            }
            if (memoryPhotoCache != null) {
                memoryPhotoCache.evictAll();
                memoryPhotoCache = null;
            }
            if (diskHtmlCache != null) {
                diskHtmlCache.close();
                diskHtmlCache = null;
            }
            if (diskStaticCache != null) {
                diskStaticCache.close();
                diskStaticCache = null;
            }
            if (h5DiskStaticCache != null) {
                h5DiskStaticCache.close();
                h5DiskStaticCache = null;
            }
            if (diskPhotoCache != null) {
                diskPhotoCache.close();
                diskPhotoCache = null;
            }
        } catch (Exception e) {

        }
    }

    @Override
    protected void finalize() throws Throwable {
        destroy();
        super.finalize();
    }

    public ConfigCache getCacheConfig() {
        return this.cacheConfig != null ? this.cacheConfig : ConfigCache.DEFAULT_CONFIG;
    }

    public String checkUrlRedirect(String url) {
        String result = url;
        while (true) {
            String key = ZBrowserUtil.removeTrackingParam(result, getCacheConfig());
            CacheItem cacheItem = memoryHtmlRedirectCache.get(key);
            if (cacheItem == null) {
                cacheItem = getFromDiskCache(diskHtmlRedirectCache, key);
                if (cacheItem == null) {
                    break;
                }
                memoryHtmlRedirectCache._put(key, cacheItem);
            }
            result = new String(cacheItem.data);
            if (cacheItem.expireTime < System.currentTimeMillis() / 1000) {
                memoryHtmlRedirectCache.remove(key);
                if (diskHtmlRedirectCache != null) {
                    try {
                        diskHtmlRedirectCache.remove(makeShortKey(key));
                    } catch (Exception e) {

                    }
                }
            }
        }
        return result;
    }

    public static class Stats {
        public long pageStart;
        public long pageFinish;
        public long pageVisible;

        public int memCacheHit;
        public int memCacheMiss;
        public int diskCacheHit;
        public int diskCacheMiss;

        public long currentMemCacheSize;
        public long currentDiskCacheSize;

        public String currentBaseUrl;
        public boolean baseUrlHit;
        public AtomicInteger preloadedUrlCount;

        public AtomicInteger totalByteFromCache;

        public long openUrl;
        public long initWebviewStart;
        public long initWebviewEnd;
        public long jumpPreloadStart;
        public long jumpPreloadEnd;
        public long callLoadUrl;

        private Stats(long pageStart, int memCacheHit, int memCacheMiss, int diskCacheHit, int diskCacheMiss) {
            this.pageStart = pageStart;
            this.memCacheHit = memCacheHit;
            this.memCacheMiss = memCacheMiss;
            this.diskCacheHit = diskCacheHit;
            this.diskCacheMiss = diskCacheMiss;
            this.currentMemCacheSize = 0;
            this.currentDiskCacheSize = 0;

            pageFinish = -1;
            pageVisible = -1;

            currentBaseUrl = "";
            baseUrlHit = false;
            preloadedUrlCount = null;

            totalByteFromCache = new AtomicInteger();

            openUrl = -1;
            initWebviewStart = -1;
            initWebviewEnd = -1;
            jumpPreloadStart = -1;
            jumpPreloadEnd = -1;
            callLoadUrl = -1;
        }

        private static Stats createEmptyStats() {
            return new Stats(System.currentTimeMillis(), 0, 0, 0, 0);
        }

        public JSONObject toJSON() {
            JSONObject jsonObject = new JSONObject();
            try {
                if (pageFinish < 0) {
                    pageFinish = System.currentTimeMillis();
                }
                if (pageVisible < 0) {
                    pageVisible = System.currentTimeMillis();
                }

                jsonObject.put("pageFinish", pageFinish - pageStart);
                jsonObject.put("pageVisible", pageVisible - pageStart);
                jsonObject.put("memCacheHit", memCacheHit);
                jsonObject.put("memCacheMiss", memCacheMiss);
                jsonObject.put("diskCacheHit", diskCacheHit);
                jsonObject.put("diskCacheMiss", diskCacheMiss);
                jsonObject.put("currentMemCacheSize", currentMemCacheSize);
                jsonObject.put("currentDiskCacheSize", currentDiskCacheSize);
                jsonObject.put("preloadedUrlHit", baseUrlHit ? 1 : 0);
                jsonObject.put("preloadedUrlCount", preloadedUrlCount == null ? 0 : preloadedUrlCount.get());
                jsonObject.put("totalByteFromCache", totalByteFromCache.get());
                //reset preloaded url count
                if (preloadedUrlCount != null) {
                    preloadedUrlCount.set(0);
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return jsonObject;
        }

        @Override
        public String toString() {
            return toJSON().toString();
        }

        public String toDebugString() {
            long initWebviewTime = initWebviewEnd - initWebviewStart;
            long jumpTime = jumpPreloadEnd - jumpPreloadStart;
            long initLoadTime = pageStart - callLoadUrl;
            long loadRenderTime = pageFinish - pageStart;

            long visibleTime = pageVisible - openUrl;
            long totalTime = pageFinish - openUrl;

            return String.format("Init webview: %d\nJump: %d\nInitLoad: %d\nLoad+render: %d\n---------\nVisible time: %d\nTotal time: %d",
                    initWebviewTime,
                    jumpTime,
                    initLoadTime,
                    loadRenderTime,
                    visibleTime,
                    totalTime);
        }
    }

    //track for stats

    //tick when user click/touch open a link
    public void onOpenUrl() {
        tempStat.openUrl = System.currentTimeMillis();
    }

    //tick when start to init webview instance
    public void onInitWebviewStarted() {
        tempStat.initWebviewStart = System.currentTimeMillis();
    }

    //tick when init webview instance finish
    public void onInitWebviewEnded() {
        tempStat.initWebviewEnd = System.currentTimeMillis();
    }

    //tick when start to call jump preload
    public void onJumpPreloadStarted() {
        tempStat.jumpPreloadStart = System.currentTimeMillis();
    }

    //tick when call jump preload finish
    public void onJumpPreloadEnded() {
        tempStat.jumpPreloadEnd = System.currentTimeMillis();
    }

    //tick when start to call webview.loadUrl()
    public void onCallWebviewLoadUrl(String url) {
        statistic = Stats.createEmptyStats();
        statistic.currentBaseUrl = url;
        statistic.preloadedUrlCount = preloadedUrlCount;
        statistic.callLoadUrl = System.currentTimeMillis();
        statistic.openUrl = statistic.callLoadUrl;
        if (tempStat.openUrl != -1) {
            statistic.openUrl = tempStat.openUrl;
        }
        statistic.initWebviewStart = tempStat.initWebviewStart;
        statistic.initWebviewEnd = tempStat.initWebviewEnd;
        statistic.jumpPreloadStart = tempStat.jumpPreloadStart;
        statistic.jumpPreloadEnd = tempStat.jumpPreloadEnd;

        //reset temp stats
        tempStat = Stats.createEmptyStats();
    }
    // end track

    private void log(String log){
        if (investigateListener != null) {
            investigateListener.submitLog(log);
        }
    }

    public interface DownloadCallback {
        void onStart(String url);

        void onDataDownloading(String url, int percent, boolean isDone);

        void onError(String url);
    }

    public interface InvestigateListener {
        void submitLog(String msg);
    }

    public interface PreloadCallback {
        void onDone(@NonNull String url);

        void onFailed(@NonNull String url);
    }

    public interface CacheCallback {
        void onCacheHit(String url);

        void onCacheMiss(String url, boolean isDownloading);
    }


}
