/*
 * Copyright (c) 2012-2018 by Zalo Group.
 * All Rights Reserved.
 */
package com.zing.zalo.zbrowser.downloader;

import android.util.Log;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.nio.channels.SocketChannel;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;

/**
 *
 * @author datbt
 */
public class ConnectionManager {

	private final Map<String, LinkedBlockingQueue<ZConnection>> connectionPoolByDomain;
	private final Map<String, HttpURLConnection> connectionPool;
	public ConnectionManager() {
		connectionPoolByDomain = new ConcurrentHashMap<>();
		connectionPool = new ConcurrentHashMap<>();
	}	

	public static SocketChannel openNewConnection(String domain, int port) throws IOException {
		SocketChannel socketChannel = SocketChannel.open();//.setOption(StandardSocketOptions.TCP_NODELAY, Boolean.TRUE);
		socketChannel.configureBlocking(false);
		socketChannel.connect(new InetSocketAddress(domain, port));
		return socketChannel;
	}
	
	public static void closeConnection(SocketChannel connection) {
		try {
			connection.close();
		} catch (IOException ignore) {			
		}
	}

	public synchronized ZConnection borrowConnection(HttpUrl url) throws IOException {
		String key = String.format("%s:%d", url.domain, url.port);
		if (connectionPoolByDomain.containsKey(key)) {
			LinkedBlockingQueue<ZConnection> listConnection = connectionPoolByDomain.get(key);

			ZConnection connection = null;
			while (connection == null) {
				connection = listConnection.poll();
				if (listConnection.isEmpty()) {
					connectionPoolByDomain.remove(key);
				}
				if (connection != null) {
					Log.d("DUNGNN2", "isConnected: " + connection.socket.isConnected() + " isExpired: " + connection.isExpired() + " domain: " + url.domain + " port: " + url.port);
					if (connection.socket.isConnected() && !connection.isExpired()) {
						System.err.println("-------reuse connection-----");
						return connection;
					} else {
						closeConnection(connection.socket);
						connection = null;
					}
				} else {
					break;
				}
			}
		}
		return new ZConnection(url.domain, url.port, openNewConnection(url.domain, url.port), null);
	}

	public synchronized void returnConnection(HttpUrl url, SocketChannel connection, ZSSLConnection sslConnection) {
		if (connection == null) {
			return;
		}
		
		if (!connection.isConnected()) {
			closeConnection(connection);
			return;
		}
		try {
			String key = String.format("%s:%d", url.domain, url.port);
			if (connectionPoolByDomain.containsKey(key)) {
				LinkedBlockingQueue<ZConnection> listConnection = connectionPoolByDomain.get(key);
				listConnection.offer(new ZConnection(url.domain, url.port, connection, sslConnection));
			} else {
				LinkedBlockingQueue<ZConnection> listConnection = new LinkedBlockingQueue<>();
				if (listConnection.offer(new ZConnection(url.domain, url.port, connection, sslConnection))) {
					connectionPoolByDomain.put(key, listConnection);
				}
			}
		} catch (Exception e) {

		}
	}

	public void closeAllHttpURLConnection() {
		for (HttpURLConnection connection : connectionPool.values()) {
			if (connection != null) {
				connection.disconnect();
			}
		}
		connectionPool.clear();
	}
	public void closeAllConnection() {
		for (Map.Entry<String, LinkedBlockingQueue<ZConnection>> entry : connectionPoolByDomain.entrySet()) {
			LinkedBlockingQueue<ZConnection> listConnection = entry.getValue();
			if (listConnection != null) {
				while (!listConnection.isEmpty()) {
					ZConnection conn = listConnection.poll();
					closeConnection(conn.socket);
				}
			}
		}
		connectionPoolByDomain.clear();
	}
	
	public void cleanOldConnection() {
		for (Map.Entry<String, LinkedBlockingQueue<ZConnection>> entry : connectionPoolByDomain.entrySet()) {
			String domain = entry.getKey();
			LinkedBlockingQueue<ZConnection> listConnection = entry.getValue();
			if (listConnection != null) {
				for (ZConnection conn : listConnection) {
					if (conn.isExpired()) {
						closeConnection(conn.socket);
						listConnection.remove(conn);
					}
				}
				if (listConnection.isEmpty()) {
					connectionPoolByDomain.remove(domain);
				}
			}
		}
	}
}
