/*
 * Copyright (c) 2012-2018 by Zalo Group.
 * All Rights Reserved.
 */
package com.zing.zalo.zbrowser.downloader;

import android.util.Log;

import com.zing.zalo.zbrowser.cache.CacheItem;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 *
 * @author datbt
 */
public class HttpResponse {

	private static final String HTTP_RESP_STATUS_REGEX = "HTTP/\\S* (\\d+) \\w+( \\w+)*";

	private static final Pattern HTTP_RESP_HEADER_LINE_REGEX = Pattern.compile("(\\S+): (.+)");

	public static final int HTTP_RESP_BUFFER_SIZE = 64 * 1024;

	private final HttpHeader responseHeaders;

	private final List<byte[]> headerBuffer;

	private boolean isReadingHeader;

	private int statusCode;

	private final ByteBuffer readingBuffer;

	public final HttpRequest request;

	private boolean endOfResponse;

	private final AtomicBoolean isProcessingResp;

	private long contentLengthResponsed;

	private ChunkedResponse chunkedResponse;

	public HttpResponse(HttpRequest request) {
		headerBuffer = new ArrayList<>();
		responseHeaders = new HttpHeader();
		isReadingHeader = true;
		statusCode = 0;
		readingBuffer = ByteBuffer.allocate(HTTP_RESP_BUFFER_SIZE);
		readingBuffer.limit(0);
		this.request = request;
		endOfResponse = false;
		isProcessingResp = new AtomicBoolean(false);
		contentLengthResponsed = 0;
		chunkedResponse = null;
	}

	private void checkEndOfResponse() {
		boolean eor = false;
		if (statusCode == HttpURLConnection.HTTP_NO_CONTENT || (300 <= statusCode && statusCode < 400)) {
			eor = true;
		} else if (!isChunkedTransferEncoding() && !responseHeaders.hasHeader("content-length")) {
			//no content-length and chunked transfer encoding exist
			eor = true;
		} else {
			try {
				long contentLengthHeader =  Long.parseLong(responseHeaders.getHeader("content-length"));
				if (contentLengthHeader == contentLengthResponsed) {
					eor = true;
				}
			} catch (Exception e) {
				//ignore
			}
		}
		if (eor) {
			endOfResponse();
		}
	}

	public void endOfResponse() {
		if (endOfResponse) {
			return;
		}
		endOfResponse = true;
		if (!isProcessingResp.get()) {
			if (statusCode != 0) {
				request.setState(HttpRequest.State.DONE);
				if (checkRedirect() == null) {
					request.responseCallback.onResponseDone(request.url.fullUrl, statusCode, responseHeaders);
				}
			} else {
				request.setState(HttpRequest.State.ERROR);
				request.responseCallback.onResponseError(request.url.fullUrl, "Remote connection broken ");
			}
		}
	}

	private boolean isChunkedTransferEncoding() {
		return "chunked".equalsIgnoreCase(responseHeaders.getHeader("transfer-encoding"));
	}

	public String checkRedirect() {
		if (300 <= statusCode && statusCode < 400 && responseHeaders.hasHeader("location")) {
			return responseHeaders.getHeader("location");
		}
		return null;
	}

	public byte[] getRecvData() {
		if (!readingBuffer.hasRemaining()) {
			return new byte[]{};
		}
		byte[] bufDataClone = new byte[readingBuffer.remaining()];
		readingBuffer.get(bufDataClone);
		return bufDataClone;
	}

	public int receive(SocketChannel socketChannel) throws IOException {
		if (request.isSecure()) {
			//https request - wrong call
			return -1;
		}
		readingBuffer.clear();
		int read = socketChannel.read(readingBuffer);
		readingBuffer.flip();
		if (read > 0) {
			isProcessingResp.set(true);
		}
		return read;
	}

	public boolean parseHeader(byte[] data, HttpResponse.Callback callback) throws Exception {
		isProcessingResp.set(true);

		try {
//		byte[] data = new byte[buffer.remaining()];
//		buffer.get(data);
			boolean isStartHeaderLine = headerBuffer.isEmpty();
			char lastChar = '\0';
			if (!isStartHeaderLine) {
				byte[] lastBuffer = headerBuffer.get(headerBuffer.size() - 1);
				lastChar = (char) lastBuffer[lastBuffer.length - 1];
			}
			int headerLineOffset = 0;
			for (int i = 0; i < data.length; ++i) {
				if (HTTP.LF == data[i] && HTTP.CR == lastChar) {
					//end header line
					//parse this header first
					if (i - headerLineOffset > 1) {
						//byte[] currentBufOfHeaderLine = new byte[i - headerLineOffset - 1]; // size = currentIdx - headerLineOffset - 2(\r\n) + 1
						byte[] currentBufOfHeaderLine = Arrays.copyOfRange(data, headerLineOffset, i - 1);
						headerBuffer.add(currentBufOfHeaderLine);
					}
					ByteArrayOutputStream headerBinary = new ByteArrayOutputStream();
					for (byte[] buf : headerBuffer) {
						try {
							headerBinary.write(buf);
						} catch (IOException e) {
							//never get here
						}
					}
					String headerStr = new String(headerBinary.toByteArray());
					if (headerStr.matches(HTTP_RESP_STATUS_REGEX)) {
						if (statusCode == 0) {
							//status of response
							statusCode = Integer.parseInt(headerStr.replaceAll(HTTP_RESP_STATUS_REGEX, "$1"));
						} else {
							//already parse statusCode -- invalid header
							throw new IOException("Invalid http header: Already parse status of response. Header line: " + headerStr + " - statusCode: " + statusCode);
						}
					} else {
						if (headerStr.isEmpty()) {
							//end of response header
							headerBuffer.clear();
							headerLineOffset = i + 1;
							isReadingHeader = false;
							break;
						} else {
							Matcher headerLineMatcher = HTTP_RESP_HEADER_LINE_REGEX.matcher(headerStr);
							if (!headerLineMatcher.find()) {
								// invalid http header
								throw new IOException("Invalid http header: Header line is invalid: " + headerStr);
							}
							responseHeaders.setHeader(headerLineMatcher.group(1).toLowerCase().trim(), headerLineMatcher.group(2).trim());
						}
					}
					//reset state
					lastChar = '\0';
					headerBuffer.clear();
					headerLineOffset = i + 1;
				} else {
					lastChar = (char) data[i];
				}
			}
			if (isReadingHeader) {
				//still reading header
				if (headerLineOffset < data.length) {
					//incomplete header data -- store to buffer
					headerBuffer.add(Arrays.copyOfRange(data, headerLineOffset, data.length));
				}
				if (endOfResponse) {
					request.setErrorState("Unexpected close connection from remote server");
					callback.onResponseError(request.url.fullUrl, request.getErrorMessage());
				}
			} else {
				//finish reading header
				if (isChunkedTransferEncoding()) {
					chunkedResponse = new ChunkedResponse();
				}
				request.setState(HttpRequest.State.RECV_DATA);
				callback.onResponseHeader(request.url.fullUrl, statusCode, responseHeaders);
				if (checkRedirect() != null) {
					//redirect
					request.setState(HttpRequest.State.DONE);
					return false;
				}
				if (headerLineOffset < data.length) {
					ByteBuffer bodyRespBuf = ByteBuffer.wrap(data, headerLineOffset, data.length - headerLineOffset);
					receiveData(bodyRespBuf, callback);
				} else {
					checkEndOfResponse();
					if (endOfResponse) {
						//no body response
						request.setState(HttpRequest.State.DONE);
						callback.onResponseDone(request.url.fullUrl, statusCode, responseHeaders);
					}
				}
			}

			return isReadingHeader;
		} finally {
			isProcessingResp.set(false);
		}
	}

	public void receiveData(ByteBuffer buf, Callback callback) throws IOException {
		isProcessingResp.set(true);
		try {
			if (buf.hasRemaining()) {
				byte[] data = new byte[buf.remaining()];
				buf.get(data);
				contentLengthResponsed += data.length;
				if (chunkedResponse == null) {
					callback.onResponseBodyData(request.url.fullUrl, data, responseHeaders);
				} else {
					//chunked response
					byte[] parseChunked = chunkedResponse.parseChunked(data);
					if (parseChunked == null) {
						//not enough data - ignore
					} else if (parseChunked.length == 0) {
						//end of stream
						//never get here
						endOfResponse();
					} else {
						callback.onResponseBodyData(request.url.fullUrl, parseChunked, responseHeaders);
						byte[] checkEOF = chunkedResponse.parseChunked(new byte[]{});
						if (checkEOF != null) {
							if (checkEOF.length == 0) {
								// end of stream
								endOfResponse();
							} else {
								//illegal state - should not get here
								throw new IOException("Illegal state when check eof chunked encoding");

							}
						}
					}
				}
			}
			checkEndOfResponse();
			if (endOfResponse) {
				request.setState(HttpRequest.State.DONE);
				if (checkRedirect() == null) {
					callback.onResponseDone(request.url.fullUrl, statusCode, responseHeaders);
				}
			}
		} finally {
			isProcessingResp.set(false);
		}
	}

	public HttpHeader getHeaders() {
		return this.responseHeaders;
	}

	public static abstract class Callback {

		public abstract void onResponseHeader(String requestUrl, int statusCode, HttpHeader responseHeaders);

		public abstract void onResponseBodyData(String requestUrl, byte[] data, HttpHeader responseHeaders);

		public abstract void onResponseDone(String requestUrl, int statusCode, HttpHeader responseHeaders);

		public abstract void onResponseError(String requestUrl, String errorMessage);

		public void onResponseDone(String requestUrl, int statusCode, CacheItem cacheItem) {
		}
	}
}
