/*
 * Copyright (c) 2012-2018 by Zalo Group.
 * All Rights Reserved.
 */
package com.zing.zalo.zbrowser.downloader;

import android.util.Log;
import android.webkit.WebResourceResponse;

import com.zing.zalo.zbrowser.cache.CacheItem;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingDeque;

import javax.net.ssl.SSLContext;

/**
 *
 * @author datbt
 */
public class ZNIOHttpClient {

	private final Selector selector;

	private final ExecutorService workerThreadPool;

	private final ExecutorService executorService;

	private final SSLContext sslContext;

	private final ConnectionManager connectionManager;

	private int maxActiveConnection;

	private final LinkedBlockingDeque<Runnable> requestQueue;

	public ZNIOHttpClient(int workerThreads, int maxConnection) {
		try {
			selector = Selector.open();
			workerThreadPool = Executors.newFixedThreadPool(workerThreads);
			executorService = Executors.newCachedThreadPool();
			sslContext = SSLContext.getDefault();
			connectionManager = new ConnectionManager();
			maxActiveConnection = maxConnection < 1 ? Integer.MAX_VALUE : maxConnection;
			requestQueue = new LinkedBlockingDeque<>();
		} catch (Throwable e) {
			throw new NullPointerException("Cannot init ZNIODownloader: " + e.getMessage());
		}
		new Thread(new Runnable() {
			@Override
			public void run() {
				runIOThread();
			}
		}, "ZNIODownloader io").start();
	}

	public ZNIOHttpClient() {
		this(1, 1);
	}

	public void setMaxConnection(int maxConnection) {
		maxActiveConnection = maxConnection < 1 ? Integer.MAX_VALUE : maxConnection;
	}

	public WebResourceResponse downloadSync(final String url, final Map<String, String> requestHeaders, final HttpResponse.Callback responseCallback) throws IOException {
		if (responseCallback == null) {
			//do nothing if callback is not provided
			return null;
		}

		try {
			HttpUrl httpUrl = HttpUrl.parseUrl(url);

			HttpRequest request = new HttpRequest(httpUrl, requestHeaders, null, 0);
			HttpURLConnection connection = (HttpURLConnection) new URL(url).openConnection();
			if (url.endsWith(".jpg") || url.endsWith(".jpeg") || url.endsWith(".png") || url.endsWith(".gif")) {
				connection.setRequestProperty("Accept", "image/webp,image/*,*/*;q=0.8");
			}

			for (Map.Entry<String, String> header : request.userRequestHeaders.entrySet()) {
				connection.setRequestProperty(header.getKey(), header.getValue());
			}
			connection.connect();

			int statusCode = connection.getResponseCode(); // Lấy mã trạng thái HTTP
			Map<String, String> responseHeaders = new HashMap<>();
			for (Map.Entry<String, List<String>> header : connection.getHeaderFields().entrySet()) {
				if (header.getKey() != null && !header.getValue().isEmpty()) {
					responseHeaders.put(header.getKey().toLowerCase(), header.getValue().get(0));
				}
			}
			HttpHeader responseHeader = new HttpHeader(responseHeaders);
			responseCallback.onResponseHeader(url, statusCode, responseHeader);

			if (statusCode == HttpURLConnection.HTTP_OK) {
				InputStream inputStream = connection.getInputStream();
				String contentType = connection.getContentType();
				String mimeType = getMimeTypeFromContentType(contentType);
				String etag = responseHeader.getHeader("etag");
				if (etag == null) {
					etag = "";
				}
				// Tạo 2 pipe stream để xử lý đồng thời dữ liệu từ input stream cho webview và ghi vào cache
				PipedInputStream pipedInputStream = new PipedInputStream();
				PipedOutputStream pipedOutputStream = new PipedOutputStream(pipedInputStream);
				String finalEtag = etag;
				Runnable cacheFileJob = () -> {
					try {
						ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();

						byte[] buffer = new byte[1024];
						int bytesRead;
						while ((bytesRead = inputStream.read(buffer)) != -1) {
							pipedOutputStream.write(buffer, 0, bytesRead);
							byteArrayOutputStream.write(buffer, 0, bytesRead);
						}
						inputStream.close();
						pipedOutputStream.close();
						byteArrayOutputStream.close();

						CacheItem cacheItem = new CacheItem(Integer.MAX_VALUE, mimeType, byteArrayOutputStream.toByteArray(), finalEtag);
						Log.e("ZBROWSER", " DONE: " + url);
						responseCallback.onResponseDone(url, statusCode, cacheItem);
					} catch (IOException e) {
						e.printStackTrace();
					}
                };

				executorService.execute(cacheFileJob);
				return new WebResourceResponse(mimeType, "UTF-8", 200, "OK", responseHeaders, pipedInputStream);
			}
		} catch (Exception e) {
			responseCallback.onResponseError(url, e.getMessage());
		}

		return null;
	}

	private String getMimeTypeFromContentType(String contentType) {
		if (contentType == null) {
			return "text/plain";
		}
		int semicolonIndex = contentType.indexOf(';');
		if (semicolonIndex != -1) {
			return contentType.substring(0, semicolonIndex).trim();
		}
		return contentType.trim();
	}

	public void downloadAsync(final String url, final Map<String, String> requestHeaders, final HttpResponse.Callback responseCallback) throws IOException {
		if (responseCallback == null) {
			//do nothing if callback is not provided
			return;
		}
		if (!selector.isOpen()) {
			throw new IOException("Http client instance has been shutdown");
		}
		Runnable downloadJob = new Runnable() {
			@Override
			public void run() {
				HttpResponse.Callback asyncCallback = makeAsyncCallback(responseCallback);
				try {
					HttpUrl httpUrl = HttpUrl.parseUrl(url);
					ZConnection connection = connectionManager.borrowConnection(httpUrl);
					SocketChannel socketChannel = connection.socket;
					HttpRequest request = new HttpRequest(httpUrl, requestHeaders, asyncCallback, 0);
					if (connection.sslConnection != null) {
						request.setSSLConnection(connection.sslConnection);
					}

					socketChannel.register(selector, socketChannel.validOps(), request);
				} catch (Exception e) {
					asyncCallback.onResponseError(url, e.getMessage());
				}

			}
		};
		if (selector.keys().size() < maxActiveConnection) {
			workerThreadPool.execute(downloadJob);
		} else {
			requestQueue.add(downloadJob);
		}
	}

	private void redirect(String url, HttpRequest parentRequest) {
		System.err.println("------------\nRedirect: " + url);
		if (parentRequest.redirectCount + 1 > 3) {
			parentRequest.responseCallback.onResponseError(url, "Too many redirects: " + (parentRequest.redirectCount + 1));
			return;
		}

		try {
			HttpUrl httpUrl = HttpUrl.parseUrl(url);

			ZConnection connection = connectionManager.borrowConnection(httpUrl);
			SocketChannel socketChannel = connection.socket;
			HttpRequest request = new HttpRequest(httpUrl, parentRequest.userRequestHeaders, parentRequest.responseCallback, parentRequest.redirectCount + 1);
			if (connection.sslConnection != null) {
				request.setSSLConnection(connection.sslConnection);
			}
			socketChannel.register(selector, socketChannel.validOps(), request);
		} catch (IOException e) {
			parentRequest.responseCallback.onResponseError(url, e.getMessage());
		}
	}

	private HttpResponse.Callback makeAsyncCallback(final HttpResponse.Callback originalCallback) {
		if (originalCallback == null) {
			return originalCallback;
		}
		HttpResponse.Callback callback = new HttpResponse.Callback() {
			@Override
			public void onResponseHeader(final String requestUrl, final int statusCode, final HttpHeader responseHeaders) {
				workerThreadPool.execute(new Runnable() {
					@Override
					public void run() {
						originalCallback.onResponseHeader(requestUrl, statusCode, responseHeaders);
					}
				});
			}

			@Override
			public void onResponseBodyData(final String requestUrl, final byte[] data, final HttpHeader responseHeaders) {
				workerThreadPool.execute(new Runnable() {
					@Override
					public void run() {
						originalCallback.onResponseBodyData(requestUrl, data, responseHeaders);
					}
				});
			}

			@Override
			public void onResponseDone(final String requestUrl, final int statusCode, final HttpHeader responseHeaders) {
				workerThreadPool.execute(new Runnable() {
					@Override
					public void run() {
						originalCallback.onResponseDone(requestUrl, statusCode, responseHeaders);
					}
				});
			}

			@Override
			public void onResponseError(final String requestUrl, final String errorMessage) {
				workerThreadPool.execute(new Runnable() {
					@Override
					public void run() {
						originalCallback.onResponseError(requestUrl, errorMessage);
					}
				});
			}
		};

		return callback;
	}

	private void runIOThread() {
		long lastCleanOldConnectionTime = System.currentTimeMillis();
		while (selector.isOpen()) {
			try {
				if (!requestQueue.isEmpty() && selector.keys().size() < maxActiveConnection) {
					Runnable job = requestQueue.poll();
					if (job != null) {
						job.run();
					}
				}
				if (System.currentTimeMillis() - lastCleanOldConnectionTime > 10000) {
					//check if connection pool has expired connections and close them every 10 sec
					connectionManager.cleanOldConnection();
					lastCleanOldConnectionTime = System.currentTimeMillis();
				}
				int count;
				try {
					selector.select(100);
				} catch (IOException ex) {
					ex.printStackTrace();
					continue;
				}
				Set<SelectionKey> selectedKeys = selector.selectedKeys();
				Iterator<SelectionKey> selectedKeysIter = selectedKeys.iterator();
				while (selectedKeysIter.hasNext()) {
					SelectionKey selectionKey = selectedKeysIter.next();
					selectedKeysIter.remove();

					if (!selectionKey.isValid()) {
						selectionKey.cancel();
						continue;
					}

					SocketChannel socket = (SocketChannel) selectionKey.channel();

					if (selectionKey.isConnectable()) {
						try {
							if (!socket.finishConnect()) {
								continue;
							}
						} catch (Exception e) {

						}
					}

					Object attachment = selectionKey.attachment();
					if (!(attachment instanceof HttpRequest)) {
						//invalid attachment
						selectionKey.cancel();
						ConnectionManager.closeConnection(socket);
						continue;
					}
					final HttpRequest request = (HttpRequest) attachment;
					if (request.getState() == HttpRequest.State.DONE) {
						//request is fully success
						selectionKey.cancel();
						connectionManager.returnConnection(request.url, socket, request.getSLLConnection());
						continue;
					}

					if (!request.isSecure()) {
						if (selectionKey.isWritable()) {
							if (!request.isSendRequestDataDone()) {
								//send http request
								HttpRequest.State sendingState = sendHttpRequest(socket, request);
								if (sendingState == HttpRequest.State.ERROR) {
									request.responseCallback.onResponseError(request.url.fullUrl, request.getErrorMessage());
									selectionKey.cancel();
									ConnectionManager.closeConnection(socket);
									continue;
								}
							}
						}
						if (selectionKey.isReadable()) {
							if (request.isSendRequestDataDone()) {
								//receive http response
								HttpRequest.State receiveState = recvHttpResponse(socket, request);
								if (receiveState == HttpRequest.State.ERROR) {
									request.responseCallback.onResponseError(request.url.fullUrl, request.getErrorMessage());
									selectionKey.cancel();
									ConnectionManager.closeConnection(socket);
								}
							}
						}
					} else {
						if (!request.isSSLConnectionInitialized()) {
							try {
								ZSSLConnection sslConnection = new ZSSLConnection(socket, sslContext, workerThreadPool, HttpResponse.HTTP_RESP_BUFFER_SIZE, request.url);

								sslConnection.setCallback(makeSSLCallback(sslConnection, request, selectionKey));

								request.setSSLConnection(sslConnection);
							} catch (Exception e) {
								// init ssl connection error
								request.setErrorState(e.getMessage());
								request.responseCallback.onResponseError(request.url.fullUrl, request.getErrorMessage());
								selectionKey.cancel();
								ConnectionManager.closeConnection(socket);
								break;
							}
						} else {
							ZSSLConnection sllConnection = request.getSLLConnection();
							if (sllConnection.isConnectionReuse()) {
								ZSSLConnection.Callback callback = makeSSLCallback(sllConnection, request, selectionKey);
								sllConnection.setCallback(callback);
							}
						}
						HttpRequest.State processState = request.processSSLData();
						if (processState == HttpRequest.State.ERROR) {
							request.responseCallback.onResponseError(request.url.fullUrl, request.getErrorMessage());
							selectionKey.cancel();
							ConnectionManager.closeConnection(socket);
						}
					}
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	private ZSSLConnection.Callback makeSSLCallback(final ZSSLConnection sslConnection, final HttpRequest request, final SelectionKey selectionKey) {
		final SocketChannel socket = (SocketChannel) selectionKey.channel();
		final HttpResponse response = request.getResponse();
		ZSSLConnection.Callback callback = new ZSSLConnection.Callback()  {
			@Override
			public void onHandshakeSuccess() {
				String rawRequest = request.buildRawHttpRequest();
				request.setState(HttpRequest.State.SENDING_REQ);
				sslConnection.send(ByteBuffer.wrap(rawRequest.getBytes()));
			}

			@Override
			public void onDataResponse(byte[] data) {
				if (request.getState() == HttpRequest.State.SENDING_REQ) {
					request.setState(HttpRequest.State.RECV_HEADER);
				}
				processHttpResponse(data, request, socket);
				if (request.getState() == HttpRequest.State.ERROR) {
					request.responseCallback.onResponseError(request.url.fullUrl, request.getErrorMessage());
					selectionKey.cancel();
					ConnectionManager.closeConnection(socket);
				}
			}

			@Override
			public void onConnectionClosed() {
				response.endOfResponse();
			}
		};
		return callback;
	}

	private HttpRequest.State sendHttpRequest(SocketChannel socketChannel, HttpRequest request) {
		try {
			request.setState(HttpRequest.State.SENDING_REQ);
			request.send(socketChannel);
			if (request.isSendRequestDataDone()) {
				request.setState(HttpRequest.State.RECV_HEADER);
			}
		} catch (Exception e) {
			e.printStackTrace();
			request.setErrorState(e.getMessage());
		}
		return request.getState();
	}

	private HttpRequest.State recvHttpResponse(SocketChannel socketChannel, final HttpRequest request) {
		final HttpResponse response = request.getResponse();
		int receiveCount;
		do {
			try {
				receiveCount = response.receive(socketChannel);
				if (receiveCount == -1) {
					//end of stream -- notify end
					response.endOfResponse();
				}
			} catch (Exception e) {
				e.printStackTrace();
				return request.setErrorState(e.getMessage());
			}
			if (receiveCount > 0) {
				final byte[] bufData = response.getRecvData();
				processHttpResponse(bufData, request, socketChannel);
			}
		} while (receiveCount > 0);
		return request.getState();
	}

	private void processHttpResponse(final byte[] bufData, final HttpRequest request, final SocketChannel socketChannel) {
		try {
			HttpResponse response = request.getResponse();
			if (request.getState() == HttpRequest.State.RECV_HEADER) {
				boolean stillParsingHeader = response.parseHeader(bufData, request.responseCallback);
				if (!stillParsingHeader) {
					final String redirectUrl = response.checkRedirect();
					if (redirectUrl != null) {
						//set current request state as DONE
						request.setState(HttpRequest.State.DONE);
						//redirect to another url
						requestQueue.addFirst(new Runnable() {
							@Override
							public void run() {
								redirect(redirectUrl, request);
							}
						});
					}
				}
			} else if (request.getState() == HttpRequest.State.RECV_DATA) {
				response.receiveData(ByteBuffer.wrap(bufData), request.responseCallback);
			} else {
				throw new IllegalStateException("Illegal state while receiving response data: " + request.getState());
			}
		} catch (Exception e) {
			request.setErrorState(e.getMessage());
//			closeConnection(socketChannel);
//			ConnectionManager.closeConnection(socketChannel);
//			request.responseCallback.onResponseError(request.url.fullUrl, e.getMessage());
		}
	}

	public void shutdown() throws IOException {
		this.selector.close();
		this.workerThreadPool.shutdown();
		this.executorService.shutdown();
		this.connectionManager.closeAllConnection();
	}
}
