package com.zing.zalo.zalosdk.beaconsdk;

import android.Manifest;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.bluetooth.BluetoothAdapter;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.ServiceConnection;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;
import android.os.RemoteException;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import com.zing.zalo.zalosdk.beaconsdk.models.ZBeacon;
import com.zing.zalo.zalosdk.beaconsdk.util.AndroidVersionUtils;

import org.altbeacon.beacon.BeaconConsumer;
import org.altbeacon.beacon.BeaconManager;
import org.altbeacon.beacon.BeaconParser;
import org.altbeacon.beacon.Identifier;
import org.altbeacon.beacon.RangeNotifier;
import org.altbeacon.beacon.Region;

import org.altbeacon.beacon.service.RangedBeacon;
import org.altbeacon.beacon.startup.BootstrapNotifier;
import org.altbeacon.beacon.startup.RegionBootstrap;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import static com.zing.zalo.zalosdk.beaconsdk.Constant.BEACON_TIME_OUT_INTERVAL;
import static com.zing.zalo.zalosdk.beaconsdk.Constant.ERROR_BEACON_LIST_EMPTY;
import static com.zing.zalo.zalosdk.beaconsdk.Constant.ERROR_BEACON_LIST_EMPTY_MSG;
import static com.zing.zalo.zalosdk.beaconsdk.Constant.FOREGROUND_BETWEEN_SCAN_PERIOD;
import static com.zing.zalo.zalosdk.beaconsdk.Constant.FOREGROUND_SCAN_PERIOD;
import static com.zing.zalo.zalosdk.beaconsdk.models.ZBeacon.NOT_SET;

public class BeaconSDKImp implements IBeaconSDK {

  private final String TAG = "BeaconSDKImp";

  private static final int MSG_SET_LIST_MONITOR = 1;
  private static final int MSG_SET_LIST_RANGING = 2;
  private static final int MSG_BEACON_SERVICE_CONNECTED = 3;
  private static final int MSG_START_SCAN = 4;
  private static final int MSG_STOP_SCAN = 5;

  private static final int MSG_ADD_MONITOR = 6;
  private static final int MSG_ADD_RANGING = 7;
  private static final int MSG_STOP_MONITOR = 8;
  private static final int MSG_STOP_RANGING = 9;
  private static final int MSG_UPDATE_SCAN_PERIOD = 10;
  @Nullable
  protected static volatile IBeaconSDK sInstance = null;
  /**
   * Private lock object for singleton initialization protecting against denial-of-service attack.
   */
  private static final Object SINGLETON_LOCK = new Object();

  private BeaconManager beaconManager;
  private final Map<String, ZBeacon> monitorMapBeacons;
  private final Map<String, ZBeacon> rangingMapBeacons;
  private final Map<String, ZBeacon> availableMapBeacons;
  private ZBeacon noFilterBeacon = new ZBeacon("", "", NOT_SET, NOT_SET);
  private Context context;

  private boolean isScanning = false;
  private boolean isUserWantScanning = false;
  private boolean isRequireFilter = true;

  private IBeaconNotifier notifier;
  private ServiceHandler handler;
  private RegionBootstrap regionBootstrap;
  private boolean serviceConnected = false;
  private BeaconConsumer beaconConsumer;
  private BootstrapNotifier bootstrapNotifier;
  private RangeNotifier rangeNotifier;
  private boolean isBackgroundMode = false;
  private boolean isForegroundService = false;
  private HandlerThread thread;
  private BluetoothAdapter bluetoothAdapter;

  private static final int DEFAULT_FOREGROUND_SCAN_PERIOD = 5000;
  private static final int DEFAULT_FOREGROUND_BETWEEN_SCAN_PERIOD = 10000;
  private static final int DEFAULT_BEACON_TIME_OUT_INTERVAL = 30000;

  /**
   * the duration in milliseconds of each Bluetooth LE scan cycle to look for beacons.
   */
  private int foregroundScanPeriod = DEFAULT_FOREGROUND_SCAN_PERIOD;

  /**
   * the duration in milliseconds between each Bluetooth LE scan cycle to look for beacons.
   */
  private int foregroundBetweenScanPeriod = DEFAULT_FOREGROUND_BETWEEN_SCAN_PERIOD;

  /**
   * the duration in milliseconds between each Beacon status check to determine enter/exit state.
   */
  private int beaconTimeoutInterval = DEFAULT_BEACON_TIME_OUT_INTERVAL;

  private Runnable timerTick;

  public static IBeaconSDK getInstance(@NonNull Context context) {

    IBeaconSDK instance = sInstance;
    if (instance == null) {
      synchronized (SINGLETON_LOCK) {
        instance = sInstance;
        if (instance == null) {
          sInstance = instance = new BeaconSDKImp(context);
        }
      }
    }
    return instance;
  }

  private BeaconSDKImp(@NonNull Context context) {

    this.context = context.getApplicationContext();
    this.beaconManager = BeaconManager.getInstanceForApplication(context);
    this.monitorMapBeacons = new HashMap<>();
    this.rangingMapBeacons = new ConcurrentHashMap<>();
    this.availableMapBeacons = new ConcurrentHashMap<>();
    this.beaconConsumer = new InternalBeaconConsumer();
    this.bootstrapNotifier = new InternalBootstrapNotifier();
    this.rangeNotifier = new InternalRangeNotifier();
//    beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout("m:2-3=0215,i:4-19,i:20-21,i:22-23,p:24-24"));
//    beaconManager.setDebug(true);
    beaconManager.setRegionStatePersistenceEnabled(false);
    synchronized (this) {
      if (thread == null) {
        thread = new HandlerThread("ZBeaconService thread");
        thread.start();
      }
    }

    handler = new ServiceHandler(thread.getLooper());
//    setListMonitorBeacons(monitorListBeacon);

    //Bluetooth
    bluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
    IntentFilter filter = new IntentFilter(BluetoothAdapter.ACTION_STATE_CHANGED);
    context.registerReceiver(bluetoothReceiver, filter);
//    Log.d(TAG, "ver:" + BuildConfig.VERSION_NAME);
    timerTick = new Runnable() {
      @Override
      public void run() {
        try {
          if (isScanning) {
            List<ZBeacon> results = new ArrayList<>();

            try {
              for (Map.Entry<String, ZBeacon> stringBeaconEntry : availableMapBeacons.entrySet()) {
                String key = stringBeaconEntry.getKey();
                ZBeacon beacon = stringBeaconEntry.getValue();
                if (System.currentTimeMillis() - beacon.getLastSeen() > beaconTimeoutInterval) {

                  rangingMapBeacons.remove(key);
                  availableMapBeacons.remove(key);

                  beacon.setState(0);
                  results.add(beacon);
                }

              }
            } catch (Exception e1) {
              e1.printStackTrace();
            }

            for (ZBeacon beacon :
                results) {
              if (notifier != null) {
                Log.d(TAG, "DUNGNN timerTick EXIT_REGION "
                    + beacon.getUUID() + ";" + beacon.getMajor() + ";" + beacon.getMinor());
                notifier.onBeaconDisconnected(beacon);
              }
            }

            handler.postDelayed(this, beaconTimeoutInterval);
          }
        } catch (Exception e) {
          e.printStackTrace();
        }

      }
    };
  }

  @Override
  public IBeaconSDK setBeaconNotifier(IBeaconNotifier notifier) {
    this.notifier = notifier;
    return this;
  }

  @Override
  public IBeaconSDK setBundleOptions(Bundle bundle) {
    if (bundle == null || bundle.isEmpty()) {
      Log.w(TAG, "setBundleOptions empty data!!");
    }

    foregroundScanPeriod = bundle.getInt(FOREGROUND_SCAN_PERIOD, DEFAULT_FOREGROUND_SCAN_PERIOD);
    foregroundBetweenScanPeriod = bundle.getInt(FOREGROUND_BETWEEN_SCAN_PERIOD, DEFAULT_FOREGROUND_SCAN_PERIOD);
    beaconTimeoutInterval = bundle.getInt(BEACON_TIME_OUT_INTERVAL, DEFAULT_BEACON_TIME_OUT_INTERVAL);
    Log.d(TAG, "setBundleOptions ScanPeriod:" + foregroundScanPeriod
        + " BetweenScanPeriod:" +foregroundBetweenScanPeriod
        + " beaconTimeoutInterval:" + beaconTimeoutInterval);

    handler.obtainMessage(MSG_UPDATE_SCAN_PERIOD, bundle).sendToTarget();
    return this;
  }

  BroadcastReceiver bluetoothReceiver = new BroadcastReceiver() {
    @Override
    public void onReceive(Context context, Intent intent) {
      String action = intent.getAction();

      if (BluetoothAdapter.ACTION_STATE_CHANGED.equals(action)) {
        int state = getStateFromAdapterState(
            intent.getIntExtra(BluetoothAdapter.EXTRA_STATE, BluetoothAdapter.ERROR));

        if (state == Constant.BLUETOOTH_STATE_OFF) {
          Log.d(TAG, "BroadcastReceiver BT OFF");
          if (notifier != null) {
            notifier.onError(state, Constant.ERROR_BLUETOOTH_OFF_MSG);
          }
          Log.w(TAG, "Bluetooth Off!. Please check error code + message from IBeaconNotifier");

          stopScan();

        } else if (state == Constant.BLUETOOTH_STATE_ON) {
          Log.d(TAG, "BroadcastReceiver BT ON , isUserWantScanning(" + isUserWantScanning+ ")");
          if (isUserWantScanning) {
            startScan();
          }

        }


      }
    }
  };

  private int getStateFromAdapterState(int state) {

    int result = Constant.BLUETOOTH_STATE_OFF;
    switch (state) {
      case BluetoothAdapter.STATE_OFF :
        result = Constant.BLUETOOTH_STATE_OFF;
        break;
      case BluetoothAdapter.STATE_TURNING_OFF :
        result = Constant.BLUETOOTH_STATE_TURNING_OFF;
        break;
      case BluetoothAdapter.STATE_ON :
        result = Constant.BLUETOOTH_STATE_ON;
        break;
      case BluetoothAdapter.STATE_TURNING_ON :
        result = Constant.BLUETOOTH_STATE_TURNING_ON;
        break;
    }
    return result;
  }

  private Region genRegion(ZBeacon beacon) {
//    String name = beacon.getName();
    String uuid = beacon.getUUID();
    int major = beacon.getMajor();
    int minor = beacon.getMinor();

    Identifier id1 = TextUtils.isEmpty(uuid) ? null : Identifier.parse(uuid);
    Identifier id2 = major == NOT_SET ? null : Identifier.fromInt(major);
    Identifier id3 = minor == NOT_SET ? null : Identifier.fromInt(minor);
    return new Region(uuid + ";" + major + ";" +minor, id1, id2, id3);
  }

  @Override
  protected void finalize() throws Throwable {
    try {
      if (thread != null) {
        if (AndroidVersionUtils.isJellyBeanMR2AndAbove()) {
          thread.quitSafely();
        } else {
          thread.quit();
        }

        thread = null;
        notifier = null;

        if (context != null) {
          context.unregisterReceiver(bluetoothReceiver);
        }
      }
    } catch (Exception e) {
      e.printStackTrace();
    } finally {
      super.finalize();
    }
  }

//  @Override
//  public List<Beacon> getListMonitorBeacons() {
//    return new ArrayList<>(monitorMapBeacons.values());
//  }
//
//  @Override
//  public List<Beacon> getListRangingBeacons() {
//    return new ArrayList<>(rangingMapBeacons.values());
//  }

  @Override
  public boolean isScanning() {
//    if (beaconManager == null || !beaconManager.isAnyConsumerBound()) {
//      isScanning = false;
//    }
    return isScanning;
  }

//  @Override
//  public void addMonitorBeacon(List<Beacon> beacons) {
//    if (beacons != null && !beacons.isEmpty()) {
////      handler.obtainMessage(MSG_ADD_MONITOR, beacons).sendToTarget();
//    }
//  }

//  @Override
//  private void addBeacons(List<ZBeacon> beacons) {
//    if (beacons != null && !beacons.isEmpty()) {
//      handler.obtainMessage(MSG_ADD_RANGING, beacons).sendToTarget();
//    }
//  }

//  @Override
//  public void stopMonitorBeacon(List<Beacon> beacons) {
//    if (beacons != null && !beacons.isEmpty()) {
////      handler.obtainMessage(MSG_STOP_MONITOR, beacons).sendToTarget();
//    }
//  }

//  @Override
//  public void stopRangingBeacon(List<Beacon> beacons) {
//    if (beacons != null && !beacons.isEmpty()) {
//      handler.obtainMessage(MSG_STOP_RANGING, beacons).sendToTarget();
//    }
//  }

//  @Override
//  public void setListMonitorBeacons(List<Beacon> monitorBeacons) {
//    if (monitorBeacons != null && !monitorBeacons.isEmpty()) {
////      handler.obtainMessage(MSG_SET_LIST_MONITOR, monitorBeacons).sendToTarget();
//    }
//  }

  List<ZBeacon> currentBeacons;
  @Override
  public void setListBeacons(List<ZBeacon> beacons) {
    if (beacons != null && !beacons.isEmpty()) {

      if (!beacons.equals(currentBeacons)) {
        currentBeacons = new ArrayList<>(beacons);
//        try {
//          availableMapBeacons.clear();
//        } catch (Exception e) {
//          e.printStackTrace();
//        }
        handler.obtainMessage(MSG_SET_LIST_MONITOR, beacons).sendToTarget();
//        handler.obtainMessage(MSG_SET_LIST_RANGING, beacons).sendToTarget();
      } else {
        Log.d(TAG, "DUNGNN setListBeacons equal return");
      }

    } else {
      if (notifier != null) {
        notifier.onError(ERROR_BEACON_LIST_EMPTY, ERROR_BEACON_LIST_EMPTY_MSG);
      }
    }

  }

  @Override
  public void startBeacons() {
//    this.isBackgroundMode = isBackgroundMode;
    this.isRequireFilter = true;
    this.isUserWantScanning = true;
    startScan();
  }

  @Override
  public void stopBeacons() {
    this.isUserWantScanning = false;
    stopScan();
  }

  @Override
  public void scanAll() {
    this.isRequireFilter = false;
    this.isUserWantScanning = true;
    startScan();
  }

  @Override
  public List<ZBeacon> getAvailableBeacons() {
    Log.d(TAG, "getAvailableBeacons b" + availableMapBeacons.size());
    List<ZBeacon> zBeacons = new ArrayList<>();
    try {
      if (!availableMapBeacons.isEmpty()) {

        for (Map.Entry<String, ZBeacon> entry : availableMapBeacons.entrySet()) {
          ZBeacon beacon = entry.getValue();
          if (beacon.getLastSeen() > 0 && System.currentTimeMillis() - beacon.getLastSeen() < beaconTimeoutInterval) {
            zBeacons.add(beacon);
          } else {
            Log.d(TAG, "miss: " + beacon);
          }
        }

      }
    } catch (Exception e) {
      e.printStackTrace();
    }
    Log.d(TAG, "getAvailableBeacons a" + zBeacons.size());
    return zBeacons;
  }

  private int checkBluetooth() {
    int result = 0;
    if (bluetoothAdapter == null || !bluetoothAdapter.isEnabled()) {
      result = Constant.ERROR_BLUETOOTH_OFF;
      if (notifier != null) {
        notifier.onError(result, Constant.ERROR_BLUETOOTH_OFF_MSG);
      }
    }

    return result;
  }

  private void startScan() {
    if (checkPermission() != 0) {
      Log.w(TAG, "check Permission FAILED!. Please check error code + message from IBeaconNotifier");
      return;
    }
    if (checkBluetooth() != 0) {
      Log.w(TAG, "Bluetooth Off!. Please check error code + message from IBeaconNotifier");
      return;
    }

    if (isScanning) {
      Log.w(TAG, "_startScan Already start !!");
      return;
    }

    isScanning = true;
    handler.obtainMessage(MSG_START_SCAN).sendToTarget();
  }

  private void stopScan() {

    if (!isScanning) {
      Log.w(TAG, "_stopScan Already stop !!");
      return;
    }
    isScanning = false;

    handler.obtainMessage(MSG_STOP_SCAN).sendToTarget();
  }

  private int checkPermission() {

    int result = 0;
    String msg = "";
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
      if (!isPermissionGranted(Manifest.permission.ACCESS_COARSE_LOCATION)) {
        result = Constant.ERROR_ACCESS_COARSE_LOCATION_DENIED;
        msg = Constant.ERROR_ACCESS_COARSE_LOCATION_DENIED_MSG;
      }
      if (!isPermissionGranted(Manifest.permission.ACCESS_FINE_LOCATION)) {
        result = Constant.ERROR_ACCESS_FINE_LOCATION_DENIED;
        msg = Constant.ERROR_ACCESS_FINE_LOCATION_DENIED_MSG;
      }

//      if (isPermissionGranted(Manifest.permission.ACCESS_COARSE_LOCATION)) {
//        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
//          if (isPermissionGranted(Manifest.permission.ACCESS_BACKGROUND_LOCATION) == false) {
//            result = Constant.ERROR_ACCESS_BACKGROUND_LOCATION_DENIED;
//            msg = Constant.ERROR_ACCESS_BACKGROUND_LOCATION_DENIED_MSG;
//          }
//        }
//      } else {
//        result = Constant.ERROR_ACCESS_COARSE_LOCATION_DENIED;
//        msg = Constant.ERROR_ACCESS_COARSE_LOCATION_DENIED_MSG;
//      }
    }

    if (notifier != null && result != 0) {
      notifier.onError(result, msg);
    }

    return result;
  }

  private boolean isPermissionGranted(String permission) {
    return ContextCompat.checkSelfPermission(context, permission) == PackageManager.PERMISSION_GRANTED;
  }

  private String getKeyBeaconOrRegion(Object o) {
    String id = "";
    int major = 0;
    int minor = 0;
    try {
      if (o instanceof Region) {
        Region ob = (Region) o;
        Log.d("BeaconSDK" , "get Key Region: " + ob.toString());
        id    = ob.getId1() == null ? "":ob.getId1().toString();
        major = ob.getId2() == null ? 0:ob.getId2().toInt();
        minor = ob.getId3() == null ? 0:ob.getId3().toInt();
      } else if (o instanceof org.altbeacon.beacon.Beacon) {
        org.altbeacon.beacon.Beacon ob = (org.altbeacon.beacon.Beacon) o;
        id    = ob.getId1() == null ? "":ob.getId1().toString();
        major = ob.getId2() == null ? 0:ob.getId2().toInt();
        minor = ob.getId3() == null ? 0:ob.getId3().toInt();
      } else if (o instanceof ZBeacon) {
        ZBeacon ob = (ZBeacon) o;
        id = ob.getUUID();
        major = ob.getMajor();
        minor = ob.getMinor();
      }
    } catch (Exception e) {
      e.printStackTrace();
    }

    String result = id + ";" + major + ";" +minor;
    return result.toUpperCase();
  }

  private final class InternalRangeNotifier implements RangeNotifier{

    @Override
    public void didRangeBeaconsInRegion(Collection<org.altbeacon.beacon.Beacon> collection, Region region) {
      if (collection == null || collection.isEmpty()) {
        Log.d(TAG, "DUNGNN collection empty return!!");
        return;
      }

      List<ZBeacon> beaconList = new ArrayList<>();
      for (org.altbeacon.beacon.Beacon beacon : collection) {
        if (beacon != null) {

          String key = getKeyBeaconOrRegion(beacon);
          ZBeacon o = rangingMapBeacons.get(key);
          if (o == null) {
            o = new ZBeacon(beacon.getId1().toString(),
                beacon.getId1().toString(),
                beacon.getId2() != null ? beacon.getId2().toInt() : 0,
                beacon.getId3() != null ? beacon.getId3().toInt() : 0);
          }
          o.setMajor(beacon.getId2() != null ? beacon.getId2().toInt() : 0);
          o.setMinor(beacon.getId3() != null ? beacon.getId3().toInt() : 0);

          o.setDistance(beacon.getDistance());
          o.setLastSeen(System.currentTimeMillis());
          o.setRssi(beacon.getRssi());
          o.setTxPower(beacon.getTxPower());

//          if (!isRequireFilter) {
          if (o.getState() == 0) {
            o.setState(1);
            if (notifier != null) {
              Log.d(TAG, "DUNGNN didRangeBeaconsInRegion ENTER_REGION " + o.getUUID() + ";" + o.getMajor() + ";" + o.getMinor());
              notifier.onBeaconConnected(o);
            }
          }
//          }
          Log.d(TAG, "DUNGNN didRangeBeaconsInRegion beacon: " + o);
          rangingMapBeacons.put(key, o);
          availableMapBeacons.put(key, o);

          beaconList.add(o);
        } else {
          Log.d(TAG, "DUNGNN didRangeBeaconsInRegion beacon NUll");
        }
      }

      if (!beaconList.isEmpty() && notifier != null) {
        notifier.onRangeBeacons(beaconList);
      }

      if (notifier == null) {
        Log.w(TAG, "DUNGNN didRangeBeaconsInRegion notifier NUll");
      }
    }
  }

  private final class InternalBootstrapNotifier implements BootstrapNotifier {

    @Override
    public Context getApplicationContext() {
      return context;
    }

    @Override
    public void didEnterRegion(Region region) {
      Log.d(TAG, "didEnterRegion " + region);

      String key = getKeyBeaconOrRegion(region);
      ZBeacon beacon = monitorMapBeacons.get(key);
      if (beacon == null) {
        beacon = new ZBeacon(region.getUniqueId(),
            region.getId1() == null ? "null" : region.getId1().toString(),
            region.getId2() == null ? 0: region.getId2().toInt(),
            region.getId3() == null ? 0: region.getId3().toInt());
      }
      if (beacon.getState() == 0) {
        beacon.setState(1);
        if (notifier != null) {
          notifier.onBeaconConnected(beacon);
        }
      }

      beacon.setLastSeen(System.currentTimeMillis());
      Log.d(TAG, "put availableMapBeacons " + beacon);

      availableMapBeacons.put(key, beacon);


      
    }

    @Override
    public void didExitRegion(Region region) {

//      String key = getKeyBeaconOrRegion(region);
//      ZBeacon beacon = monitorMapBeacons.get(key);
//      if (beacon == null) {
//        beacon = new ZBeacon(region.getUniqueId(),
//            region.getId1() == null ? "null" : region.getId1().toString(),
//            region.getId2() == null ? 0: region.getId2().toInt(),
//            region.getId3() == null ? 0: region.getId3().toInt());
//      }
//      beacon.setState(0);
//
//      Log.d(TAG, "DUNGNN beacon exit key: " + key);
//      if (rangingMapBeacons.containsKey(key)) {
//        Log.d(TAG, "DUNGNN update ranging");
//        rangingMapBeacons.get(key).setState(0);
//      }
//      availableMapBeacons.remove(key);
//
//      if (notifier != null) {
//        notifier.onBeaconDisconnected(beacon);
//      }
    }

    @Override
    public void didDetermineStateForRegion(int i, Region region) {

    }
  }

  private final class InternalBeaconConsumer implements BeaconConsumer {

    private Intent serviceIntent;

    /**
     * Method reserved for system use
     */
    @Override
    public void onBeaconServiceConnect() {
      Log.d(TAG, "InternalBeaconConsumer onBeaconServiceConnect");
      serviceConnected = true;
      handler.obtainMessage(MSG_BEACON_SERVICE_CONNECTED).sendToTarget();
    }

    @Override
    public boolean bindService(Intent intent, ServiceConnection conn, int arg2) {
      Log.d(TAG, "InternalBeaconConsumer bindService");
      this.serviceIntent = intent;
      context.startService(intent);
      return context.bindService(intent, conn, arg2);

    }

    @Override
    public Context getApplicationContext() {
      return context;
    }

    @Override
    public void unbindService(ServiceConnection conn) {
      Log.d(TAG, "InternalBeaconConsumer unbindService");
      context.unbindService(conn);
      context.stopService(serviceIntent);
      serviceConnected = false;
    }
  }

  private final class ServiceHandler extends Handler {
    public ServiceHandler(Looper looper) {
      super(looper);
    }


    @Override
    public void handleMessage(Message msg) {
      switch (msg.what) {
        case MSG_SET_LIST_MONITOR:
          _setListMonitorBeacons(msg);
          break;
//        case MSG_SET_LIST_RANGING:
//          _setListRangingBeacons(msg);
//          break;
        case MSG_BEACON_SERVICE_CONNECTED:
          _onServiceConnected();
          break;
        case MSG_START_SCAN:
          _startScan();
          break;
        case MSG_STOP_SCAN:
          _stopScan();
          break;
        case MSG_UPDATE_SCAN_PERIOD:
          _updatePeriod();
          break;

      }

    }

    private void _updatePeriod() {
      try {
        if (isScanning && beaconManager != null) {
          RangedBeacon.setSampleExpirationMilliseconds(5000);
          beaconManager.setForegroundBetweenScanPeriod(foregroundBetweenScanPeriod);
          beaconManager.setForegroundScanPeriod(foregroundScanPeriod);
          beaconManager.updateScanPeriods();
        }

      } catch (Exception e) {
        e.printStackTrace();
      }
    }

    private void _startScan() {

      if (isBackgroundMode) {

        if (beaconManager.isBound(beaconConsumer)) {
          //Unbind foreground consumer
          beaconManager.unbind(beaconConsumer);
        }

        if (regionBootstrap != null) {
          regionBootstrap.disable();
          regionBootstrap = null;
        }

        beaconManager.setBackgroundMode(true);

        if (regionBootstrap == null) {
          Iterator<Map.Entry<String, ZBeacon>> iter = monitorMapBeacons.entrySet().iterator();
          List<Region> regions = new ArrayList<>();
          while (iter.hasNext()) {
            Map.Entry<String, ZBeacon> entry = iter.next();
            ZBeacon beacon = entry.getValue();
            Region region = genRegion(beacon);
            regions.add(region);
          }

          Log.d(TAG, "_startScan background use regionbootstrap consumer");
          regionBootstrap = new RegionBootstrap(bootstrapNotifier, regions);
          beaconManager.addRangeNotifier(rangeNotifier);

          if (beaconManager.isAnyConsumerBound()) {
            Log.d(TAG, "_startScan background "
                + (rangingMapBeacons.isEmpty() ? " ranging empty -> do nothing" : " start ranging"));
            for (Map.Entry entry : rangingMapBeacons.entrySet()) {
              ZBeacon beacon = (ZBeacon) entry.getValue();
              try {
                Region region = genRegion(beacon);
                beaconManager.startRangingBeaconsInRegion(region);
              } catch (RemoteException e) {
                e.printStackTrace();
              }

            }
          }
        } else {
          Log.w(TAG, "_startScan regionBootstrap NOT NULL, check stopScan cycle");
        }

      } else {

        if (regionBootstrap != null) {
          regionBootstrap.disable();
          regionBootstrap = null;
        }

        if (!beaconManager.isBound(beaconConsumer)) {

//          _setupService();

          Log.d(TAG, "_startScan normal use sdk consumer");

          beaconManager.bind(beaconConsumer);
        }
        beaconManager.setBackgroundMode(false);
        RangedBeacon.setSampleExpirationMilliseconds(5000);
        beaconManager.setForegroundBetweenScanPeriod(foregroundBetweenScanPeriod);
        beaconManager.setForegroundScanPeriod(foregroundScanPeriod);
        try {
          beaconManager.updateScanPeriods();
        } catch (RemoteException e) {
          e.printStackTrace();
        }

//        beaconManager.setEnableScheduledScanJobs(true);
        beaconManager.addMonitorNotifier(bootstrapNotifier);
        beaconManager.addRangeNotifier(rangeNotifier);
      }

//      if (!isRequireFilter) {
        handler.removeCallbacks(timerTick);
        handler.postDelayed(timerTick, beaconTimeoutInterval);
//      }

    }

    private void _stopScan() {

//      handler.removeCallbacks(timerTick);
      beaconManager.removeAllRangeNotifiers();
      beaconManager.removeAllMonitorNotifiers();
      currentBeacons = null;
//      if (!isRequireFilter) {
        handler.removeCallbacks(timerTick);
//      }
      if (isBackgroundMode) {
        Log.d(TAG, "_stopScan background regionBootstrap disable");
        if (regionBootstrap != null) {
          regionBootstrap.disable();
          regionBootstrap = null;
        }
      } else {

        Log.d(TAG, "_stopScan beaconManager unbind");
        try {
          beaconManager.stopAllRangingBeaconsInRegion();
          beaconManager.stopAllMonitoringBeaconsInRegion();
        } catch (RemoteException e) {
          e.printStackTrace();
        }
        beaconManager.unbind(beaconConsumer);

      }


    }

    private void _onServiceConnected() {
//      long time = System.currentTimeMillis();
      Log.d(TAG, "_onServiceConnected start monitor + ranging beacons");
      try {
        beaconManager.stopAllMonitoringBeaconsInRegion();
        beaconManager.stopAllRangingBeaconsInRegion();

        List<Region> monitorRegions = new ArrayList<>();
        for (Map.Entry entry : monitorMapBeacons.entrySet()) {
          ZBeacon beacon = (ZBeacon) entry.getValue();
          Region region = genRegion(beacon);
          Log.d(TAG, "monitor: " + beacon);
          monitorRegions.add(region);
        }

        List<Region> rangingRegions = new ArrayList<>();
        for (Map.Entry entry : rangingMapBeacons.entrySet()) {
          ZBeacon beacon = (ZBeacon) entry.getValue();
          Region region = genRegion(beacon);
//          Log.d(TAG, "ranging: " + beacon);
          rangingRegions.add(region);
        }
        if (isScanning) {

          if (!isRequireFilter) {
            Log.d(TAG, "_onServiceConnected ko filter");
            Region region = genRegion(noFilterBeacon);
            beaconManager.startMonitoringBeaconsInRegion(region);
            beaconManager.startRangingBeaconsInRegion(region);
          } else {
            if (rangingRegions.isEmpty() /*|| monitorRegions.isEmpty()*/) {
              if (notifier != null) {
                notifier.onError(ERROR_BEACON_LIST_EMPTY, ERROR_BEACON_LIST_EMPTY_MSG);
              }
            } else {
              Log.d(TAG, "_onServiceConnected co filter");
              if (!rangingRegions.isEmpty()) {
                beaconManager.startRangingBeaconsInRegion(rangingRegions);
              }
              if (!monitorRegions.isEmpty()) {
                beaconManager.startMonitoringBeaconsInRegion(monitorRegions);
              }
            }
          }
        }

//        Log.d(TAG, "_onServiceConnected dur: " + (System.currentTimeMillis() - time));
      } catch (RemoteException e) {
        e.printStackTrace();
      }
    }

    private void _setListMonitorBeacons(Message msg) {
      try {
//        Log.d(TAG, "_setListMonitorBeacons remove old monitor beacons");
//        long duration = System.currentTimeMillis();
        monitorMapBeacons.clear();
        rangingMapBeacons.clear();

        List<Region> filterList = new ArrayList<>();
        List<ZBeacon> list = (ArrayList<ZBeacon>) msg.obj;
//        Log.d(TAG, "_setListMonitorBeacons start new monitor beacons");
        for (ZBeacon beacon : list) {
          String key = getKeyBeaconOrRegion(beacon);
          Region region = genRegion(beacon);
          filterList.add(region);
          Log.d(TAG, "_setListMonitorBeacons beacon: " + beacon);
          monitorMapBeacons.put(key, beacon);
          rangingMapBeacons.put(key, beacon);
        }

        try {
          if (!filterList.isEmpty() && isScanning) {

            handler.removeCallbacks(timerTick);
            handler.postDelayed(timerTick, beaconTimeoutInterval);

            beaconManager.stopAllMonitoringBeaconsInRegion();
            beaconManager.stopAllRangingBeaconsInRegion();
            beaconManager.startMonitoringBeaconsInRegion(filterList);
            beaconManager.startRangingBeaconsInRegion(filterList);
          }

        } catch (RemoteException e) {
          if (notifier != null) {
            notifier.onError(
                Constant.ERROR_UNKNOWN, "set whiteList add: " + e.getMessage());
          }
          e.printStackTrace();
        }

//        Log.d(TAG, "_setListMonitorBeacons dur: " + (System.currentTimeMillis() - duration));
      } catch (Exception e) {
        e.printStackTrace();
      }
    }

    private void _setListRangingBeacons(Message msg) {
      try {
        Log.d(TAG, "_setListRangingBeacons remove old ranging beacons");
        long duration = System.currentTimeMillis();
        rangingMapBeacons.clear();

        try {
          beaconManager.stopAllRangingBeaconsInRegion();
//          if (!filterList.isEmpty()) {
//            beaconManager.stopRangingBeaconsInRegion(filterList);
//          }
        } catch (RemoteException e) {
          if (notifier != null) {
            notifier.onError(
                Constant.ERROR_UNKNOWN, "set ranging remove: " + e.getMessage());
          }
          e.printStackTrace();
        }

        List<Region> filterList = new ArrayList<>();
        List<ZBeacon> list = (ArrayList<ZBeacon>) msg.obj;
        Log.d(TAG, "_setListRangingBeacons start new ranging beacons");
        for (ZBeacon beacon : list) {
          String key = getKeyBeaconOrRegion(beacon);
          Region region = genRegion(beacon);
          filterList.add(region);
          Log.d(TAG, "_setListRangingBeacons beacon: " + beacon);

          rangingMapBeacons.put(key, beacon);
        }

        try {
          if (!filterList.isEmpty() && isScanning) {
            beaconManager.startRangingBeaconsInRegion(filterList);
          }

        } catch (RemoteException e) {
          if (notifier != null) {
            notifier.onError(
                Constant.ERROR_UNKNOWN, "set ranging add: " + e.getMessage());
          }
          e.printStackTrace();
        }

        Log.d(TAG, "_setListRangingBeacons dur: " + (System.currentTimeMillis() - duration));
      } catch (Exception e) {
        e.printStackTrace();
      }

    }

  }

}
