package org.altbeacon.beacon.service;

import android.os.SystemClock;
import androidx.annotation.RestrictTo;
import androidx.annotation.RestrictTo.Scope;

import org.altbeacon.beacon.logging.LogManager;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/**
 * Calculate a RSSI value on base of an arbitrary list of measured RSSI values
 * The list is clipped by a certain length at start and end and the average
 * is calculate by simple arithmetic average
 */
public class RunningAverageRssiFilter implements RssiFilter {

    private static final String TAG = "RunningAverageRssiFilter";
    public static final long DEFAULT_SAMPLE_EXPIRATION_MILLISECONDS = 20000; /* 20 seconds */
    private static long sampleExpirationMilliseconds = DEFAULT_SAMPLE_EXPIRATION_MILLISECONDS;
    private ArrayList<Measurement> mMeasurements = new ArrayList<Measurement>();
    private static final double DEFAULT_DELTA = 1;

    @Override
    public void addMeasurement(Integer rssi) {
        Measurement measurement = new Measurement();
        measurement.rssi = rssi;
        measurement.timestamp = SystemClock.elapsedRealtime();
        mMeasurements.add(measurement);
//        LogManager.w(TAG, "addMeasurement: " + rssi);
    }

    @Override
    public boolean noMeasurementsAvailable() {
        return mMeasurements.size() == 0;
    }


    @Override
    public int getMeasurementCount() { return mMeasurements.size(); }

    @Override
    public double calculateRssi() {
        refreshMeasurements();
        int size = mMeasurements.size();
        if (size == 0) {
            return 0;
        }
        try {
            int startIndex = 0;
            int endIndex = size -1;
//        if (size > 2) {
//            startIndex = size/10+1;
//            endIndex = size-size/10-2;
//        }

            double sum = 0;
            int previousRSSI = 0;
            List<AvarageRSSI> avarageRSSIS = new ArrayList<>();
            int count = 0;
            int weight =0;
            for (int i = startIndex; i <= endIndex; i++) {
                int currentRSSI = mMeasurements.get(i).rssi;
                if (i == startIndex) {
                    weight = currentRSSI;
                    count = 1;
                } else {
                    double avarage = (currentRSSI + previousRSSI)*1.0 /2;
                    if (Math.abs(currentRSSI - avarage) < DEFAULT_DELTA) {
                        weight += currentRSSI;
                        ++count;
                        if (i == endIndex) {
                            avarageRSSIS.add(new AvarageRSSI(count, weight));
                        }
                    } else {
                        avarageRSSIS.add(new AvarageRSSI(count, weight));
                        weight = currentRSSI;
                        count = 1;
                    }
                }
                previousRSSI = currentRSSI;

                sum += currentRSSI;
            }
            double runningAverage = sum/(endIndex-startIndex+1);

            double c3 = 0;
            if (!avarageRSSIS.isEmpty()) {
                Collections.sort(avarageRSSIS);
                double t=0;
                int a = 0;
                for (AvarageRSSI avarageRSSI :
                    avarageRSSIS) {
                    if (avarageRSSI.size > 2) {
                        t += avarageRSSI.sum;
                        a += avarageRSSI.size;
                    }
//                    LogManager.w("C3TEST", "avarageRSSI size: %d sum: %s",
//                        avarageRSSI.size, avarageRSSI.sum);
                }
                if (a > 0) {
                    c3 = t/a;
                }
                if (c3 == 0) {
//                    c3 = 1.0*avarageRSSIS.get(0).sum / avarageRSSIS.get(0).size;
                    c3 = runningAverage;
                }
            } else {
                c3 = runningAverage;
            }
//            c1rssi = runningAverage;
//            LogManager.e("C3TEST", "Result on %s measurements: %s c3: %s",
//                size, runningAverage, c3);
//            LogManager.e("C3TEST", "===============================");
            return c3;
        } catch (Exception e) {

        }

        return 0;
    }

    private synchronized void refreshMeasurements() {
        ArrayList<Measurement> newMeasurements = new ArrayList<Measurement>();
        Iterator<Measurement> iterator = mMeasurements.iterator();
        while (iterator.hasNext()) {
            Measurement measurement = iterator.next();
            if (SystemClock.elapsedRealtime() - measurement.timestamp < sampleExpirationMilliseconds ) {
                newMeasurements.add(measurement);
            }
        }
        mMeasurements = newMeasurements;
        Collections.sort(mMeasurements);
    }

    private class Measurement implements Comparable<Measurement> {
        Integer rssi;
        long timestamp;
        @Override
        public int compareTo(Measurement arg0) {
            return rssi.compareTo(arg0.rssi);
        }
    }

    private class AvarageRSSI implements Comparable<AvarageRSSI> {
        Integer size;
        int sum;

        public AvarageRSSI(int size, int sum) {
            this.size = size;
            this.sum = sum;
        }
        @Override
        public int compareTo(AvarageRSSI arg0) {
            return size.equals(arg0.size) ? 0 : (size - arg0.size > 0 ? -1 : 1);
        }
    }

    public static void setSampleExpirationMilliseconds(long newSampleExpirationMilliseconds) {
        sampleExpirationMilliseconds = newSampleExpirationMilliseconds;
    }

    @RestrictTo(Scope.TESTS)
    static long getSampleExpirationMilliseconds() {
        return sampleExpirationMilliseconds;
    }
}
