package com.zing.zalo.devicetrackingsdk;

import android.app.Activity;
import android.app.Application;
import android.content.Context;

import com.zing.zalo.devicetrackingsdk.DeviceTracking.GetInfoListener;
import com.zing.zalo.devicetrackingsdk.abstracts.IDeviceTracking;
import com.zing.zalo.devicetrackingsdk.abstracts.IEventTracker;
import com.zing.zalo.devicetrackingsdk.model.EventStorage;
import com.zing.zalo.devicetrackingsdk.model.PreloadInfo;
import com.zing.zalo.zalosdk.core.SettingsManager;
import com.zing.zalo.zalosdk.core.exception.InitializedException;
import com.zing.zalo.zalosdk.core.helper.AppInfo;
import com.zing.zalo.zalosdk.core.helper.DeviceHelper;
import com.zing.zalo.zalosdk.core.helper.Utils;
import com.zing.zalo.zalosdk.core.log.Log;
import com.zing.zalo.zalosdk.service.AppLifeCycleMonitorService;
import com.zing.zalo.zalosdk.service.CatchExceptionService;
import com.zing.zalo.zalosdk.service.Foreground;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ZingAnalyticsManager implements IDeviceTracking {
  private static ZingAnalyticsManager manager = new ZingAnalyticsManager();
  private boolean isInitialized;

  private BaseAppInfoStorage mStorage = null;
  private EventStorage eventStorage;
  private IEventTracker eventTracker;
  private Application app;

  private boolean zaAutoActivityTracking = true;
  private List<PendingEvent> pendingEvents;
  private boolean pendingDispatch;

  public static ZingAnalyticsManager getInstance() {
    return manager;
  }

  private ZingAnalyticsManager() {
    pendingEvents = new ArrayList<>();
  }

  public ZingAnalyticsManager init(Application app) {
    String appId = AppInfo.getAppId(app);
    if (appId == null) appId = "";
    return init(app, appId);
  }

  public ZingAnalyticsManager init(Application app, String appId) {
    synchronized (this) {
      if (isInitialized) return this;

      this.app = app;

      final Context context = app.getApplicationContext();
      mStorage = new BaseAppInfoStorage(context);
      eventStorage = new EventStorage(context);

      zaAutoActivityTracking = AppInfo.isAutoTrackingOpenApp(app.getApplicationContext());
      if (zaAutoActivityTracking) {
        AppLifeCycleMonitorService.Instance.start(app);
      }

      eventTracker = new EventTracker(context, mStorage, eventStorage, DeviceTracking.getInstance(), appId);
      eventTracker.setMaxEventsStored(EventTracker.tempMaxEventStored);
      eventTracker.setDispatchEventsInterval(EventTracker.tempDipatchEventsInterval);
      eventTracker.setStoreEventsInterval(EventTracker.tempStoreEventsInterval);

      for (PendingEvent event : pendingEvents) {
        if (event.params == null) {
          event.params = new HashMap<>();
        }
        fillPreloadParams(event.params);
        eventTracker.addEvent(event.action, event.params);
      }

      if (pendingDispatch) {
        eventTracker.dispatchEvents();
      }

      isInitialized = true;
    }

    return this;
  }

  public void finalize() {
    manager = null;
  }

  private void checkInitialize() {
    if (!isInitialized)
      throw new InitializedException("Missing call ZingAnalyticsManager.getInstance().init() in Application.onCreate()");
  }

  @Override
  public String getDeviceId() {
    return DeviceTracking.getInstance().getDeviceId();
  }

  @Override
  public void getDeviceId(GetInfoListener getDeviceIdListener) {
    DeviceTracking.getInstance().getDeviceId(getDeviceIdListener);
  }

  @Override
  public void sendOpenView(String viewName) {
    long startTime = System.currentTimeMillis();
    Map<String, String> params = new HashMap<String, String>();
    params.put("name", viewName);
    params.put("ts", String.valueOf(startTime));
    addEvent("start_screen", params);
  }

  @Override
  public void addException(String name, String message, String reason, String stacktrace) {
    Map<String, String> params = new HashMap<String, String>();
    params.put("name", name);
    params.put("message", message);
    params.put("reason", reason);
    params.put("stack_trace", stacktrace);
    params.put("app_exception", "true");
    addEvent("crash_log", params);
  }

  @Override
  public void addEvent(String action, String category, String label, long value) {
    Map<String, String> params = new HashMap<String, String>();
    params.put("category", category);
    params.put("label", label);
    params.put("value", "" + value);
    addEvent(action, params);
  }


  @Override
  public void onResume(Activity activity) {
    if (!com.zing.zalo.zalosdk.core.helper.Utils.belowSandwich()) return;
    Foreground.get(app).onActivityResumed(activity);
  }

  @Override
  public void onPause(Activity activity) {
    if (!com.zing.zalo.zalosdk.core.helper.Utils.belowSandwich()) return;
    Foreground.get(app).onActivityPaused(activity);
  }

  @Override
  public void addEvent(String action, Map<String, String> params) {
    synchronized (this) {
      if (!isInitialized) {
        pendingEvents.add(new PendingEvent(action, params));
      } else {
        if (params == null) {
          params = new HashMap<String, String>();
        }
        fillPreloadParams(params);
        eventTracker.addEvent(action, params);
      }
    }
  }

  private void fillPreloadParams(Map<String, String> params) {
    try {
      Map<String, String> _params = new HashMap<>();
      PreloadInfo preloadInfo = DeviceHelper.getPreloadInfo(app.getApplicationContext());

      if (preloadInfo != null) {
        _params.put("preloadDefault", AppInfo.getPreloadChannel(app.getApplicationContext()));
        _params.put("preload", preloadInfo.preload);
        if (!preloadInfo.isPreloaded()) {
          _params.put("preloadFailed", preloadInfo.error);
        }
      }

      String listDeviceId = Utils.loadListDeviceIDWakeUp(app.getApplicationContext());
      _params.put("wakeupInfo", listDeviceId);
      params.putAll(_params);
    } catch (Exception ex) {
      ex.printStackTrace();
    }

  }

  @Override
  public void setMaxEventsStored(int num) {
    if (isInitialized)
      eventTracker.setMaxEventsStored(num);
    else
      EventTracker.tempMaxEventStored = num;
  }

  @Override
  public void setDispatchEventsInterval(long ms) {
    if (isInitialized)
      eventTracker.setDispatchEventsInterval(ms);
    else
      EventTracker.tempDipatchEventsInterval = ms;
  }

  @Override
  public void setStoreEventsInterval(long ms) {
    if (isInitialized)
      eventTracker.setStoreEventsInterval(ms);
    else
      EventTracker.tempStoreEventsInterval = ms;
  }

  @Override
  public void dispatchEvents() {
    if (isInitialized) {
      eventTracker.dispatchEvents();
    } else {
      pendingDispatch = true;
    }
  }

  @Override
  public void storeEvents() {
    checkInitialize();
    eventTracker.storeEvents();
  }

  public BaseAppInfoStorage getStorage() {
    checkInitialize();
    return mStorage;
  }

  public boolean getAutoActivityTracking() {
    return this.zaAutoActivityTracking;
  }

  public boolean isPreload(Context context) {
    PreloadInfo preloadInfo = DeviceHelper.getPreloadInfo(context);
    return preloadInfo != null &&
          preloadInfo.isPreloaded();
  }

  public void isPreload(final Context context, final DeviceHelper.PreloadReadListener preloadReadListener) {
    DeviceHelper.getPreloadInfo(context, preloadReadListener);
  }

  public IEventTracker getEventTracker() {
    return eventTracker;
  }

  private class PendingEvent {
    String action;
    Map<String, String> params;

    PendingEvent(String action, Map<String, String> params) {
      this.action = action;
      this.params = params;
    }
  }
}
