package com.zing.zalo.zalosdk.service;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;

import com.zing.zalo.zalosdk.core.helper.Utils;
import com.zing.zalo.zalosdk.core.log.Log;

import java.lang.ref.WeakReference;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;


/**
 * Usage:
 * 
 * 1. Get the Foreground Singleton, passing a Context or Application object
 * unless you are sure that the Singleton has definitely already been
 * initialised elsewhere.
 * 
 * 2.a) Perform a direct, synchronous check: Foreground.isForeground() /
 * .isBackground()
 * 
 * or
 * 
 * 2.b) Register to be notified (useful in Service or other non-UI components):
 * 
 * Foreground.Listener myListener = new Foreground.Listener(){ public void
 * onBecameForeground(){ // ... whatever you want to do } public void
 * onBecameBackground(){ // ... whatever you want to do } }
 * 
 * public void onCreate(){ super.onCreate();
 * Foreground.get(this).addListener(listener); }
 * 
 * public void onDestroy(){ super.onCreate();
 * Foreground.get(this).removeListener(listener); }
 */
@SuppressLint("NewApi")
public class Foreground{

	public interface AppCycleListener {

		public void onBecameForeground(WeakReference<Activity> activity);

		public void onBecameBackground(WeakReference<Activity> activity);

		public void onActivityResume(WeakReference<Activity> activity);
		
	}
	
	public static final long CHECK_DELAY = 500;
	public static final String TAG = Foreground.class.getSimpleName();

	private static Foreground instance;

	private boolean foreground = false, paused = true;
	private Handler handler = new Handler(Looper.getMainLooper());
	private List<AppCycleListener> listeners = new CopyOnWriteArrayList<AppCycleListener>();
	private Runnable check;

	private static CallBackLifeCycle backLifeCycle;
	
	protected Foreground(){
		if (!Utils.belowSandwich()){
			backLifeCycle = new CallBackLifeCycle();
		}
	}
	
	/**
	 * Its not strictly necessary to use this method - usually invoking get with
	 * a Context gives us a path to retrieve the Application and initialise, but
	 * sometimes (e.g. in test harness) the ApplicationContext is != the
	 * Application, and the docs make no guarantees.
	 * 
	 * @param application
	 * @return an initialised Foreground instance
	 */
	public static Foreground init(Application application) {
		if (instance == null) {
			instance = new Foreground();
			if (!Utils.belowSandwich()){
				application.registerActivityLifecycleCallbacks(backLifeCycle);
			}
		}
		return instance;
	}

	public static Foreground get(Application application) {
		if (instance == null) {
			init(application);
		}
		return instance;
	}
	
	public static Foreground get(Context ctx) {
		if (instance == null) {
			Context appCtx = ctx.getApplicationContext();
			if (appCtx instanceof Application) {
				init((Application) appCtx);
			}
			throw new IllegalStateException(
					"Foreground is not initialised and "
							+ "cannot obtain the Application object");
		}
		return instance;
	}

	public static Foreground get() {
		if (instance == null) {
			throw new IllegalStateException(
					"Foreground is not initialised - invoke "
							+ "at least once with parameterised init/get");
		}
		return instance;
	}

	public boolean isForeground() {
		return foreground;
	}

	public boolean isBackground() {
		return !foreground;
	}

	public void addListener(AppCycleListener listener) {
		if(listeners.contains(listener)) return;
		listeners.add(listener);
	}

	public void removeListener(AppCycleListener listener) {
		listeners.remove(listener);
	}

	public void onActivityResumed(Activity activity) {
		paused = false;
		boolean wasBackground = !foreground;
		foreground = true;

		if (check != null)
			handler.removeCallbacks(check);
		
		WeakReference<Activity> act = new WeakReference<Activity>(activity);
		
		if (wasBackground) {
			for (AppCycleListener l : listeners) {
				try {
					l.onBecameForeground(act);
				} catch (Exception exc) {
				}
			}
		}
		
		for (AppCycleListener l : listeners) {
			try {
				l.onActivityResume(act);
			} catch (Exception exc) {
			}
		}
	}
	
	public void onActivityPaused(final Activity activity) {
		paused = true;

		if (check != null)
			handler.removeCallbacks(check);
		final WeakReference<Activity> act = new WeakReference<Activity>(activity);
		handler.postDelayed(check = new Runnable() {
			@Override
			public void run() {
				if (foreground && paused) {
					foreground = false;
					for (AppCycleListener l : listeners) {
						try {
							l.onBecameBackground(act);
						} catch (Exception exc) {
						}
					}
				} else {
				}
			}
		}, CHECK_DELAY);
	}
	
	
	
	class CallBackLifeCycle implements Application.ActivityLifecycleCallbacks{
		
		public CallBackLifeCycle(){
			
		}
		
		@Override
		public void onActivityResumed(Activity activity) {
			instance.onActivityResumed(activity);
		}

		@Override
		public void onActivityPaused(final Activity activity) {
			instance.onActivityPaused(activity);
		}

		@Override
		public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
		}

		@Override
		public void onActivityStarted(Activity activity) {
		}

		@Override
		public void onActivityStopped(Activity activity) {
		}

		@Override
		public void onActivitySaveInstanceState(Activity activity, Bundle outState) {
		}

		@Override
		public void onActivityDestroyed(Activity activity) {
		}
	}
}
