package com.zing.zalo.devicetrackingsdk;

import android.content.Context;
import android.os.Handler;
import android.os.Handler.Callback;
import android.os.HandlerThread;
import android.os.Message;

import com.zing.zalo.devicetrackingsdk.abstracts.IEventTracker;
import com.zing.zalo.devicetrackingsdk.abstracts.IEventTrackerDelegate;
import com.zing.zalo.devicetrackingsdk.event.Event;
import com.zing.zalo.devicetrackingsdk.model.EventStorage;
import com.zing.zalo.zalosdk.Constant;
import com.zing.zalo.zalosdk.core.helper.AppInfo;
import com.zing.zalo.zalosdk.core.helper.Utils;
import com.zing.zalo.zalosdk.core.http.HttpClientFactory;
import com.zing.zalo.zalosdk.core.http.HttpClientRequest;
import com.zing.zalo.zalosdk.core.http.HttpClientRequest.Type;
import com.zing.zalo.zalosdk.core.log.Log;
import com.zing.zalo.zalosdk.core.servicemap.ServiceMapManager;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

import static com.zing.zalo.zalosdk.Constant.API_EVENT_TRACKING;
import static com.zing.zalo.zalosdk.core.servicemap.ServiceMapManager.KEY_URL_CENTRALIZED;

public class EventTracker implements IEventTracker, Callback {
  private static final int ACT_DISPATCH_EVENTS = 0x5000;
  private static final int ACT_PUSH_EVENTS = 0x5001;
  private static final int ACT_STORE_EVENTS = 0x5002;
  private static final int ACT_LOAD_EVENTS = 0x5004;

  private DeviceTracking deviceTracker;
  private BaseAppInfoStorage storage;
  private EventStorage eventStorage;

  public static int tempMaxEventStored = ConstantZingAnalytics.DEFAULT_MAX_EVENTS_STORED;
  public static long tempDipatchEventsInterval = ConstantZingAnalytics.DEFAULT_DISPATCH_EVENTS_INTERVAL;
  public static long tempStoreEventsInterval = ConstantZingAnalytics.DEFAULT_STORE_EVENTS_INTERVAL;

  private int maxEventStored = ConstantZingAnalytics.DEFAULT_MAX_EVENTS_STORED;
  private long dipatchEventsInterval = ConstantZingAnalytics.DEFAULT_DISPATCH_EVENTS_INTERVAL;
  private long storeEventsInterval = ConstantZingAnalytics.DEFAULT_STORE_EVENTS_INTERVAL;
  private long maxEventDispatch = ConstantZingAnalytics.DEFAULT_DISPATCH_MAX_COUNT_EVENT;

  private HandlerThread thread;
  private Handler handler;
  private Timer dispatchTimer;
  private Timer storeTimer;
  private boolean isInitialized;

  private HttpClientFactory mHttpClientFactory = new HttpClientFactory();
  Context context;
  String appId;

  private IEventTrackerDelegate delegate;

  public EventTracker(Context context, BaseAppInfoStorage storage, EventStorage _eventStorage, DeviceTracking deviceTracker, String _appId) {

    thread = new HandlerThread("zdt-event-tracker", HandlerThread.MIN_PRIORITY);
    thread.start();
    handler = new Handler(thread.getLooper(), this);

    this.context = context;
    this.storage = storage;
    this.eventStorage = _eventStorage;
    this.deviceTracker = deviceTracker;
    appId = _appId;
    //load events
    Message msg = new Message();
    msg.what = ACT_LOAD_EVENTS;
    handler.sendMessage(msg);

    //schedule timer
    if (dipatchEventsInterval > 0) {
      scheduleDispatchTimer();
    }

    if (storeEventsInterval > 0) {
      scheduleStoreTimer();
    }

    Log.v("start zdt-event-tracker thread");
  }

  public void addEvent(String action, Map<String, String> params) {
   // android.util.Log.d("addEvent", action + " " + params.toString());
    long timestamp = System.currentTimeMillis();
    Event e = createEvent(action, null, timestamp, params);
    Message msg = new Message();
    msg.what = ACT_PUSH_EVENTS;
    msg.obj = e;
    handler.sendMessage(msg);
  }

  public void setMaxEventsStored(int num) {
    this.maxEventStored = num;
  }

  public void setDispatchEventsInterval(long ms) {
    if (ms == dipatchEventsInterval) return;

    if (ms <= 0 && dispatchTimer != null) {
      cancelDispatchTimer();
      return;
    }

    if (ms < ConstantZingAnalytics.MIN_DISPATCH_EVENTS_INTERVAL) {
      ms = ConstantZingAnalytics.MIN_DISPATCH_EVENTS_INTERVAL;
    }

    dipatchEventsInterval = ms;
    scheduleDispatchTimer();
  }

  public void setMaxDispatchEventCount(int count) {
    if (count < 0) return;
    this.maxEventDispatch = count;
  }

  private void cancelDispatchTimer() {
    if (dispatchTimer != null) {
      Log.v("cancel dispatch timer");
      dispatchTimer.cancel();
      dispatchTimer = null;
    }
  }

  private void scheduleDispatchTimer() {
    Log.v("schedule dispatch timer");
    cancelDispatchTimer();
    //schedule dispatch timer
    try {
      dispatchTimer = new Timer();
      dispatchTimer.scheduleAtFixedRate(new TimerTask() {

        @Override
        public void run() {
          dispatchEvents();
        }
      }, dipatchEventsInterval, dipatchEventsInterval);
    } catch (Exception ex) {
      ex.printStackTrace();
    }
	}

  public void setStoreEventsInterval(long ms) {
    if (ms == storeEventsInterval) return;
    synchronized (this) {
      if (ms <= 0) {
        cancelStoreTimer();
        return;
      }

      if (ms < ConstantZingAnalytics.MIN_STORE_EVENTS_INTERVAL) {
        ms = ConstantZingAnalytics.MIN_STORE_EVENTS_INTERVAL;
      }

      storeEventsInterval = ms;
      scheduleStoreTimer();
    }
  }

  private synchronized void cancelStoreTimer() {
    if (storeTimer != null) {
      Log.v("cancel store events timer");
      storeTimer.cancel();
      storeTimer = null;
    }
  }

  private void scheduleStoreTimer() {
    Log.v("schedule store events timer");
    cancelStoreTimer();

    try {
      storeTimer = new Timer();
      storeTimer.scheduleAtFixedRate(new TimerTask() {

        @Override
        public void run() {
          storeEvents();
        }
      }, storeEventsInterval, storeEventsInterval);
    } catch (Exception ex) {
      ex.printStackTrace();
    }
	}

  public void dispatchEvents() {
    try {
      /*synchronized (this) */
      {
        Message msg = new Message();
        msg.what = ACT_DISPATCH_EVENTS;
        handler.sendMessage(msg);
      }
    } catch (Exception ex) {
      ex.printStackTrace();
    }
  }

  public void storeEvents() {
    try {
      synchronized (this) {
        Message msg = new Message();
        msg.what = ACT_STORE_EVENTS;
        handler.sendMessage(msg);
      }
    } catch (Exception ex) {
      ex.printStackTrace();
    }
  }

  @Override
  public void setEventTrackerDelegate(IEventTrackerDelegate delegate) {
    this.delegate = delegate;
  }

  private Event createEvent(String action, String name, long timestamp, Map<String, String> params) {
    if (params == null) {
      params = new HashMap<>();
    }

    if (name != null && !params.containsKey("name")) {
      params.put("name", name);
    }

    return new Event(action, timestamp, com.zing.zalo.zalosdk.core.helper.Utils.mapToJSONObject(params));
  }

  @Override
  public boolean handleMessage(Message msg) {
    Log.d("EventTracker - handleMessage()", "EventTracker");
    switch (msg.what) {
      case ACT_DISPATCH_EVENTS:

        deviceTracker.getDeviceId(new DeviceTracking.GetInfoListener() {
          @Override
          public void onGetDeviceIdComplete(String deviceId) {
            doDispatchEvent(deviceId);
          }
        });
        break;
      case ACT_STORE_EVENTS:
        doStoreEvents();
        break;
      case ACT_PUSH_EVENTS:
        pushEvent((Event) msg.obj);
        break;
      case ACT_LOAD_EVENTS:
        loadEvents();
        break;
      default:
        return false;
    }
    return true;
  }

  public void doDispatchEvent(String deviceId) {
    try {
      if (eventStorage != null && eventStorage.getEvents().size() == 0 || !isInitialized){
        Log.d("EVENT_TRACKER: " + "No events to dispatch");
        return;
      }
      //send by block?
      JSONArray appData = new JSONArray();
      JSONObject eventData = prepareEventData();
      String zdId = deviceId;
      String sdkId = deviceTracker.getSDKId();
      String privateKey = deviceTracker.getPrivateKey();
      if (zdId == null) zdId = "";
      if (sdkId == null) sdkId = "";

      String an = AppInfo.getAppName(context);
      String av = AppInfo.getVersionName(context);
      String appId = this.appId;
      String viewer = this.storage.getViewer();
      if (viewer == null) viewer = "";

      Date date = new Date();
      String ts = "" + date.getTime();
      String strEventData = eventData.toString();
      String strAppData = appData.toString();
      String strSocialAcc = "[]";
      String packageName = context.getPackageName();
      String[] params = {"pl", "appId", "viewer",
            "data", "apps", "ts", "zdId", "an", "av", "et", "gzip", "socialAcc", "packageName"};
      String[] values = {"android", appId, viewer,
            strEventData, strAppData, ts, zdId, an, av, "0", "0", strSocialAcc, packageName};

      String url = ServiceMapManager.getInstance().urlFor(KEY_URL_CENTRALIZED, API_EVENT_TRACKING);
      url += "?packageName=" + context.getPackageName();

      HttpClientRequest request = mHttpClientFactory.newRequest(Type.POST, url);
      String sig = Utils.getSignature(params, values, Constant.TRK_SECRECT_KEY);
      request.addParams("pl", "android");
      request.addParams("appId", appId);
      request.addParams("viewer", viewer);
      request.addParams("zdId", zdId);
      request.addParams("data", strEventData);
      request.addParams("apps", strAppData);
      request.addParams("ts", ts);
      request.addParams("sig", sig);
      request.addParams("an", an);
      request.addParams("av", av);
      request.addParams("gzip", "0");
      request.addParams("et", "0");
      request.addParams("socialAcc", strSocialAcc);
      request.addParams("packageName", context.getPackageName());
      JSONObject jsonObject = request.getJSON();

      if (request.liveResponseCode >= 400 && request.liveResponseCode <= 599) {
        eventStorage.clearEvents();
      } else if (jsonObject != null) {
        //Log.d(jsonObject.toString());
        int error_code = jsonObject.getInt("error");
        if (error_code == 0) {
          eventStorage.clearEvents();
        }
      }
    } catch (Exception e) {
      e.printStackTrace();
      eventStorage.clearEvents();
    }
  }

  private void doStoreEvents() {
    if (!isInitialized) return;
    eventStorage.saveEvents();
  }

  private void pushEvent(Event e) {
    if (!isInitialized) return;
    if (delegate != null) {
      delegate.onPushEvent(e);
    }

    eventStorage.addEvent(e);
    limitEventsSize();

    List<Event> events = eventStorage.getEvents();
    if (events != null && events.size() >= maxEventDispatch) {
      new Thread(new Runnable() {

        @Override
        public void run() {
          dispatchEvents();
        }
      }).start();
    }
  }

  private void loadEvents() {
    eventStorage.loadEvents();
    isInitialized = true;
  }

  private JSONObject prepareEventData() {
    JSONObject data = new JSONObject();
    JSONObject deviceInfoData = deviceTracker.prepareTrackingData();

    JSONArray jsonEvents = new JSONArray();
    JSONObject jsonEvent, extras;
    try {
      for (Event e : eventStorage.getEvents()) {
        if (delegate != null) {
          delegate.onDispatchEvent(e);
        }

        jsonEvent = new JSONObject();
        extras = e.getParams();
        if (extras.has("name")) {
          jsonEvent.put("name", extras.get("name"));
        }
        jsonEvent.put("extras", extras);
        jsonEvent.put("act", e.getAction());
        jsonEvent.put("ts", e.getTimestamp());

        jsonEvents.put(jsonEvent);
      }
      data.put("evt", jsonEvents);
      data.put("dat", deviceInfoData);
    } catch (Exception ex) {
      ex.printStackTrace();
    }
    return data;
  }

  private void limitEventsSize() {
    if (!isInitialized) return;
    List<Event> events = eventStorage.getEvents();
    if (events.size() > maxEventStored) {
      Log.v(Constant.LOG_TAG, "exceed max number of events %d > %d", events.size(), maxEventStored);
      Event[] removeEvents = events.subList(0, events.size() - maxEventStored).toArray(new Event[]{});
      Event e;
      for (int i = 0; i < removeEvents.length; i++) {
        e = removeEvents[i];
        eventStorage.removeEvent(e);
      }
    }
  }
}
