package com.zing.zalo.zalosdk.oauth;


import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.Intent;

import com.zing.zalo.devicetrackingsdk.DeviceTracking;
import com.zing.zalo.zalosdk.Constant;
import com.zing.zalo.zalosdk.core.exception.InitializedException;
import com.zing.zalo.zalosdk.core.log.Log;
import com.zing.zalo.zalosdk.core.type.LangType;
import com.zing.zalo.zalosdk.payment.direct.Utils;

import org.json.JSONObject;

/**
 * Main interface of the SDK
 */
@SuppressWarnings("deprecation")
public class ZaloSDK  /*implements IDeviceTracking*/{
		
	private boolean bIsInitialized = false;
	private Authenticator mAuthenticator;
	private LocalizedString mLocalizedString;
	private OauthStorage mStorage;
	public static ZaloSDK Instance = new ZaloSDK();
	private ZaloAPIService zaloAPIService;
	/**
	 * Private ctor
	 */
	ZaloSDK() { 
		mLocalizedString = new LocalizedString();
		
	} 

	/**
	 * Initialize the SDK
	 * @param app Application
	 */
	protected synchronized void initialize(final Application app) {
		if (bIsInitialized) return;
		final Context context = app.getApplicationContext();
		mStorage = new OauthStorage(context);
		mAuthenticator = new Authenticator(context, mStorage, mLocalizedString);
		bIsInitialized = true;
		zaloAPIService = new ZaloAPIService(mStorage);
	}

	public Authenticator getAuthenticator() {
		return mAuthenticator;
	}

	public LocalizedString getLocalizedString() {
		return mLocalizedString;
	}
	
	
	/**
	 * Supply your own LocalizedString object if you want to customize the messages used by the SDK
	 * @param localizedString
	 */
	public void setLocalizedString(LocalizedString localizedString) {
		mLocalizedString = localizedString;
	}
	
	//================================================================================
    // Oauth methods
    //================================================================================
	
	/**
	 * Authenticate with Zalo App
	 * @param activity The login activity
	 * @param listener OAuthCompleteListener listener to receive authenticate event
	 */
//	public void authenticate(Activity activity, OAuthCompleteListener listener) {
//		authenticate(activity, LoginVia.APP, listener);
//	}
	
	/**
	 * Authenticate by using Zalo account
	 * @param activity Activity the login activity
	 * @param loginVia not support, SDK will login with Zalo app only.
	 * @param listener OAuthCompleteListener listener to receive authenticate event
	 */
	//@Override
//	public void authenticate(final Activity activity, final LoginVia loginVia, OAuthCompleteListener listener) {
//		checkInitialize();
//		mAuthenticator.authenticate(activity, loginVia, false, listener);
//	}

	public void authenticateZaloWithAuthenType(
			final Activity activity, final LoginVia loginVia, String codeChallenge, JSONObject extInfo,
			OAuthCompleteListener listener) {
		checkInitialize();
		mAuthenticator.authenticate(activity, loginVia, codeChallenge, false, extInfo, listener);
	}

	public void authenticateZaloWithAuthenType(
			final Activity activity, final LoginVia loginVia, String codeChallenge, OAuthCompleteListener listener) {
		authenticateZaloWithAuthenType(activity, loginVia, codeChallenge, null, listener);
	}


	/**
	 * Authenticate Zalo for zalo plugin
	 * @param activity Activity the login activity
	 * @param listener OAuthCompleteListener listener to receive authenticate event
	 */
	protected void authenticateZaloPlugin(final Activity activity, String codeChallenge, OAuthCompleteListener listener) {
		checkInitialize();
		mAuthenticator.authenticate(activity, LoginVia.APP_OR_WEB, codeChallenge, true, listener);
	}

	public void getAccessTokenByOAuthCode(Context context, String oauthCode, String codeVerifier, ZaloOpenAPICallback callback) {
		checkInitialize();
		zaloAPIService.getAccessTokenByOAuthCode(context, oauthCode, codeVerifier, callback);
	}
	public void getAccessTokenByRefreshToken(Context context, String refresh_token, ZaloOpenAPICallback callback) {
		checkInitialize();
		zaloAPIService.getAccessTokenByRefreshToken(context, refresh_token, callback);
	}

	public void setOauthCompletedListener(OAuthCompleteListener listener) {
		checkInitialize();
		mAuthenticator.setOAuthCompleteListener(listener);
	}

	/**
	 * Clear authenticate info in cache
	 * @throws InitializedException This exception would be thrown if the SDK hasn't initialized yet.
	 */
	public void unauthenticate() {
		checkInitialize();
		mAuthenticator.unauthenticate();
	}
	
	/**
	 * Check if users have already authenticated. 
	 * @param callback Callback will be called after verify with server. If passed null, no server verification will be made.
	 * @return True if oauth code cached, otherwise false
	 */
	public boolean isAuthenticate(String refreshToken, ValidateCallback callback) {
		checkInitialize();
		return mAuthenticator.isAuthenticate(refreshToken, callback);
	}

	public boolean isGuestSessionValid(String guestSession, ValidateCallback callback) {
		checkInitialize();
		return mAuthenticator.isGuestSessionValid(guestSession, callback);
	}
	
	/**
	 * Check if an oauth code is valid
	 * @param callback Callback will be called after verify with server. If passed null, no server verification will be made.
	 * @return True if oauth code cached, otherwise false
	 */
//	protected boolean isAuthenticate(String oauthCode, ValidateOAuthCodeCallback callback) {
//		checkInitialize();
//		return mAuthenticator.isAuthenticate(oauthCode, callback);
//	}

	/**
     * Provides an implementation for {@link Activity#
     * onActivityResult} that handles the intent data returned from Zalo or Facebook app
     *
     * @param activity The Activity that is forwarding the onActivityResult call.
     * @param requestCode     The requestCode parameter from the forwarded call. 
     * @param resultCode      An int containing the resultCode parameter from the forwarded
     *                        call.
     * @param data            The Intent passed as the data parameter from the forwarded
     *                        call.
     * @return A boolean indicating whether the requestCode matched a pending request
     */
    public final boolean onActivityResult(Activity activity, int requestCode, int resultCode, Intent data) {
    	return mAuthenticator.onActivityResult(activity, requestCode, resultCode, data);
    }

//	/**
//	 * Request a permission in Zalo app
//	 * @param activity The activity call this method
//	 * @param permission The permission want to be granted
//	 */
//	public void requestPermissions(Activity activity, Enum<Permissions> permission, OAuthCompleteListener listener) {
//		checkInitialize();
//		mAuthenticator.requestPermissions(activity, permission, listener);
//	}

//	/**
//	 * Get the oauth code stored in cache
//	 * @return Oauth code
//	 * @throws InitializedException This exception would be thrown if SDK hasn't initialized yet.
//	 */
	public String getGuestSession() {
		checkInitialize();
		return mStorage.getLatestLoginChannel().equals(LoginChannel.GUEST.toString()) ?
				mStorage.getOAuthCode() : "";
	}

	public void clearGuestSession() {
		checkInitialize();
		if (mStorage.getLatestLoginChannel().equals(LoginChannel.GUEST.toString())) {
			mStorage.clearGuestSession();
		}
	}

	/**
	 * Get the lastest login channel
	 * @return channel
	 */
	public String getLatestLoginChannel() {
		//checkInitialize();
		return mStorage.getLatestLoginChannel();
	}

	public boolean latestLoginChannelIsGuest() {
		String channel = mStorage.getLatestLoginChannel();
		return LoginChannel.GUEST.toString().equals(channel);
	}

	public String getGrantType(boolean useOauthCode) {
		boolean isGuest = latestLoginChannelIsGuest();
		return useOauthCode ?
				(isGuest ? "guest_oauth_code" : "authorization_code") :
				(isGuest ? "guest_refresh_token" : "refresh_token");
	}
	/**
	 * Get the current app id
	 * @return App id
	 */
	public long getAppID() {
		try {
			return Long.parseLong(ZaloSDKApplication.appID);
		} catch (Exception ex) {
			return 0;
		}
	}
	
//	/**
//	 * Get the current user's Zalo id
//	 * @return Zalo id
//	 * @throws InitializedException This exception would be thrown if SDK hasn't initialized yet.
//	 */
//	public long getZaloId()  {
//		checkInitialize();
//		return mStorage.getZaloId();
//	}
	
	/**
	 * Get the display name of current Zalo user
	 * @return display name
	 * @throws InitializedException This exception would be thrown if SDK hasn't initialized yet. 
	 */
	public String getZaloDisplayname() {
		checkInitialize();
		return mStorage.getZaloDisplayName();
	}

	/**
	 * Get the current user's social id
	 * @return social id
	 * @throws InitializedException This exception would be thrown if SDK hasn't initialized yet. 
	 */
	public String getSocialId()  {
		checkInitialize();
		return mStorage.getSocialId();
	}

	/**
	 * Get Zalo user's profile
	 * https://developers.zalo.me/docs/api/open-api/tai-lieu/thong-tin-nguoi-dung-post-28
	 * @param ctx The context call this method
	 * @param callback
	 * @param fields : id, birthday, gender, picture, name ex: {"id", "birthday", "gender", "picture", "name"}
	 */
	public void getProfile(Context ctx, String access_token,ZaloOpenAPICallback callback, String[] fields) {
		checkInitialize();
		zaloAPIService.getProfile(ctx, access_token, callback, fields);
	}

	/**
	 * Lấy danh sách tất cả bạn bè của người dùng đã sử dụng ứng dụng
	 * https://developers.zalo.me/docs/api/open-api/tai-lieu/danh-sach-ban-be-post-34
	 * @param ctx The context call this method
     * @param  position position
     * @param  count count
	 * @param callback
	 * @param fields : Hỗ trợ các field: id, name, picture, gender
	 */
	public void getFriendListUsedApp(Context ctx, String access_token, int position, int count, ZaloOpenAPICallback callback, String[] fields) {
		checkInitialize();
		zaloAPIService.getFriendListUsedApp(ctx, access_token, position, count, callback, fields);
	}

	/**
	 * Lấy danh sách bạn bè chưa sử dụng ứng dụng và có thể nhắn tin mời sử dụng ứng dụng
	 * https://developers.zalo.me/docs/api/open-api/tai-lieu/danh-sach-ban-be-post-34
	 * @param ctx The context call this method
     * @param  position position
     * @param  count count
     * @param callback
	 * @param fields : Hỗ trợ các field: id, name, picture, gender
	 */
	public void getFriendListInvitable(Context ctx, String access_token, int position, int count, ZaloOpenAPICallback callback,
												String[] fields){
		checkInitialize();
		zaloAPIService.getFriendListInvitable(ctx, access_token, position, count, callback, fields);
	}

	/**
	 * https://developers.zalo.me/docs/api/open-api/tai-lieu/moi-su-dung-ung-dung-post-41
	 * @param ctx Context
	 * @param friendId ex: {"friend-id1", "friend-id2", "friend-id3"}
	 * @param message String
	 * @param callback ZaloOpenAPICallback
	 */
	public void inviteFriendUseApp(Context ctx, String access_token, String[] friendId, String message, ZaloOpenAPICallback callback){
		checkInitialize();
		zaloAPIService.inviteFriendUseApp(ctx, access_token, friendId, message, callback);
	}

	//================================================================================
    // Utility methods
    //================================================================================
	
	public void checkInitialize() {
		if(!bIsInitialized) throw new InitializedException("Missing call declare com.zing.zalo.zalosdk.oauth.ZaloSDKApplication in Application or call wrap init");
	}

	public String getVersion() {
		return Constant.VERSION;
	}

	public void setLogLevel(int logLevel) {
		Log.setLogLevel(logLevel);
	}

	public String getDeviceId() {
		return DeviceTracking.getInstance().getDeviceId();
	}

	public void getDeviceId(DeviceTracking.GetInfoListener getDeviceIdListener) {
		 DeviceTracking.getInstance().getDeviceId(getDeviceIdListener);
	}

	public String getGuestDeviceId() {
		checkInitialize();
		return mStorage.getGuestDeviceId();
	}

//    public int getIsGuestCertificated() {
//        checkInitialize();
//        return mStorage.getIsGuestCertificated();
//    }

	public String getSDKId() {
		return DeviceTracking.getInstance().getSDKId();
	}

	public String getPrivateKey() {
		return DeviceTracking.getInstance().getPrivateKey();
	}


	public DeviceTracking getBaseAppInfo(){
		return DeviceTracking.getInstance();
	}

	public OauthStorage getOauthStorage(){
		return mStorage;
	}

	/**
	 * Set language for ZaloSDK
	 * language: LangType
	 */
	public void setLanguageSDK(Context context, LangType language){
		Utils.setLanguage(context, language);
	}

	public void registerZalo(Activity activity, OAuthCompleteListener listener) {
		checkInitialize();
		mAuthenticator.registerZalo(activity, listener);
	}

	/**
	 * call before call ZaloSDKApplication.wrap(Application app)
	 * @param appID
	 */
	public void setApplicationID(long appID){
		ZaloSDKApplication.appID = String.valueOf(appID);
	}

	/**
	 * Check login status of zalo app
	 * Status:
	 *   + 1: logged in
	 *   + 0: not logged in
	 *   + <0: error, ie. app zalo not installed
	 * @param callback callback
	 */
	public void getZaloLoginStatus(final GetZaloLoginStatus callback) {
		checkInitialize();
		mAuthenticator.getZaloLoginStatus(callback);
	}
}
