package com.zing.zalo.zalosdk.core.servicemap;

import android.content.Context;
import android.os.AsyncTask;
import android.text.TextUtils;
import com.zing.zalo.zalosdk.Constant;
import com.zing.zalo.zalosdk.core.http.HttpClientFactory;
import com.zing.zalo.zalosdk.core.http.HttpClientRequest;
import com.zing.zalo.zalosdk.core.log.Log;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;


public class ServiceMapManager {
    public static final String KEY_URL_OAUTH = "oauth_http_s";
    public static final String KEY_URL_GRAPH = "graph_http_s";
    public static final String KEY_URL_CENTRALIZED = "centralized_http_s";

    private static final String URL_OAUTH = "https://oauth.zaloapp.com";
    private static final String URL_GRAPH = "https://graph.zaloapp.com";
    private static final String URL_CENTRALIZED = "https://ztevents.zaloapp.com";

    private static final String URL_DEV_OAUTH = "https://dev-oauth.zaloapp.com";
    private static final String URL_DEV_GRAPH = URL_GRAPH;
    private static final String URL_DEV_CENTRALIZED = URL_CENTRALIZED;

    private static final long ONE_DATE_DURATION = 1000 * 60 * 60 * 24;
    private long expireTime = -1;
    private HttpClientFactory httpClientFactory = new HttpClientFactory();

    public static final String[] SERVICE_MAP_URLS = new String[]{
            "https://mp3.zing.vn/zdl/service_map_all.bin",
            "https://zaloapp.com/zdl/service_map_all.bin",
            "https://news.zing.vn/zdl/service_map_all.bin",
            "https://n.zing.vn/zdl/service_map_all.bin",
            "https://srv.mp3.zing.vn/zdl/service_map_all.bin"
    };

    private Map<String, String> urls;
    private static ServiceMapManager instance;
    private ServiceMapStorage storage;

    public static synchronized ServiceMapManager getInstance() {
        if (instance == null) {
            instance = new ServiceMapManager();
        }

        return instance;
    }

    public static synchronized void destroyInstance() {
        if(instance != null) {
            instance = null;
        }
    }

    private ServiceMapManager() {
        urls = new HashMap<>();
        if (Constant.IS_DEV) {
            urls.put(KEY_URL_OAUTH, URL_DEV_OAUTH);
            urls.put(KEY_URL_GRAPH, URL_DEV_GRAPH);
            urls.put(KEY_URL_CENTRALIZED, URL_DEV_CENTRALIZED);
        } else {
            urls.put(KEY_URL_OAUTH, URL_OAUTH);
            urls.put(KEY_URL_GRAPH, URL_GRAPH);
            urls.put(KEY_URL_CENTRALIZED, URL_CENTRALIZED);
        }
    }

    public void load(final Context context) {

        updateMapUrlsFromPreference(context);
        if (isExpiredTime(context) && !Constant.IS_DEV) {
            final DownloadServiceMapFilesAsyncTask serviceTask = new DownloadServiceMapFilesAsyncTask(httpClientFactory, new ServiceMapListener() {
                @Override
                public void receiveJSONObject(JSONObject dataObject) {
                    if(dataObject == null) {
                        Log.v("Service map not found!");
                        return;
                    }

                    try {
                        String urlOauth = dataObject.getJSONArray(KEY_URL_OAUTH).getString(0);
                        String urlGraph = dataObject.getJSONArray(KEY_URL_GRAPH).getString(0);
                        String urlCentralized = dataObject.getJSONArray(KEY_URL_CENTRALIZED).getString(0);
                        Log.v("Got service map: ");
                        Log.v("oath: " + urlOauth);
                        Log.v("graph: " + urlGraph);
                        Log.v("centralized: " + urlCentralized);
                        updateMapUrls(urlOauth, urlGraph, urlCentralized, context);
                    } catch (Exception e) {
                        Log.v(e.getMessage());
                    }
                }
            });
            serviceTask.execute();
        }
    }

    public String urlFor(String key, String path) {
        String url = urls.get(key);
        if (url == null) {
            Log.v("Url for" + key + "not found");
            return path;
        }

        if (!url.endsWith("/") && !path.startsWith("/")) {
            return url + "/" + path;
        }

        return url + path;
    }

    private void updateMapUrls(final String urlOauth, final String urlGraph, final String urlCentralized, Context context) {
        ServiceMapStorage storage = getStorage(context);

        storage.setKeyUrlCentralized(urlCentralized);
        storage.setKeyUrlGraph(urlGraph);
        storage.setKeyUrlOauth(urlOauth);

        urls.put(KEY_URL_OAUTH, urlOauth);
        urls.put(KEY_URL_GRAPH, urlGraph);
        urls.put(KEY_URL_CENTRALIZED, urlCentralized);

        long currentTimeMillis = System.currentTimeMillis();
        expireTime = currentTimeMillis + ONE_DATE_DURATION;

        storage.setExpireTime(expireTime);
    }

    private void updateMapUrlsFromPreference(Context context) {
        ServiceMapStorage storage = getStorage(context);

        String urlOauth = storage.getKeyUrlOauth();
        String urlGraph = storage.getKeyUrlGraph();
        String urlCentralized = storage.getKeyUrlCentralized();

        if (!TextUtils.isEmpty(urlOauth) && !TextUtils.isEmpty(urlGraph) && !TextUtils.isEmpty(urlCentralized)) {
            urls.put(KEY_URL_OAUTH, urlOauth);
            urls.put(KEY_URL_GRAPH, urlGraph);
            urls.put(KEY_URL_CENTRALIZED, urlCentralized);
        }
    }

    public boolean isExpiredTime(Context context) {
        long currentTimeMillis = System.currentTimeMillis();
        ServiceMapStorage storage = getStorage(context);

        if (expireTime == -1) {
            expireTime = (storage.getExpireTime() != 0) ? storage.getExpireTime() : currentTimeMillis;
        }

        return currentTimeMillis >= expireTime;
    }

    public static class DownloadServiceMapFilesAsyncTask extends AsyncTask<Integer, Void, JSONObject> {

        private final HttpClientFactory httpClientFactory;
        private ServiceMapListener listener;

        DownloadServiceMapFilesAsyncTask(HttpClientFactory httpClientFactory, ServiceMapListener listener) {
            this.httpClientFactory = httpClientFactory;
            this.listener = listener;
        }

        @Override
        protected JSONObject doInBackground(Integer... integers) {
            for (String serviceMapUrl : SERVICE_MAP_URLS) {
                try {
                    HttpClientRequest request = httpClientFactory.newRequest(HttpClientRequest.Type.GET, serviceMapUrl);
                    String str = request.getText();
                    String decryptString = ServiceMapTools.decryptString(str);
                    return new JSONObject(decryptString);
                } catch (Exception e) {
                    Log.e("DownloadServiceMapFilesAsyncTask",e.getMessage());
                }
            }
            return null;
        }

        @Override
        protected void onPostExecute(JSONObject result) {
            listener.receiveJSONObject(result);
        }
    }

    public void setHttpClientFactory(HttpClientFactory httpClientFactory) {
        this.httpClientFactory = httpClientFactory;
    }

    public ServiceMapStorage getStorage(Context context) {
        if(storage == null) {
            storage = new ServiceMapStorage(context);
        }
        return storage;
    }

    public void setStorage(ServiceMapStorage storage) {
        this.storage = storage;
    }
}



