package com.zing.zalo.zalosdk.core.http;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;

import org.json.JSONException;
import org.json.JSONObject;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.text.TextUtils;

import com.zing.zalo.devicetrackingsdk.DeviceTracking;
import com.zing.zalo.zalosdk.Constant;
import com.zing.zalo.zalosdk.ZaloOAuthResultCode;
import com.zing.zalo.zalosdk.core.log.Log;
import com.zing.zalo.zalosdk.core.type.FromSourceType;

public class HttpClientRequest {
	
	private static final String LINE_END = "\r\n";
	private static final String TWO_HYPHENS = "--";
	private static final String BOUNDARY = "*****";
	
	public enum Type {
		GET, POST, POST_MULIIPART
	}

	public int liveResponseCode;

	private Type mType;
	private String mUrl;
	
	private Map<String, String> mParams;
	private Map<String, String> mHeader;
	private String fileName;
	private String fileKey;
	private byte[] data;
		
	public HttpClientRequest(Type type, String url) {
		this.mType = type;
		this.mUrl = url;
		this.mHeader = new HashMap<>();
		this.mParams = new HashMap<>();
		addHeader("SDK-Source","ZaloSDK-"+ Constant.VERSION);
		addHeader("extInfo", DeviceTracking.getInstance().prepareCommonHeaderData().toString());
	}
	public void addHeader(String name, String value) {
		mHeader.put(name, value);
	}

	public void addParams(String name, String value) {
		mParams.put(name, value);
	}
	
	public void setMultipartFile(String fileName, String fileKey, byte[] data) {
		this.fileName = fileName;
		this.fileKey = fileKey;
		this.data = data;
	}

	public void setBody(byte[] data) {
		this.data = data;
	}

	private String getQueryString(){
		StringBuilder sb = new StringBuilder("");
		if (mParams != null){
			for(String key : mParams.keySet()) {
				try {
					sb.append(URLEncoder.encode(key, "UTF-8"));
			        sb.append("=");
			        sb.append(URLEncoder.encode(mParams.get(key), "UTF-8"));
			        sb.append("&");
				} catch (Exception e) {
					Log.w(e.toString());
				}
			}
		}
		return sb.toString();
	}

	public String getJSONString() throws JSONException {
		JSONObject jsonObject = new JSONObject();
		if (mParams != null) {
			for(String key : mParams.keySet()) {
				try {
					jsonObject.put(key, mParams.get(key));
				} catch (Exception e) {
					Log.w(e.toString());
				}
			}

		}
		return jsonObject.toString(4);

	}

	public String getParamsString() {
		StringBuilder sb = new StringBuilder("{");
		if (mParams != null) {
			for(String key : mParams.keySet()) {
				try {
					sb.append(key);
			        sb.append("=");
			        sb.append(mParams.get(key));
				} catch (Exception e) {
					Log.w(e.toString());
				}
			}
			
		}
		sb.append("}");
		return sb.toString();
	}
	

	public InputStream getResponse() throws Exception{

			if (mType == Type.POST){//POST
				Log.v("connect server : " + mUrl + " : " + getQueryString());
				HttpURLConnection connection = (HttpURLConnection) new URL(mUrl).openConnection();
				connection.setDoOutput(true); // Triggers POST.

				connection.setRequestProperty("Accept-Charset", "UTF-8");
				for (String key : mHeader.keySet()) {
					connection.setRequestProperty(key, mHeader.get(key));
				}

				OutputStream output;
				if(this.data != null) {
					output = connection.getOutputStream();
					output.write(this.data);
				} else {
					connection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded;charset=" + "UTF-8");
					output = connection.getOutputStream();
					output.write(getQueryString().getBytes("UTF-8"));
				}

				output.flush();
				output.close();

				liveResponseCode = connection.getResponseCode();
				Log.v("connect server POST " + liveResponseCode );
				return connection.getInputStream();
			}
			else if(mType == Type.POST_MULIIPART) {
				// Open a HTTP connection to the URL
				Log.v("connect server : " + mUrl + " : " + getQueryString());
				HttpURLConnection connection = (HttpURLConnection) new URL(mUrl).openConnection();
				connection.setDoInput(true); 
				connection.setDoOutput(true);
				connection.setUseCaches(false); 
				connection.setConnectTimeout(2 * 60 * 1000);
				connection.setRequestMethod("POST");
				connection.setRequestProperty("Connection", "Keep-Alive");
				connection.setRequestProperty("ENCTYPE", "multipart/form-data");

				connection.setRequestProperty("Content-Type", "multipart/form-data;boundary=" + BOUNDARY);
				if(fileName != null) {
					connection.setRequestProperty("uploaded_file", fileName);
				}

				for (String key : mHeader.keySet()) {
					connection.setRequestProperty(key, mHeader.get(key));
				}

				DataOutputStream dos = new DataOutputStream(connection.getOutputStream());			
				for (String key : mParams.keySet()) {
					dos.writeBytes(TWO_HYPHENS + BOUNDARY + LINE_END);
					dos.writeBytes("Content-Disposition: form-data; name=\"" + key + "\"" + LINE_END);
					dos.writeBytes("Content-Type: text/plain; charset=UTF-8" + LINE_END);
					dos.writeBytes(LINE_END);
					dos.write(mParams.get(key).getBytes("UTF-8"));
					dos.writeBytes(LINE_END);
				}
				
				if(data != null && fileName != null && fileKey != null) {
					dos.writeBytes(TWO_HYPHENS + BOUNDARY + LINE_END);
					dos.writeBytes("Content-Disposition: form-data; name=" + fileKey + ";filename=" + fileName + "" + LINE_END);
					dos.writeBytes(LINE_END);
					dos.write(data);
				}

				dos.writeBytes(LINE_END);
				dos.writeBytes(TWO_HYPHENS + BOUNDARY + TWO_HYPHENS + LINE_END);
				dos.flush();
				dos.close();

				int responseCode = connection.getResponseCode();
				Log.v("connect server POST_MULTIPART " + responseCode);
				liveResponseCode = connection.getResponseCode();
				if(responseCode >= 300) {
					String newLine = System.getProperty("line.separator");
					BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getErrorStream()));
					StringBuilder result = new StringBuilder();
					String line; boolean flag = false;
					while ((line = reader.readLine()) != null) {
					    result.append(flag? newLine: "").append(line);
					    flag = true;
					}
					Log.v(result.toString());
					return null;
				}
				
				return connection.getInputStream();
			}
			else {//GET
				StringBuilder sb = new StringBuilder(mUrl);
				if(sb.toString().contains("?"))
					sb.append("&platform=android");
				else 
					sb.append("?platform=android");
				for(String key : mParams.keySet()) {
					try {
						sb.append("&");
						sb.append(URLEncoder.encode(key, "UTF-8"));
				        sb.append("=");
				        if (mParams.get(key) != null) {
							sb.append(URLEncoder.encode(mParams.get(key), "UTF-8").replace("+", "%20"));
						} else {
							sb.append(URLEncoder.encode("", "UTF-8"));
						}
					} catch (Exception e) {
						Log.w(e.toString());
					}
				}
				Log.v("connect server GET : " +sb.toString());
				HttpURLConnection connection = (HttpURLConnection) new URL(sb.toString()).openConnection();
				for (String key : mHeader.keySet()) {
					connection.setRequestProperty(key, mHeader.get(key));
				}
				liveResponseCode = connection.getResponseCode();
				return connection.getInputStream();
			}

	}
	
	
	
	public String getText() {
		try {
			BufferedReader reader = new BufferedReader(new InputStreamReader(getResponse()));
			String line;
			StringBuilder result = new StringBuilder();
			while ((line = reader.readLine()) != null) {
				if(result.length() > 0) {
					result.append("\n");
				}

				result.append(line);
			}
			reader.close();
//			Log.v("connect server RESPONSE getText() from: "+ mUrl+ "=" +result.toString());
			return result.toString();
		} catch (Exception ex) {
			Log.w(ex.toString());
			return null;
		}
	}

	public JSONObject getJSON() {
		String s = "";
		try {
			BufferedReader reader = new BufferedReader(new InputStreamReader(getResponse()));
			String line;
			StringBuilder result = new StringBuilder();
			while ((line = reader.readLine()) != null) {
				result.append(line);
				result.append("\n");
			}
			reader.close();
			s = result.toString();
//			Log.v("connect server RESPONSE getJSON() from: "+ mUrl+ "=" +s);
			JSONObject jsonObject = new JSONObject(s);
			int extCode = jsonObject.optInt("error");
			int error = ZaloOAuthResultCode.findById(extCode);
			jsonObject.put("error", error);
			jsonObject.put("extCode", extCode);
			return jsonObject;
		} catch (Exception ex) {

			try {
				JSONObject jsonObject = new JSONObject();
				jsonObject.put("errorMsg", s + "  " +  ex.toString());
				return jsonObject;
			} catch (JSONException e) {
				e.printStackTrace();
			}

			Log.w(ex.toString());
			return null;
		}
	}

	public Bitmap getImage() {
		try {
			InputStream inputStream = getResponse();
			Bitmap image =null;
			if (inputStream != null) {
				 image = BitmapFactory.decodeStream(inputStream);
				 inputStream.close();
			}
			return image;
		} catch (Exception ex) {
//			Log.e(getClass().getName(), ex.getMessage(), ex);
			return null;
		}
	}
	
	public static byte[] getByteArray(String url) {
		if (TextUtils.isEmpty(url)) return null;
		
		URL toDownload;
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		try {
			toDownload = new URL(url);
			byte[] chunk = new byte[4096];
	        int bytesRead;
	        InputStream stream = toDownload.openStream();

	        while ((bytesRead = stream.read(chunk)) > 0) {
	            outputStream.write(chunk, 0, bytesRead);
	        }
		} catch (MalformedURLException e) {
			Log.w(e.toString());
			return null;
		}catch (IOException e) {
			Log.w(e.toString());
	        return null;
	    }
	    
	    return outputStream.toByteArray();
	}

	public int getResponseCode() {		
		try {
			HttpURLConnection c = (HttpURLConnection) new URL(mUrl).openConnection();
			return c.getResponseCode();
		} catch (Exception ex) {
			Log.w(ex.toString());
			return 0;
		}
	}
	
}
