package com.zing.zalo.devicetrackingsdk;

import android.content.Context;
import android.os.Handler;
import android.os.Handler.Callback;
import android.os.HandlerThread;
import android.os.Message;
import android.text.TextUtils;

import com.zing.zalo.zalosdk.Constant;
import com.zing.zalo.zalosdk.core.log.Log;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

public class DeviceTracking implements Callback {
	static final int ACT_LOAD = 0x4000;
	static final int ACT_GEN = 0x4001;
	static final int ACT_GEN_SDK_ID = 0x4002;

	private BaseAppInfoStorage storage;
	private Context context;

	private HandlerThread thread;
    private Handler handler;
	public boolean isInitialized;

	private boolean isGeneratingDeviceId;
	private boolean isGeneratingSDKId;
    private List<GetInfoListener> infoListeners;
    private String appID;

    Api api;

	private static DeviceTracking baseAppInfoInstance = new DeviceTracking();


	public static DeviceTracking getInstance(){
	    return baseAppInfoInstance;
    }

	public JSONObject prepareTrackingData(){
		return api.prepareTrackingData(context, storage.getDeviceId(), System.currentTimeMillis());
	}

	public JSONObject prepareCommonHeaderData(){
		return api.prepareCommonHeaderData(context);
	}

	public JSONObject prepareExtraInfo(int source){
		return api.prepareExtraInfo(source);
	}

	public JSONObject prepareDeviceIdData(){
		return api.prepareDeviceIdData(context);
	}

    DeviceTracking(){
        infoListeners = new ArrayList<>();
        api = new Api();
	}

	DeviceTracking(Api api, Handler handler) {
		infoListeners = new ArrayList<>();
		this.api =  api;
		this.handler = handler;
	}


	public synchronized void initDeviceTracking(Context context, BaseAppInfoStorage storage, String appID) {
		if (isInitialized){
			return;
		}

		this.storage = storage;
		this.context = context;
		this.appID = appID;

		api.appId = appID;
		api.isPreInstalled = storage.ispreInstalled();

		isInitialized = true;
		sendMessage(ACT_LOAD);
	}


	public interface GetInfoListener {
		void onGetDeviceIdComplete(String deviceId);
	}
	
	public String getSDKId() {
		if (!isInitialized){
			return null;
		}
		
		return storage.getSDKId();
	}
	
	public String getPrivateKey() {
		if (!isInitialized){
			return null;
		}

		String privateKey = storage.getPrivateKey();
		if (!TextUtils.isEmpty(privateKey)) {
			return privateKey;
		}

		sendMessage(ACT_GEN_SDK_ID);

		return null;
	}

	private void callbackAndResetListener(String deviceId){
        ArrayList<GetInfoListener> listeners;
        synchronized (this) {
            if (infoListeners == null || infoListeners.size() == 0) return;

            listeners = new ArrayList<>(infoListeners);
            infoListeners.clear();
        }

        for(GetInfoListener listener : listeners) {
            listener.onGetDeviceIdComplete(deviceId);
        }
	}

	public String getDeviceId() {
		if (!isInitialized){
			return null;
		}
		if (!isDeviceIdValid()) {
			sendMessage(ACT_GEN);
		}

		return storage.getDeviceId();
	}

	public long getDeviceIdExpireTime() {
        if (!isInitialized){
            return 0;
        }

        return storage.getDeviceIdExpiredTime();
    }


	public void getDeviceId(GetInfoListener getDeviceIdListener) {
		if(hasDeviceId()) {
			if(getDeviceIdListener != null) {
				getDeviceIdListener.onGetDeviceIdComplete(storage.getDeviceId());
			}

			if (!isDeviceIdValid()) {
				sendMessage(ACT_GEN);
			}
		} else {
			if (getDeviceIdListener != null) {
				synchronized (this) {
					infoListeners.add(getDeviceIdListener);
				}
			}

			if(isInitialized) {
				sendMessage(ACT_GEN);
			}
		}
	}

	private boolean hasDeviceId() {
		if(!isInitialized || storage == null) return false;
		String deviceId = storage.getDeviceId();

		return deviceId != null && !deviceId.equals("");
	}

	private boolean isDeviceIdValid() {
		if(!isInitialized || storage == null) return false;
		String deviceId = storage.getDeviceId();
		long expiredTime = storage.getDeviceIdExpiredTime();

        return expiredTime > System.currentTimeMillis() && deviceId != null && !deviceId.equals("");
    }

	/**
	 * Send message the handler, should call this method on man looper
	 */
	private synchronized void sendMessage(int what) {
		if(handler == null) {
			thread = new HandlerThread("zdt-device-tracker", HandlerThread.NORM_PRIORITY);
			thread.start();
			handler = new Handler(thread.getLooper(), this);
		}

		Message message = handler.obtainMessage(what);
		if (message != null){
			handler.sendMessage(message);
		}
	}
	
	@Override
	public boolean handleMessage(Message msg) {
		switch (msg.what) {
		case ACT_LOAD:
			loadDeviceId();
            storage.loadSDKId();
			generateSDKIdIfNeeded();

			break;
		case ACT_GEN:
			generateDeviceIdIfNeeded();
			break;
		case ACT_GEN_SDK_ID:
			generateSDKIdIfNeeded();
			break;
		default:
			return false;
		}
		return true;
	}

	private void loadDeviceId() {
		storage.loadDeviceId();

		if (isDeviceIdValid()) {
			callbackAndResetListener(storage.getDeviceId());
		} else {
			generateDeviceIdIfNeeded();
		}

	}

	private void generateSDKIdIfNeeded() {
        String sdkId = storage.getSDKId();
        String privateKey = storage.getPrivateKey();

        if (!TextUtils.isEmpty(sdkId) && !TextUtils.isEmpty(privateKey)) {
            return;
        }
		synchronized (this) {
			if(isGeneratingSDKId) return;
			isGeneratingSDKId = true;
		}

		try {
			api.viewer = storage.getViewer();
			Api.SDKIdResponse response = api.getSdkId(context);
			storage.setSDKId(response.sdkId, response.privateKey);
		}catch (Exception e) {
			Log.e(e);
		}

		synchronized (this) {
			isGeneratingSDKId = false;
		}
	}
	
	private void generateDeviceIdIfNeeded() {
		synchronized (this) {
			if(isGeneratingDeviceId || isDeviceIdValid()) return;
			isGeneratingDeviceId = true;
		}


		api.viewer = storage.getViewer();
		api.sdkId = storage.getSDKId();

		try {
			Api.DeviceIdResponse resp = api.getDeviceId(context, storage.getDeviceId());
			storage.setDeviceId(resp.deviceId, resp.expireTime);
			callbackAndResetListener(resp.deviceId == null ? "" : resp.deviceId);

		} catch (Exception ex) {
//			Log.e(ex);
		}

		synchronized (this) {
			isGeneratingDeviceId = false;
		}
	}

	public String getVersion() {
		return Constant.VERSION;
	}
}
