package me.zwave.zway;

import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

public final class ZWay {
    public final static int deviceAdded = 0x01;
    public final static int deviceRemoved = 0x02;
    public final static int instanceAdded = 0x04;
    public final static int instanceRemoved = 0x08;
    public final static int commandAdded = 0x10;
    public final static int commandRemoved = 0x20;
    private final long jzway;

    public Controller controller;
    public Map<Integer, Device> devices;

    private Set<DeviceCallback> deviceCallbacks;
    private Set<StatusCallback> statusCallbacks;
    private Set<TerminateCallback> terminateCallbacks;

    static {
        System.loadLibrary("jzway");
    }

    public ZWay(String name, String port, int speed, String config_folder, String translations_folder, String zddx_folder) throws Exception {
        jzway = jni_zwayInit(name, port, speed, config_folder, translations_folder, zddx_folder);

        controller = new Controller();
        devices = new HashMap<>();

        deviceCallbacks = new HashSet<>();
        statusCallbacks = new HashSet<>();
        terminateCallbacks = new HashSet<>();
    }

    @Override
    public void finalize() {
        jni_finalize(jzway);
    }

    public void bind(DeviceCallback func) throws Exception {
        deviceCallbacks.add(func);
    }

    public void unbind(DeviceCallback func) throws Exception {
        deviceCallbacks.remove(func);
    }

    public void bind(StatusCallback func) {
        statusCallbacks.add(func);
    }

    public void unbind(StatusCallback func) {
        statusCallbacks.remove(func);
    }

    public void bind(TerminateCallback func) {
        terminateCallbacks.add(func);
    }

    public void unbind(TerminateCallback func) {
        terminateCallbacks.remove(func);
    }

    public static final Map<String, Integer> commandClassIdByName = new HashMap<>();
    static {
        /* BEGIN AUTOGENERATED CC TEMPLATE
        commandClassIdByName.put("%cc_camel_case_name%", %cc_id%);
        * END AUTOGENERATED CC TEMPLATE */
        /* BEGIN AUTOGENERATED CC CODE */
        commandClassIdByName.put("basic", 0x20);
        commandClassIdByName.put("wakeup", 0x84);
        commandClassIdByName.put("noOperation", 0x0);
        commandClassIdByName.put("battery", 0x80);
        commandClassIdByName.put("manufacturerSpecific", 0x72);
        commandClassIdByName.put("proprietary", 0x88);
        commandClassIdByName.put("configuration", 0x70);
        commandClassIdByName.put("sensorBinary", 0x30);
        commandClassIdByName.put("association", 0x85);
        commandClassIdByName.put("meter", 0x32);
        commandClassIdByName.put("meterPulse", 0x35);
        commandClassIdByName.put("sensorMultilevel", 0x31);
        commandClassIdByName.put("sensorConfiguration", 0x9e);
        commandClassIdByName.put("switchAll", 0x27);
        commandClassIdByName.put("switchBinary", 0x25);
        commandClassIdByName.put("switchMultilevel", 0x26);
        commandClassIdByName.put("multiChannelAssociation", 0x8e);
        commandClassIdByName.put("multiChannel", 0x60);
        commandClassIdByName.put("nodeNaming", 0x77);
        commandClassIdByName.put("thermostatSetPoint", 0x43);
        commandClassIdByName.put("thermostatMode", 0x40);
        commandClassIdByName.put("thermostatFanMode", 0x44);
        commandClassIdByName.put("thermostatFanState", 0x45);
        commandClassIdByName.put("thermostatOperatingState", 0x42);
        commandClassIdByName.put("thermostatSetBack", 0x47);
        commandClassIdByName.put("alarmSensor", 0x9c);
        commandClassIdByName.put("doorLock", 0x62);
        commandClassIdByName.put("doorLockLogging", 0x4c);
        commandClassIdByName.put("userCode", 0x63);
        commandClassIdByName.put("time", 0x8a);
        commandClassIdByName.put("timeParameters", 0x8b);
        commandClassIdByName.put("clock", 0x81);
        commandClassIdByName.put("sceneActivation", 0x2b);
        commandClassIdByName.put("sceneControllerConf", 0x2d);
        commandClassIdByName.put("sceneActuatorConf", 0x2c);
        commandClassIdByName.put("indicator", 0x87);
        commandClassIdByName.put("protection", 0x75);
        commandClassIdByName.put("scheduleEntryLock", 0x4e);
        commandClassIdByName.put("climateControlSchedule", 0x46);
        commandClassIdByName.put("meterTableMonitor", 0x3d);
        commandClassIdByName.put("alarm", 0x71);
        commandClassIdByName.put("powerLevel", 0x73);
        commandClassIdByName.put("zWavePlusInfo", 0x5e);
        commandClassIdByName.put("firmwareUpdate", 0x7a);
        commandClassIdByName.put("associationGroupInformation", 0x59);
        commandClassIdByName.put("switchColor", 0x33);
        commandClassIdByName.put("soundSwitch", 0x79);
        commandClassIdByName.put("barrierOperator", 0x66);
        commandClassIdByName.put("simpleAVControl", 0x94);
        commandClassIdByName.put("security", 0x98);
        commandClassIdByName.put("securityS2", 0x9f);
        commandClassIdByName.put("entryControl", 0x6f);
        commandClassIdByName.put("hail", 0x82);
        commandClassIdByName.put("cRC16", 0x56);
        commandClassIdByName.put("multiCmd", 0x8f);
        commandClassIdByName.put("supervision", 0x6c);
        commandClassIdByName.put("applicationStatus", 0x22);
        commandClassIdByName.put("version", 0x86);
        commandClassIdByName.put("deviceResetLocally", 0x5a);
        commandClassIdByName.put("centralScene", 0x5b);
        /* END AUTOGENERATED CC CODE */
    }
    
    public static final Map<Integer, String> commandClassNameById = new HashMap<>();
    static {
        /* BEGIN AUTOGENERATED CC TEMPLATE
        commandClassNameById.put(%cc_id%, "%cc_camel_case_name%");
        * END AUTOGENERATED CC TEMPLATE */
        /* BEGIN AUTOGENERATED CC CODE */
        commandClassNameById.put(0x20, "basic");
        commandClassNameById.put(0x84, "wakeup");
        commandClassNameById.put(0x0, "noOperation");
        commandClassNameById.put(0x80, "battery");
        commandClassNameById.put(0x72, "manufacturerSpecific");
        commandClassNameById.put(0x88, "proprietary");
        commandClassNameById.put(0x70, "configuration");
        commandClassNameById.put(0x30, "sensorBinary");
        commandClassNameById.put(0x85, "association");
        commandClassNameById.put(0x32, "meter");
        commandClassNameById.put(0x35, "meterPulse");
        commandClassNameById.put(0x31, "sensorMultilevel");
        commandClassNameById.put(0x9e, "sensorConfiguration");
        commandClassNameById.put(0x27, "switchAll");
        commandClassNameById.put(0x25, "switchBinary");
        commandClassNameById.put(0x26, "switchMultilevel");
        commandClassNameById.put(0x8e, "multiChannelAssociation");
        commandClassNameById.put(0x60, "multiChannel");
        commandClassNameById.put(0x77, "nodeNaming");
        commandClassNameById.put(0x43, "thermostatSetPoint");
        commandClassNameById.put(0x40, "thermostatMode");
        commandClassNameById.put(0x44, "thermostatFanMode");
        commandClassNameById.put(0x45, "thermostatFanState");
        commandClassNameById.put(0x42, "thermostatOperatingState");
        commandClassNameById.put(0x47, "thermostatSetBack");
        commandClassNameById.put(0x9c, "alarmSensor");
        commandClassNameById.put(0x62, "doorLock");
        commandClassNameById.put(0x4c, "doorLockLogging");
        commandClassNameById.put(0x63, "userCode");
        commandClassNameById.put(0x8a, "time");
        commandClassNameById.put(0x8b, "timeParameters");
        commandClassNameById.put(0x81, "clock");
        commandClassNameById.put(0x2b, "sceneActivation");
        commandClassNameById.put(0x2d, "sceneControllerConf");
        commandClassNameById.put(0x2c, "sceneActuatorConf");
        commandClassNameById.put(0x87, "indicator");
        commandClassNameById.put(0x75, "protection");
        commandClassNameById.put(0x4e, "scheduleEntryLock");
        commandClassNameById.put(0x46, "climateControlSchedule");
        commandClassNameById.put(0x3d, "meterTableMonitor");
        commandClassNameById.put(0x71, "alarm");
        commandClassNameById.put(0x73, "powerLevel");
        commandClassNameById.put(0x5e, "zWavePlusInfo");
        commandClassNameById.put(0x7a, "firmwareUpdate");
        commandClassNameById.put(0x59, "associationGroupInformation");
        commandClassNameById.put(0x33, "switchColor");
        commandClassNameById.put(0x79, "soundSwitch");
        commandClassNameById.put(0x66, "barrierOperator");
        commandClassNameById.put(0x94, "simpleAVControl");
        commandClassNameById.put(0x98, "security");
        commandClassNameById.put(0x9f, "securityS2");
        commandClassNameById.put(0x6f, "entryControl");
        commandClassNameById.put(0x82, "hail");
        commandClassNameById.put(0x56, "cRC16");
        commandClassNameById.put(0x8f, "multiCmd");
        commandClassNameById.put(0x6c, "supervision");
        commandClassNameById.put(0x22, "applicationStatus");
        commandClassNameById.put(0x86, "version");
        commandClassNameById.put(0x5a, "deviceResetLocally");
        commandClassNameById.put(0x5b, "centralScene");
        /* END AUTOGENERATED CC CODE */
    }
    
    // Controller methods

    public boolean isRunning() { return jni_isRunning(jzway); }
    public boolean isIdle() { return jni_isIdle(jzway); }
    public void stop() { jni_stop(jzway); }
    public void discover() { jni_discover(jzway); }

    /* BEGIN AUTOGENERATED CC TEMPLATE
    public Device.Instance.%cc_capitalized_name% %cc_camel_case_name%(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.%cc_capitalized_name%) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("%cc_camel_case_name%"));
    }
    * END AUTOGENERATED CC TEMPLATE */
    /* BEGIN AUTOGENERATED CC CODE */
    public Device.Instance.Basic basic(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Basic) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("basic"));
    }
    public Device.Instance.Wakeup wakeup(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Wakeup) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("wakeup"));
    }
    public Device.Instance.NoOperation noOperation(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.NoOperation) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("noOperation"));
    }
    public Device.Instance.Battery battery(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Battery) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("battery"));
    }
    public Device.Instance.ManufacturerSpecific manufacturerSpecific(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.ManufacturerSpecific) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("manufacturerSpecific"));
    }
    public Device.Instance.Proprietary proprietary(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Proprietary) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("proprietary"));
    }
    public Device.Instance.Configuration configuration(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Configuration) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("configuration"));
    }
    public Device.Instance.SensorBinary sensorBinary(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.SensorBinary) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("sensorBinary"));
    }
    public Device.Instance.Association association(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Association) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("association"));
    }
    public Device.Instance.Meter meter(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Meter) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("meter"));
    }
    public Device.Instance.MeterPulse meterPulse(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.MeterPulse) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("meterPulse"));
    }
    public Device.Instance.SensorMultilevel sensorMultilevel(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.SensorMultilevel) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("sensorMultilevel"));
    }
    public Device.Instance.SensorConfiguration sensorConfiguration(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.SensorConfiguration) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("sensorConfiguration"));
    }
    public Device.Instance.SwitchAll switchAll(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.SwitchAll) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("switchAll"));
    }
    public Device.Instance.SwitchBinary switchBinary(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.SwitchBinary) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("switchBinary"));
    }
    public Device.Instance.SwitchMultilevel switchMultilevel(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.SwitchMultilevel) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("switchMultilevel"));
    }
    public Device.Instance.MultiChannelAssociation multiChannelAssociation(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.MultiChannelAssociation) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("multiChannelAssociation"));
    }
    public Device.Instance.MultiChannel multiChannel(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.MultiChannel) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("multiChannel"));
    }
    public Device.Instance.NodeNaming nodeNaming(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.NodeNaming) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("nodeNaming"));
    }
    public Device.Instance.ThermostatSetPoint thermostatSetPoint(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.ThermostatSetPoint) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("thermostatSetPoint"));
    }
    public Device.Instance.ThermostatMode thermostatMode(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.ThermostatMode) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("thermostatMode"));
    }
    public Device.Instance.ThermostatFanMode thermostatFanMode(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.ThermostatFanMode) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("thermostatFanMode"));
    }
    public Device.Instance.ThermostatFanState thermostatFanState(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.ThermostatFanState) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("thermostatFanState"));
    }
    public Device.Instance.ThermostatOperatingState thermostatOperatingState(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.ThermostatOperatingState) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("thermostatOperatingState"));
    }
    public Device.Instance.ThermostatSetBack thermostatSetBack(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.ThermostatSetBack) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("thermostatSetBack"));
    }
    public Device.Instance.AlarmSensor alarmSensor(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.AlarmSensor) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("alarmSensor"));
    }
    public Device.Instance.DoorLock doorLock(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.DoorLock) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("doorLock"));
    }
    public Device.Instance.DoorLockLogging doorLockLogging(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.DoorLockLogging) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("doorLockLogging"));
    }
    public Device.Instance.UserCode userCode(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.UserCode) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("userCode"));
    }
    public Device.Instance.Time time(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Time) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("time"));
    }
    public Device.Instance.TimeParameters timeParameters(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.TimeParameters) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("timeParameters"));
    }
    public Device.Instance.Clock clock(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Clock) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("clock"));
    }
    public Device.Instance.SceneActivation sceneActivation(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.SceneActivation) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("sceneActivation"));
    }
    public Device.Instance.SceneControllerConf sceneControllerConf(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.SceneControllerConf) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("sceneControllerConf"));
    }
    public Device.Instance.SceneActuatorConf sceneActuatorConf(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.SceneActuatorConf) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("sceneActuatorConf"));
    }
    public Device.Instance.Indicator indicator(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Indicator) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("indicator"));
    }
    public Device.Instance.Protection protection(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Protection) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("protection"));
    }
    public Device.Instance.ScheduleEntryLock scheduleEntryLock(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.ScheduleEntryLock) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("scheduleEntryLock"));
    }
    public Device.Instance.ClimateControlSchedule climateControlSchedule(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.ClimateControlSchedule) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("climateControlSchedule"));
    }
    public Device.Instance.MeterTableMonitor meterTableMonitor(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.MeterTableMonitor) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("meterTableMonitor"));
    }
    public Device.Instance.Alarm alarm(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Alarm) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("alarm"));
    }
    public Device.Instance.PowerLevel powerLevel(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.PowerLevel) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("powerLevel"));
    }
    public Device.Instance.ZWavePlusInfo zWavePlusInfo(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.ZWavePlusInfo) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("zWavePlusInfo"));
    }
    public Device.Instance.FirmwareUpdate firmwareUpdate(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.FirmwareUpdate) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("firmwareUpdate"));
    }
    public Device.Instance.AssociationGroupInformation associationGroupInformation(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.AssociationGroupInformation) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("associationGroupInformation"));
    }
    public Device.Instance.SwitchColor switchColor(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.SwitchColor) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("switchColor"));
    }
    public Device.Instance.SoundSwitch soundSwitch(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.SoundSwitch) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("soundSwitch"));
    }
    public Device.Instance.BarrierOperator barrierOperator(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.BarrierOperator) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("barrierOperator"));
    }
    public Device.Instance.SimpleAVControl simpleAVControl(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.SimpleAVControl) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("simpleAVControl"));
    }
    public Device.Instance.Security security(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Security) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("security"));
    }
    public Device.Instance.SecurityS2 securityS2(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.SecurityS2) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("securityS2"));
    }
    public Device.Instance.EntryControl entryControl(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.EntryControl) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("entryControl"));
    }
    public Device.Instance.Hail hail(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Hail) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("hail"));
    }
    public Device.Instance.CRC16 cRC16(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.CRC16) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("cRC16"));
    }
    public Device.Instance.MultiCmd multiCmd(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.MultiCmd) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("multiCmd"));
    }
    public Device.Instance.Supervision supervision(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Supervision) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("supervision"));
    }
    public Device.Instance.ApplicationStatus applicationStatus(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.ApplicationStatus) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("applicationStatus"));
    }
    public Device.Instance.Version version(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.Version) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("version"));
    }
    public Device.Instance.DeviceResetLocally deviceResetLocally(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.DeviceResetLocally) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("deviceResetLocally"));
    }
    public Device.Instance.CentralScene centralScene(Integer deviceId, Integer instanceId) {
        return ((ZWay.Device.Instance.CentralScene) devices.get(deviceId).instances.get(instanceId).commandClassesByName.get("centralScene"));
    }
    /* END AUTOGENERATED CC CODE */

    public static Type commandClassClassById(Integer id) {
        switch (id) {
            /* BEGIN AUTOGENERATED CC TEMPLATE
            case %cc_id%:
                return ZWay.Device.Instance.%cc_capitalized_name%.class;
            * END AUTOGENERATED CC TEMPLATE */
            /* BEGIN AUTOGENERATED CC CODE */
            case 0x20:
                return ZWay.Device.Instance.Basic.class;
            case 0x84:
                return ZWay.Device.Instance.Wakeup.class;
            case 0x0:
                return ZWay.Device.Instance.NoOperation.class;
            case 0x80:
                return ZWay.Device.Instance.Battery.class;
            case 0x72:
                return ZWay.Device.Instance.ManufacturerSpecific.class;
            case 0x88:
                return ZWay.Device.Instance.Proprietary.class;
            case 0x70:
                return ZWay.Device.Instance.Configuration.class;
            case 0x30:
                return ZWay.Device.Instance.SensorBinary.class;
            case 0x85:
                return ZWay.Device.Instance.Association.class;
            case 0x32:
                return ZWay.Device.Instance.Meter.class;
            case 0x35:
                return ZWay.Device.Instance.MeterPulse.class;
            case 0x31:
                return ZWay.Device.Instance.SensorMultilevel.class;
            case 0x9e:
                return ZWay.Device.Instance.SensorConfiguration.class;
            case 0x27:
                return ZWay.Device.Instance.SwitchAll.class;
            case 0x25:
                return ZWay.Device.Instance.SwitchBinary.class;
            case 0x26:
                return ZWay.Device.Instance.SwitchMultilevel.class;
            case 0x8e:
                return ZWay.Device.Instance.MultiChannelAssociation.class;
            case 0x60:
                return ZWay.Device.Instance.MultiChannel.class;
            case 0x77:
                return ZWay.Device.Instance.NodeNaming.class;
            case 0x43:
                return ZWay.Device.Instance.ThermostatSetPoint.class;
            case 0x40:
                return ZWay.Device.Instance.ThermostatMode.class;
            case 0x44:
                return ZWay.Device.Instance.ThermostatFanMode.class;
            case 0x45:
                return ZWay.Device.Instance.ThermostatFanState.class;
            case 0x42:
                return ZWay.Device.Instance.ThermostatOperatingState.class;
            case 0x47:
                return ZWay.Device.Instance.ThermostatSetBack.class;
            case 0x9c:
                return ZWay.Device.Instance.AlarmSensor.class;
            case 0x62:
                return ZWay.Device.Instance.DoorLock.class;
            case 0x4c:
                return ZWay.Device.Instance.DoorLockLogging.class;
            case 0x63:
                return ZWay.Device.Instance.UserCode.class;
            case 0x8a:
                return ZWay.Device.Instance.Time.class;
            case 0x8b:
                return ZWay.Device.Instance.TimeParameters.class;
            case 0x81:
                return ZWay.Device.Instance.Clock.class;
            case 0x2b:
                return ZWay.Device.Instance.SceneActivation.class;
            case 0x2d:
                return ZWay.Device.Instance.SceneControllerConf.class;
            case 0x2c:
                return ZWay.Device.Instance.SceneActuatorConf.class;
            case 0x87:
                return ZWay.Device.Instance.Indicator.class;
            case 0x75:
                return ZWay.Device.Instance.Protection.class;
            case 0x4e:
                return ZWay.Device.Instance.ScheduleEntryLock.class;
            case 0x46:
                return ZWay.Device.Instance.ClimateControlSchedule.class;
            case 0x3d:
                return ZWay.Device.Instance.MeterTableMonitor.class;
            case 0x71:
                return ZWay.Device.Instance.Alarm.class;
            case 0x73:
                return ZWay.Device.Instance.PowerLevel.class;
            case 0x5e:
                return ZWay.Device.Instance.ZWavePlusInfo.class;
            case 0x7a:
                return ZWay.Device.Instance.FirmwareUpdate.class;
            case 0x59:
                return ZWay.Device.Instance.AssociationGroupInformation.class;
            case 0x33:
                return ZWay.Device.Instance.SwitchColor.class;
            case 0x79:
                return ZWay.Device.Instance.SoundSwitch.class;
            case 0x66:
                return ZWay.Device.Instance.BarrierOperator.class;
            case 0x94:
                return ZWay.Device.Instance.SimpleAVControl.class;
            case 0x98:
                return ZWay.Device.Instance.Security.class;
            case 0x9f:
                return ZWay.Device.Instance.SecurityS2.class;
            case 0x6f:
                return ZWay.Device.Instance.EntryControl.class;
            case 0x82:
                return ZWay.Device.Instance.Hail.class;
            case 0x56:
                return ZWay.Device.Instance.CRC16.class;
            case 0x8f:
                return ZWay.Device.Instance.MultiCmd.class;
            case 0x6c:
                return ZWay.Device.Instance.Supervision.class;
            case 0x22:
                return ZWay.Device.Instance.ApplicationStatus.class;
            case 0x86:
                return ZWay.Device.Instance.Version.class;
            case 0x5a:
                return ZWay.Device.Instance.DeviceResetLocally.class;
            case 0x5b:
                return ZWay.Device.Instance.CentralScene.class;
            /* END AUTOGENERATED CC CODE */
            default:
                return null;
        }
    }

    public static String commandClassNameById(Integer id) {
        switch (id) {
            /* BEGIN AUTOGENERATED CC TEMPLATE
            case %cc_id%:
                return "%cc_camel_case_name%";
            * END AUTOGENERATED CC TEMPLATE */
            /* BEGIN AUTOGENERATED CC CODE */
            case 0x20:
                return "basic";
            case 0x84:
                return "wakeup";
            case 0x0:
                return "noOperation";
            case 0x80:
                return "battery";
            case 0x72:
                return "manufacturerSpecific";
            case 0x88:
                return "proprietary";
            case 0x70:
                return "configuration";
            case 0x30:
                return "sensorBinary";
            case 0x85:
                return "association";
            case 0x32:
                return "meter";
            case 0x35:
                return "meterPulse";
            case 0x31:
                return "sensorMultilevel";
            case 0x9e:
                return "sensorConfiguration";
            case 0x27:
                return "switchAll";
            case 0x25:
                return "switchBinary";
            case 0x26:
                return "switchMultilevel";
            case 0x8e:
                return "multiChannelAssociation";
            case 0x60:
                return "multiChannel";
            case 0x77:
                return "nodeNaming";
            case 0x43:
                return "thermostatSetPoint";
            case 0x40:
                return "thermostatMode";
            case 0x44:
                return "thermostatFanMode";
            case 0x45:
                return "thermostatFanState";
            case 0x42:
                return "thermostatOperatingState";
            case 0x47:
                return "thermostatSetBack";
            case 0x9c:
                return "alarmSensor";
            case 0x62:
                return "doorLock";
            case 0x4c:
                return "doorLockLogging";
            case 0x63:
                return "userCode";
            case 0x8a:
                return "time";
            case 0x8b:
                return "timeParameters";
            case 0x81:
                return "clock";
            case 0x2b:
                return "sceneActivation";
            case 0x2d:
                return "sceneControllerConf";
            case 0x2c:
                return "sceneActuatorConf";
            case 0x87:
                return "indicator";
            case 0x75:
                return "protection";
            case 0x4e:
                return "scheduleEntryLock";
            case 0x46:
                return "climateControlSchedule";
            case 0x3d:
                return "meterTableMonitor";
            case 0x71:
                return "alarm";
            case 0x73:
                return "powerLevel";
            case 0x5e:
                return "zWavePlusInfo";
            case 0x7a:
                return "firmwareUpdate";
            case 0x59:
                return "associationGroupInformation";
            case 0x33:
                return "switchColor";
            case 0x79:
                return "soundSwitch";
            case 0x66:
                return "barrierOperator";
            case 0x94:
                return "simpleAVControl";
            case 0x98:
                return "security";
            case 0x9f:
                return "securityS2";
            case 0x6f:
                return "entryControl";
            case 0x82:
                return "hail";
            case 0x56:
                return "cRC16";
            case 0x8f:
                return "multiCmd";
            case 0x6c:
                return "supervision";
            case 0x22:
                return "applicationStatus";
            case 0x86:
                return "version";
            case 0x5a:
                return "deviceResetLocally";
            case 0x5b:
                return "centralScene";
            /* END AUTOGENERATED CC CODE */
            default:
                return null;
        }
    }

    // Callback stub
    private void statusCallback(boolean result, Object obj) {
        for (StatusCallback sc : statusCallbacks) {
            sc.statusCallback(result, obj);
        }
    }

    private void deviceCallback(int type, int deviceId, int instanceId, int commandClassId) throws Exception {
        if (type == deviceAdded) {
            Device device = new Device(this, deviceId);
            this.devices.put(deviceId, device);
            device.instances.put(0, device.new Instance(this, device, 0));
        } else if (type == deviceRemoved) {
            devices.remove(deviceId);
        } else if (type == instanceAdded) {
            Device device = devices.get(deviceId);
            device.instances.put(instanceId, device.new Instance(this, device, instanceId));
        } else if (type == instanceRemoved) {
            devices.get(deviceId).instances.remove(instanceId);
        } else if (type == commandAdded) {
            Device device = devices.get(deviceId);
            Device.Instance instance = device.instances.get(instanceId);
            switch (commandClassId) {
                /* BEGIN AUTOGENERATED CC TEMPLATE
                case %cc_id%:
                    Device.Instance.%cc_capitalized_name% command%cc_capitalized_name% = instance.new %cc_capitalized_name%(this, instance);
                    instance.commandClasses.put(commandClassId, command%cc_capitalized_name%);
                    instance.commandClassesByName.put("%cc_camel_case_name%", command%cc_capitalized_name%);
                    break;
                 * END AUTOGENERATED CC TEMPLATE */
                /* BEGIN AUTOGENERATED CC CODE */
                case 0x20:
                    Device.Instance.Basic commandBasic = instance.new Basic(this, instance);
                    instance.commandClasses.put(commandClassId, commandBasic);
                    instance.commandClassesByName.put("basic", commandBasic);
                    break;
                case 0x84:
                    Device.Instance.Wakeup commandWakeup = instance.new Wakeup(this, instance);
                    instance.commandClasses.put(commandClassId, commandWakeup);
                    instance.commandClassesByName.put("wakeup", commandWakeup);
                    break;
                case 0x0:
                    Device.Instance.NoOperation commandNoOperation = instance.new NoOperation(this, instance);
                    instance.commandClasses.put(commandClassId, commandNoOperation);
                    instance.commandClassesByName.put("noOperation", commandNoOperation);
                    break;
                case 0x80:
                    Device.Instance.Battery commandBattery = instance.new Battery(this, instance);
                    instance.commandClasses.put(commandClassId, commandBattery);
                    instance.commandClassesByName.put("battery", commandBattery);
                    break;
                case 0x72:
                    Device.Instance.ManufacturerSpecific commandManufacturerSpecific = instance.new ManufacturerSpecific(this, instance);
                    instance.commandClasses.put(commandClassId, commandManufacturerSpecific);
                    instance.commandClassesByName.put("manufacturerSpecific", commandManufacturerSpecific);
                    break;
                case 0x88:
                    Device.Instance.Proprietary commandProprietary = instance.new Proprietary(this, instance);
                    instance.commandClasses.put(commandClassId, commandProprietary);
                    instance.commandClassesByName.put("proprietary", commandProprietary);
                    break;
                case 0x70:
                    Device.Instance.Configuration commandConfiguration = instance.new Configuration(this, instance);
                    instance.commandClasses.put(commandClassId, commandConfiguration);
                    instance.commandClassesByName.put("configuration", commandConfiguration);
                    break;
                case 0x30:
                    Device.Instance.SensorBinary commandSensorBinary = instance.new SensorBinary(this, instance);
                    instance.commandClasses.put(commandClassId, commandSensorBinary);
                    instance.commandClassesByName.put("sensorBinary", commandSensorBinary);
                    break;
                case 0x85:
                    Device.Instance.Association commandAssociation = instance.new Association(this, instance);
                    instance.commandClasses.put(commandClassId, commandAssociation);
                    instance.commandClassesByName.put("association", commandAssociation);
                    break;
                case 0x32:
                    Device.Instance.Meter commandMeter = instance.new Meter(this, instance);
                    instance.commandClasses.put(commandClassId, commandMeter);
                    instance.commandClassesByName.put("meter", commandMeter);
                    break;
                case 0x35:
                    Device.Instance.MeterPulse commandMeterPulse = instance.new MeterPulse(this, instance);
                    instance.commandClasses.put(commandClassId, commandMeterPulse);
                    instance.commandClassesByName.put("meterPulse", commandMeterPulse);
                    break;
                case 0x31:
                    Device.Instance.SensorMultilevel commandSensorMultilevel = instance.new SensorMultilevel(this, instance);
                    instance.commandClasses.put(commandClassId, commandSensorMultilevel);
                    instance.commandClassesByName.put("sensorMultilevel", commandSensorMultilevel);
                    break;
                case 0x9e:
                    Device.Instance.SensorConfiguration commandSensorConfiguration = instance.new SensorConfiguration(this, instance);
                    instance.commandClasses.put(commandClassId, commandSensorConfiguration);
                    instance.commandClassesByName.put("sensorConfiguration", commandSensorConfiguration);
                    break;
                case 0x27:
                    Device.Instance.SwitchAll commandSwitchAll = instance.new SwitchAll(this, instance);
                    instance.commandClasses.put(commandClassId, commandSwitchAll);
                    instance.commandClassesByName.put("switchAll", commandSwitchAll);
                    break;
                case 0x25:
                    Device.Instance.SwitchBinary commandSwitchBinary = instance.new SwitchBinary(this, instance);
                    instance.commandClasses.put(commandClassId, commandSwitchBinary);
                    instance.commandClassesByName.put("switchBinary", commandSwitchBinary);
                    break;
                case 0x26:
                    Device.Instance.SwitchMultilevel commandSwitchMultilevel = instance.new SwitchMultilevel(this, instance);
                    instance.commandClasses.put(commandClassId, commandSwitchMultilevel);
                    instance.commandClassesByName.put("switchMultilevel", commandSwitchMultilevel);
                    break;
                case 0x8e:
                    Device.Instance.MultiChannelAssociation commandMultiChannelAssociation = instance.new MultiChannelAssociation(this, instance);
                    instance.commandClasses.put(commandClassId, commandMultiChannelAssociation);
                    instance.commandClassesByName.put("multiChannelAssociation", commandMultiChannelAssociation);
                    break;
                case 0x60:
                    Device.Instance.MultiChannel commandMultiChannel = instance.new MultiChannel(this, instance);
                    instance.commandClasses.put(commandClassId, commandMultiChannel);
                    instance.commandClassesByName.put("multiChannel", commandMultiChannel);
                    break;
                case 0x77:
                    Device.Instance.NodeNaming commandNodeNaming = instance.new NodeNaming(this, instance);
                    instance.commandClasses.put(commandClassId, commandNodeNaming);
                    instance.commandClassesByName.put("nodeNaming", commandNodeNaming);
                    break;
                case 0x43:
                    Device.Instance.ThermostatSetPoint commandThermostatSetPoint = instance.new ThermostatSetPoint(this, instance);
                    instance.commandClasses.put(commandClassId, commandThermostatSetPoint);
                    instance.commandClassesByName.put("thermostatSetPoint", commandThermostatSetPoint);
                    break;
                case 0x40:
                    Device.Instance.ThermostatMode commandThermostatMode = instance.new ThermostatMode(this, instance);
                    instance.commandClasses.put(commandClassId, commandThermostatMode);
                    instance.commandClassesByName.put("thermostatMode", commandThermostatMode);
                    break;
                case 0x44:
                    Device.Instance.ThermostatFanMode commandThermostatFanMode = instance.new ThermostatFanMode(this, instance);
                    instance.commandClasses.put(commandClassId, commandThermostatFanMode);
                    instance.commandClassesByName.put("thermostatFanMode", commandThermostatFanMode);
                    break;
                case 0x45:
                    Device.Instance.ThermostatFanState commandThermostatFanState = instance.new ThermostatFanState(this, instance);
                    instance.commandClasses.put(commandClassId, commandThermostatFanState);
                    instance.commandClassesByName.put("thermostatFanState", commandThermostatFanState);
                    break;
                case 0x42:
                    Device.Instance.ThermostatOperatingState commandThermostatOperatingState = instance.new ThermostatOperatingState(this, instance);
                    instance.commandClasses.put(commandClassId, commandThermostatOperatingState);
                    instance.commandClassesByName.put("thermostatOperatingState", commandThermostatOperatingState);
                    break;
                case 0x47:
                    Device.Instance.ThermostatSetBack commandThermostatSetBack = instance.new ThermostatSetBack(this, instance);
                    instance.commandClasses.put(commandClassId, commandThermostatSetBack);
                    instance.commandClassesByName.put("thermostatSetBack", commandThermostatSetBack);
                    break;
                case 0x9c:
                    Device.Instance.AlarmSensor commandAlarmSensor = instance.new AlarmSensor(this, instance);
                    instance.commandClasses.put(commandClassId, commandAlarmSensor);
                    instance.commandClassesByName.put("alarmSensor", commandAlarmSensor);
                    break;
                case 0x62:
                    Device.Instance.DoorLock commandDoorLock = instance.new DoorLock(this, instance);
                    instance.commandClasses.put(commandClassId, commandDoorLock);
                    instance.commandClassesByName.put("doorLock", commandDoorLock);
                    break;
                case 0x4c:
                    Device.Instance.DoorLockLogging commandDoorLockLogging = instance.new DoorLockLogging(this, instance);
                    instance.commandClasses.put(commandClassId, commandDoorLockLogging);
                    instance.commandClassesByName.put("doorLockLogging", commandDoorLockLogging);
                    break;
                case 0x63:
                    Device.Instance.UserCode commandUserCode = instance.new UserCode(this, instance);
                    instance.commandClasses.put(commandClassId, commandUserCode);
                    instance.commandClassesByName.put("userCode", commandUserCode);
                    break;
                case 0x8a:
                    Device.Instance.Time commandTime = instance.new Time(this, instance);
                    instance.commandClasses.put(commandClassId, commandTime);
                    instance.commandClassesByName.put("time", commandTime);
                    break;
                case 0x8b:
                    Device.Instance.TimeParameters commandTimeParameters = instance.new TimeParameters(this, instance);
                    instance.commandClasses.put(commandClassId, commandTimeParameters);
                    instance.commandClassesByName.put("timeParameters", commandTimeParameters);
                    break;
                case 0x81:
                    Device.Instance.Clock commandClock = instance.new Clock(this, instance);
                    instance.commandClasses.put(commandClassId, commandClock);
                    instance.commandClassesByName.put("clock", commandClock);
                    break;
                case 0x2b:
                    Device.Instance.SceneActivation commandSceneActivation = instance.new SceneActivation(this, instance);
                    instance.commandClasses.put(commandClassId, commandSceneActivation);
                    instance.commandClassesByName.put("sceneActivation", commandSceneActivation);
                    break;
                case 0x2d:
                    Device.Instance.SceneControllerConf commandSceneControllerConf = instance.new SceneControllerConf(this, instance);
                    instance.commandClasses.put(commandClassId, commandSceneControllerConf);
                    instance.commandClassesByName.put("sceneControllerConf", commandSceneControllerConf);
                    break;
                case 0x2c:
                    Device.Instance.SceneActuatorConf commandSceneActuatorConf = instance.new SceneActuatorConf(this, instance);
                    instance.commandClasses.put(commandClassId, commandSceneActuatorConf);
                    instance.commandClassesByName.put("sceneActuatorConf", commandSceneActuatorConf);
                    break;
                case 0x87:
                    Device.Instance.Indicator commandIndicator = instance.new Indicator(this, instance);
                    instance.commandClasses.put(commandClassId, commandIndicator);
                    instance.commandClassesByName.put("indicator", commandIndicator);
                    break;
                case 0x75:
                    Device.Instance.Protection commandProtection = instance.new Protection(this, instance);
                    instance.commandClasses.put(commandClassId, commandProtection);
                    instance.commandClassesByName.put("protection", commandProtection);
                    break;
                case 0x4e:
                    Device.Instance.ScheduleEntryLock commandScheduleEntryLock = instance.new ScheduleEntryLock(this, instance);
                    instance.commandClasses.put(commandClassId, commandScheduleEntryLock);
                    instance.commandClassesByName.put("scheduleEntryLock", commandScheduleEntryLock);
                    break;
                case 0x46:
                    Device.Instance.ClimateControlSchedule commandClimateControlSchedule = instance.new ClimateControlSchedule(this, instance);
                    instance.commandClasses.put(commandClassId, commandClimateControlSchedule);
                    instance.commandClassesByName.put("climateControlSchedule", commandClimateControlSchedule);
                    break;
                case 0x3d:
                    Device.Instance.MeterTableMonitor commandMeterTableMonitor = instance.new MeterTableMonitor(this, instance);
                    instance.commandClasses.put(commandClassId, commandMeterTableMonitor);
                    instance.commandClassesByName.put("meterTableMonitor", commandMeterTableMonitor);
                    break;
                case 0x71:
                    Device.Instance.Alarm commandAlarm = instance.new Alarm(this, instance);
                    instance.commandClasses.put(commandClassId, commandAlarm);
                    instance.commandClassesByName.put("alarm", commandAlarm);
                    break;
                case 0x73:
                    Device.Instance.PowerLevel commandPowerLevel = instance.new PowerLevel(this, instance);
                    instance.commandClasses.put(commandClassId, commandPowerLevel);
                    instance.commandClassesByName.put("powerLevel", commandPowerLevel);
                    break;
                case 0x5e:
                    Device.Instance.ZWavePlusInfo commandZWavePlusInfo = instance.new ZWavePlusInfo(this, instance);
                    instance.commandClasses.put(commandClassId, commandZWavePlusInfo);
                    instance.commandClassesByName.put("zWavePlusInfo", commandZWavePlusInfo);
                    break;
                case 0x7a:
                    Device.Instance.FirmwareUpdate commandFirmwareUpdate = instance.new FirmwareUpdate(this, instance);
                    instance.commandClasses.put(commandClassId, commandFirmwareUpdate);
                    instance.commandClassesByName.put("firmwareUpdate", commandFirmwareUpdate);
                    break;
                case 0x59:
                    Device.Instance.AssociationGroupInformation commandAssociationGroupInformation = instance.new AssociationGroupInformation(this, instance);
                    instance.commandClasses.put(commandClassId, commandAssociationGroupInformation);
                    instance.commandClassesByName.put("associationGroupInformation", commandAssociationGroupInformation);
                    break;
                case 0x33:
                    Device.Instance.SwitchColor commandSwitchColor = instance.new SwitchColor(this, instance);
                    instance.commandClasses.put(commandClassId, commandSwitchColor);
                    instance.commandClassesByName.put("switchColor", commandSwitchColor);
                    break;
                case 0x79:
                    Device.Instance.SoundSwitch commandSoundSwitch = instance.new SoundSwitch(this, instance);
                    instance.commandClasses.put(commandClassId, commandSoundSwitch);
                    instance.commandClassesByName.put("soundSwitch", commandSoundSwitch);
                    break;
                case 0x66:
                    Device.Instance.BarrierOperator commandBarrierOperator = instance.new BarrierOperator(this, instance);
                    instance.commandClasses.put(commandClassId, commandBarrierOperator);
                    instance.commandClassesByName.put("barrierOperator", commandBarrierOperator);
                    break;
                case 0x94:
                    Device.Instance.SimpleAVControl commandSimpleAVControl = instance.new SimpleAVControl(this, instance);
                    instance.commandClasses.put(commandClassId, commandSimpleAVControl);
                    instance.commandClassesByName.put("simpleAVControl", commandSimpleAVControl);
                    break;
                case 0x98:
                    Device.Instance.Security commandSecurity = instance.new Security(this, instance);
                    instance.commandClasses.put(commandClassId, commandSecurity);
                    instance.commandClassesByName.put("security", commandSecurity);
                    break;
                case 0x9f:
                    Device.Instance.SecurityS2 commandSecurityS2 = instance.new SecurityS2(this, instance);
                    instance.commandClasses.put(commandClassId, commandSecurityS2);
                    instance.commandClassesByName.put("securityS2", commandSecurityS2);
                    break;
                case 0x6f:
                    Device.Instance.EntryControl commandEntryControl = instance.new EntryControl(this, instance);
                    instance.commandClasses.put(commandClassId, commandEntryControl);
                    instance.commandClassesByName.put("entryControl", commandEntryControl);
                    break;
                case 0x82:
                    Device.Instance.Hail commandHail = instance.new Hail(this, instance);
                    instance.commandClasses.put(commandClassId, commandHail);
                    instance.commandClassesByName.put("hail", commandHail);
                    break;
                case 0x56:
                    Device.Instance.CRC16 commandCRC16 = instance.new CRC16(this, instance);
                    instance.commandClasses.put(commandClassId, commandCRC16);
                    instance.commandClassesByName.put("cRC16", commandCRC16);
                    break;
                case 0x8f:
                    Device.Instance.MultiCmd commandMultiCmd = instance.new MultiCmd(this, instance);
                    instance.commandClasses.put(commandClassId, commandMultiCmd);
                    instance.commandClassesByName.put("multiCmd", commandMultiCmd);
                    break;
                case 0x6c:
                    Device.Instance.Supervision commandSupervision = instance.new Supervision(this, instance);
                    instance.commandClasses.put(commandClassId, commandSupervision);
                    instance.commandClassesByName.put("supervision", commandSupervision);
                    break;
                case 0x22:
                    Device.Instance.ApplicationStatus commandApplicationStatus = instance.new ApplicationStatus(this, instance);
                    instance.commandClasses.put(commandClassId, commandApplicationStatus);
                    instance.commandClassesByName.put("applicationStatus", commandApplicationStatus);
                    break;
                case 0x86:
                    Device.Instance.Version commandVersion = instance.new Version(this, instance);
                    instance.commandClasses.put(commandClassId, commandVersion);
                    instance.commandClassesByName.put("version", commandVersion);
                    break;
                case 0x5a:
                    Device.Instance.DeviceResetLocally commandDeviceResetLocally = instance.new DeviceResetLocally(this, instance);
                    instance.commandClasses.put(commandClassId, commandDeviceResetLocally);
                    instance.commandClassesByName.put("deviceResetLocally", commandDeviceResetLocally);
                    break;
                case 0x5b:
                    Device.Instance.CentralScene commandCentralScene = instance.new CentralScene(this, instance);
                    instance.commandClasses.put(commandClassId, commandCentralScene);
                    instance.commandClassesByName.put("centralScene", commandCentralScene);
                    break;
                 /* END AUTOGENERATED CC CODE */
            }
        } else if (type == commandRemoved) {
            Device.Instance instance = devices.get(deviceId).instances.get(instanceId);
            instance.commandClasses.remove(commandClassId);
            switch (commandClassId) {
                /* BEGIN AUTOGENERATED CC TEMPLATE
                case %cc_id%:
                    instance.commandClassesByName.remove("%cc_camel_case_name%");
                    break;
                 * END AUTOGENERATED CC TEMPLATE */
                /* BEGIN AUTOGENERATED CC CODE */
                case 0x20:
                    instance.commandClassesByName.remove("basic");
                    break;
                case 0x84:
                    instance.commandClassesByName.remove("wakeup");
                    break;
                case 0x0:
                    instance.commandClassesByName.remove("noOperation");
                    break;
                case 0x80:
                    instance.commandClassesByName.remove("battery");
                    break;
                case 0x72:
                    instance.commandClassesByName.remove("manufacturerSpecific");
                    break;
                case 0x88:
                    instance.commandClassesByName.remove("proprietary");
                    break;
                case 0x70:
                    instance.commandClassesByName.remove("configuration");
                    break;
                case 0x30:
                    instance.commandClassesByName.remove("sensorBinary");
                    break;
                case 0x85:
                    instance.commandClassesByName.remove("association");
                    break;
                case 0x32:
                    instance.commandClassesByName.remove("meter");
                    break;
                case 0x35:
                    instance.commandClassesByName.remove("meterPulse");
                    break;
                case 0x31:
                    instance.commandClassesByName.remove("sensorMultilevel");
                    break;
                case 0x9e:
                    instance.commandClassesByName.remove("sensorConfiguration");
                    break;
                case 0x27:
                    instance.commandClassesByName.remove("switchAll");
                    break;
                case 0x25:
                    instance.commandClassesByName.remove("switchBinary");
                    break;
                case 0x26:
                    instance.commandClassesByName.remove("switchMultilevel");
                    break;
                case 0x8e:
                    instance.commandClassesByName.remove("multiChannelAssociation");
                    break;
                case 0x60:
                    instance.commandClassesByName.remove("multiChannel");
                    break;
                case 0x77:
                    instance.commandClassesByName.remove("nodeNaming");
                    break;
                case 0x43:
                    instance.commandClassesByName.remove("thermostatSetPoint");
                    break;
                case 0x40:
                    instance.commandClassesByName.remove("thermostatMode");
                    break;
                case 0x44:
                    instance.commandClassesByName.remove("thermostatFanMode");
                    break;
                case 0x45:
                    instance.commandClassesByName.remove("thermostatFanState");
                    break;
                case 0x42:
                    instance.commandClassesByName.remove("thermostatOperatingState");
                    break;
                case 0x47:
                    instance.commandClassesByName.remove("thermostatSetBack");
                    break;
                case 0x9c:
                    instance.commandClassesByName.remove("alarmSensor");
                    break;
                case 0x62:
                    instance.commandClassesByName.remove("doorLock");
                    break;
                case 0x4c:
                    instance.commandClassesByName.remove("doorLockLogging");
                    break;
                case 0x63:
                    instance.commandClassesByName.remove("userCode");
                    break;
                case 0x8a:
                    instance.commandClassesByName.remove("time");
                    break;
                case 0x8b:
                    instance.commandClassesByName.remove("timeParameters");
                    break;
                case 0x81:
                    instance.commandClassesByName.remove("clock");
                    break;
                case 0x2b:
                    instance.commandClassesByName.remove("sceneActivation");
                    break;
                case 0x2d:
                    instance.commandClassesByName.remove("sceneControllerConf");
                    break;
                case 0x2c:
                    instance.commandClassesByName.remove("sceneActuatorConf");
                    break;
                case 0x87:
                    instance.commandClassesByName.remove("indicator");
                    break;
                case 0x75:
                    instance.commandClassesByName.remove("protection");
                    break;
                case 0x4e:
                    instance.commandClassesByName.remove("scheduleEntryLock");
                    break;
                case 0x46:
                    instance.commandClassesByName.remove("climateControlSchedule");
                    break;
                case 0x3d:
                    instance.commandClassesByName.remove("meterTableMonitor");
                    break;
                case 0x71:
                    instance.commandClassesByName.remove("alarm");
                    break;
                case 0x73:
                    instance.commandClassesByName.remove("powerLevel");
                    break;
                case 0x5e:
                    instance.commandClassesByName.remove("zWavePlusInfo");
                    break;
                case 0x7a:
                    instance.commandClassesByName.remove("firmwareUpdate");
                    break;
                case 0x59:
                    instance.commandClassesByName.remove("associationGroupInformation");
                    break;
                case 0x33:
                    instance.commandClassesByName.remove("switchColor");
                    break;
                case 0x79:
                    instance.commandClassesByName.remove("soundSwitch");
                    break;
                case 0x66:
                    instance.commandClassesByName.remove("barrierOperator");
                    break;
                case 0x94:
                    instance.commandClassesByName.remove("simpleAVControl");
                    break;
                case 0x98:
                    instance.commandClassesByName.remove("security");
                    break;
                case 0x9f:
                    instance.commandClassesByName.remove("securityS2");
                    break;
                case 0x6f:
                    instance.commandClassesByName.remove("entryControl");
                    break;
                case 0x82:
                    instance.commandClassesByName.remove("hail");
                    break;
                case 0x56:
                    instance.commandClassesByName.remove("cRC16");
                    break;
                case 0x8f:
                    instance.commandClassesByName.remove("multiCmd");
                    break;
                case 0x6c:
                    instance.commandClassesByName.remove("supervision");
                    break;
                case 0x22:
                    instance.commandClassesByName.remove("applicationStatus");
                    break;
                case 0x86:
                    instance.commandClassesByName.remove("version");
                    break;
                case 0x5a:
                    instance.commandClassesByName.remove("deviceResetLocally");
                    break;
                case 0x5b:
                    instance.commandClassesByName.remove("centralScene");
                    break;
                 /* END AUTOGENERATED CC CODE */
            }
        } else {
            System.out.println("Unhandled deviceCallback: type = " + type + ", id = " + deviceId + ", instance = " + instanceId + ", commandClass = " + commandClassId);
        }

        for (DeviceCallback dc : deviceCallbacks) {
            dc.deviceCallback(type, deviceId, instanceId, commandClassId);
        }
    }

    private void terminateCallback() {
        for (TerminateCallback tc : terminateCallbacks) {
            tc.terminateCallback();
        }
    }

    // JNI functions
    private native long jni_zwayInit(String name, String port, int speed, String config_folder, String translations_folder, String zddx_folder);
    private native void jni_finalize(long ptr);
    private native void jni_discover(long ptr);
    private native void jni_stop(long ptr);
    private native boolean jni_isIdle(long ptr);
    private native boolean jni_isRunning(long ptr);
    private native void jni_addNodeToNetwork(long ptr, boolean startStop);
    private native void jni_removeNodeFromNetwork(long ptr, boolean startStop);
    private native void jni_controllerChange(long ptr, boolean startStop);
    private native void jni_setSUCNodeId(long ptr, int nodeId);
    private native void jni_setSISNodeId(long ptr, int nodeId);
    private native void jni_disableSUCNodeId(long ptr, int nodeId);
    private native void jni_setDefault(long ptr);
    private native void jni_requestNetworkUpdate(long ptr);
    private native void jni_setLearnMode(long ptr, boolean startStop);
    private native int[] jni_backup(long ptr);
    private native void jni_restore(long ptr, int[] data, boolean full);
    private native void jni_nodeProvisioningDSKAdd(long ptr, int[] dsk);
    private native void jni_nodeProvisioningDSKRemove(long ptr, int[] dsk);
    private native long jni_zdataFind(long dh, String path, long jzway);
    private native long jni_zdataControllerFind(String path, long jzway);
    private native long jni_zdataDeviceFind(String path, int deviceId, long jzway);
    private native long jni_zdataInstanceFind(String path, int deviceId, int instanceId, long jzway);
    private native long jni_zdataCommandClassFind(String path, int deviceId, int instanceId, int commandClassId, long jzway);
    private native void jni_deviceSendNOP(long ptr, int nodeId, Object arg);
    private native void jni_deviceAwakeQueue(long ptr, int nodeId);
    private native void jni_deviceInterviewForce(long ptr, int deviceId);
    private native boolean jni_deviceIsInterviewDone(long ptr, int deviceId);
    private native void jni_deviceDelayCommunication(long ptr, int deviceID, int delay);
    private native void jni_deviceAssignReturnRoute(long ptr, int deviceID, int nodeId);
    private native void jni_deviceAssignPriorityReturnRoute(long ptr, int deviceID, int nodeId, int repeater1, int repeater2, int repeater3, int repeater4);
    private native void jni_deviceDeleteReturnRoute(long ptr, int deviceID);
    private native void jni_deviceAssignSUCReturnRoute(long ptr, int deviceID);
    private native void jni_deviceAssignPrioritySUCReturnRoute(long ptr, int deviceID, int repeater1, int repeater2, int repeater3, int repeater4);
    private native void jni_deviceDeleteSUCReturnRoute(long ptr, int deviceID);
    private native void jni_ZDDXSaveToXML(long ptr);
    private native void jni_commandInterview(long ptr, int deviceId, int instanceId, int ccId);


    /* BEGIN AUTOGENERATED FC TEMPLATE
    private native void jni_fc_%function_short_camel_case_name%(long ptr, %params_java_declarations%Object callbackArg);
     * END AUTOGENERATED FC TEMPLATE */
    /* BEGIN AUTOGENERATED FC CODE */
    private native void jni_fc_getSerialApiCapabilities(long ptr, Object callbackArg);
    private native void jni_fc_serialApiSetTimeouts(long ptr, int ackTimeout, int byteTimeout, Object callbackArg);
    private native void jni_fc_serialApiGetInitData(long ptr, Object callbackArg);
    private native void jni_fc_serialApiApplicationNodeInfo(long ptr, boolean listening, boolean optional, boolean flirs1000, boolean flirs250, int generic_class, int specific_class, int[] nif, Object callbackArg);
    private native void jni_fc_watchdogStart(long ptr, Object callbackArg);
    private native void jni_fc_watchdogStop(long ptr, Object callbackArg);
    private native void jni_fc_getHomeId(long ptr, Object callbackArg);
    private native void jni_fc_getControllerCapabilities(long ptr, Object callbackArg);
    private native void jni_fc_getVersion(long ptr, Object callbackArg);
    private native void jni_fc_getSucNodeId(long ptr, Object callbackArg);
    private native void jni_fc_enableSuc(long ptr, boolean enable, boolean sis, Object callbackArg);
    private native void jni_fc_setSucNodeId(long ptr, int node_id, boolean enable, boolean sis, Object callbackArg);
    private native void jni_fc_memoryGetByte(long ptr, int offset, Object callbackArg);
    private native void jni_fc_memoryGetBuffer(long ptr, int offset, int length, Object callbackArg);
    private native void jni_fc_memoryPutByte(long ptr, int offset, int data, Object callbackArg);
    private native void jni_fc_memoryPutBuffer(long ptr, int offset, int[] data, Object callbackArg);
    private native void jni_fc_nvmGetId(long ptr, Object callbackArg);
    private native void jni_fc_nvmExtReadLongByte(long ptr, int offset, Object callbackArg);
    private native void jni_fc_nvmExtReadLongBuffer(long ptr, int offset, int length, Object callbackArg);
    private native void jni_fc_nvmExtWriteLongByte(long ptr, int offset, int data, Object callbackArg);
    private native void jni_fc_nvmExtWriteLongBuffer(long ptr, int offset, int[] data, Object callbackArg);
    private native void jni_fc_isFailedNode(long ptr, int node_id, Object callbackArg);
    private native void jni_fc_sendDataAbort(long ptr, Object callbackArg);
    private native void jni_fc_serialApiSoftReset(long ptr, Object callbackArg);
    private native void jni_fc_sendData(long ptr, int node_id, int[] data, String description, Object callbackArg);
    private native void jni_fc_sendDataSecure(long ptr, int node_id, int[] data, String description, int key_class, Object callbackArg);
    private native void jni_fc_getNodeProtocolInfo(long ptr, int node_id, Object callbackArg);
    private native void jni_fc_getRoutingTableLine(long ptr, int node_id, boolean remove_bad, boolean remove_repeaters, Object callbackArg);
    private native void jni_fc_assignReturnRoute(long ptr, int node_id, int dest_id, Object callbackArg);
    private native void jni_fc_assignSucReturnRoute(long ptr, int node_id, Object callbackArg);
    private native void jni_fc_deleteReturnRoute(long ptr, int node_id, Object callbackArg);
    private native void jni_fc_deleteSucReturnRoute(long ptr, int node_id, Object callbackArg);
    private native void jni_fc_setDefault(long ptr, Object callbackArg);
    private native void jni_fc_sendSucNodeId(long ptr, int node_id, Object callbackArg);
    private native void jni_fc_sendNodeInformation(long ptr, int node_id, Object callbackArg);
    private native void jni_fc_requestNodeInformation(long ptr, int node_id, Object callbackArg);
    private native void jni_fc_removeFailedNode(long ptr, int node_id, Object callbackArg);
    private native void jni_fc_replaceFailedNode(long ptr, int node_id, Object callbackArg);
    private native void jni_fc_requestNetworkUpdate(long ptr, Object callbackArg);
    private native void jni_fc_requestNodeNeighbourUpdate(long ptr, int node_id, Object callbackArg);
    private native void jni_fc_setLearnMode(long ptr, boolean startStop, Object callbackArg);
    private native void jni_fc_addNodeToNetwork(long ptr, boolean startStop, boolean highPower, Object callbackArg);
    private native void jni_fc_smartStartEnable(long ptr, Object callbackArg);
    private native void jni_fc_removeNodeFromNetwork(long ptr, boolean startStop, boolean highPower, Object callbackArg);
    private native void jni_fc_removeNodeIdFromNetwork(long ptr, int nodeId, boolean startStop, boolean highPower, Object callbackArg);
    private native void jni_fc_controllerChange(long ptr, boolean startStop, boolean highPower, Object callbackArg);
    private native void jni_fc_createNewPrimary(long ptr, boolean startStop, Object callbackArg);
    private native void jni_fc_zmeFreqChange(long ptr, int freq, Object callbackArg);
    private native void jni_fc_rfPowerLevelSet(long ptr, int level, Object callbackArg);
    private native void jni_fc_rfPowerLevelGet(long ptr, Object callbackArg);
    private native void jni_fc_sendTestFrame(long ptr, int node_id, int level, Object callbackArg);
    private native void jni_fc_flashAutoProgSet(long ptr, Object callbackArg);
    private native void jni_fc_exploreRequestInclusion(long ptr, Object callbackArg);
    private native void jni_fc_exploreRequestExclusion(long ptr, Object callbackArg);
    private native void jni_fc_zmeBootloaderFlash(long ptr, int seg, Object callbackArg);
    private native void jni_fc_zmeBootloaderLoadFlash(long ptr, int[] data, Object callbackArg);
    private native void jni_fc_zmeCapabilities(long ptr, int[] data, Object callbackArg);
    private native void jni_fc_zmeLicenseSet(long ptr, int[] data, Object callbackArg);
    private native void jni_fc_applicationCommandHandlerInject(long ptr, int node_id, int[] data, Object callbackArg);
    private native void jni_fc_getBackgroundRssi(long ptr, Object callbackArg);
    private native void jni_fc_serialApiSetupSetIma(long ptr, boolean enable, Object callbackArg);
    private native void jni_fc_serialApiSetupSetTxPower(long ptr, int power, Object callbackArg);
    private native void jni_fc_clearNetworkStats(long ptr, Object callbackArg);
    private native void jni_fc_getNetworkStats(long ptr, Object callbackArg);
    private native void jni_fc_getPriorityRoute(long ptr, int node_id, Object callbackArg);
    private native void jni_fc_setPriorityRoute(long ptr, int node_id, int repeater1, int repeater2, int repeater3, int repeater4, int route_speed, Object callbackArg);
    private native void jni_fc_setPromiscuousMode(long ptr, boolean enable, Object callbackArg);
    private native void jni_fc_assignPriorityReturnRoute(long ptr, int node_id, int dest_id, int repeater1, int repeater2, int repeater3, int repeater4, int route_speed, Object callbackArg);
    private native void jni_fc_assignPrioritySucReturnRoute(long ptr, int node_id, int repeater1, int repeater2, int repeater3, int repeater4, int route_speed, Object callbackArg);
    private native void jni_fc_zmeSerialapiOptionsJammingDetectionSet(long ptr, int ch1_threshold, int ch2_threshold, int ch3_threshold, Object callbackArg);
    private native void jni_fc_zmeSerialapiOptionsAdvancedImaSet(long ptr, boolean enable, Object callbackArg);
    private native void jni_fc_zmeSerialapiOptionsStaticApiSet(long ptr, boolean enable, Object callbackArg);
    private native void jni_fc_zmeSerialapiOptionsUartSpeedSet(long ptr, int speed, Object callbackArg);
    private native void jni_fc_getLongRangeChannel(long ptr, Object callbackArg);
    private native void jni_fc_setLongRangeChannel(long ptr, int channel, Object callbackArg);
    private native void jni_fc_getLongRangeNodes(long ptr, Object callbackArg);
     /* END AUTOGENERATED FC CODE */

    /* BEGIN AUTOGENERATED CC TEMPLATE
     * BEGIN AUTOGENERATED CMD TEMPLATE
    private native void jni_cc_%function_short_camel_case_name%(long ptr, int deviceId, int instanceId, %params_java_declarations%Object callbackArg);
     * END AUTOGENERATED CMD TEMPLATE
     * END AUTOGENERATED CC TEMPLATE */
    /* BEGIN AUTOGENERATED CC CODE */
    private native void jni_cc_basicGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_basicSet(long ptr, int deviceId, int instanceId, int value, Object callbackArg);
    private native void jni_cc_wakeupGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_wakeupCapabilitiesGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_wakeupSleep(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_wakeupSet(long ptr, int deviceId, int instanceId, int interval, int notification_node_id, Object callbackArg);
    private native void jni_cc_batteryGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_manufacturerSpecificGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_manufacturerSpecificDeviceIdGet(long ptr, int deviceId, int instanceId, int type, Object callbackArg);
    private native void jni_cc_proprietaryGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_proprietarySet(long ptr, int deviceId, int instanceId, int[] data, Object callbackArg);
    private native void jni_cc_configurationGet(long ptr, int deviceId, int instanceId, int parameter, Object callbackArg);
    private native void jni_cc_configurationSet(long ptr, int deviceId, int instanceId, int parameter, int value, int size, Object callbackArg);
    private native void jni_cc_configurationSetDefault(long ptr, int deviceId, int instanceId, int parameter, Object callbackArg);
    private native void jni_cc_configurationDefaultReset(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_sensorBinaryGet(long ptr, int deviceId, int instanceId, int sensorType, Object callbackArg);
    private native void jni_cc_associationGet(long ptr, int deviceId, int instanceId, int group_id, Object callbackArg);
    private native void jni_cc_associationSet(long ptr, int deviceId, int instanceId, int group_id, int include_node, Object callbackArg);
    private native void jni_cc_associationRemove(long ptr, int deviceId, int instanceId, int group_id, int exclude_node, Object callbackArg);
    private native void jni_cc_associationGroupingsGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_meterGet(long ptr, int deviceId, int instanceId, int scale, Object callbackArg);
    private native void jni_cc_meterReset(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_meterSupported(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_meterPulseGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_sensorMultilevelGet(long ptr, int deviceId, int instanceId, int sensor_type, Object callbackArg);
    private native void jni_cc_sensorConfigurationGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_sensorConfigurationSet(long ptr, int deviceId, int instanceId, int mode, float value, Object callbackArg);
    private native void jni_cc_switchAllGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_switchAllSet(long ptr, int deviceId, int instanceId, int mode, Object callbackArg);
    private native void jni_cc_switchAllSetOn(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_switchAllSetOff(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_switchBinaryGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_switchBinarySet(long ptr, int deviceId, int instanceId, boolean value, int duration, Object callbackArg);
    private native void jni_cc_switchMultilevelGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_switchMultilevelSet(long ptr, int deviceId, int instanceId, int level, int duration, Object callbackArg);
    private native void jni_cc_switchMultilevelStartLevelChange(long ptr, int deviceId, int instanceId, int dir, int duration, boolean ignoreStartLevel, int startLevel, int incdec, int step, Object callbackArg);
    private native void jni_cc_switchMultilevelStopLevelChange(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_multichannelAssociationGet(long ptr, int deviceId, int instanceId, int group_id, Object callbackArg);
    private native void jni_cc_multichannelAssociationSet(long ptr, int deviceId, int instanceId, int group_id, int include_node, int include_instance, Object callbackArg);
    private native void jni_cc_multichannelAssociationRemove(long ptr, int deviceId, int instanceId, int group_id, int exclude_node, int exclude_instance, Object callbackArg);
    private native void jni_cc_multichannelAssociationGroupingsGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_multichannelGet(long ptr, int deviceId, int instanceId, int cc_id, Object callbackArg);
    private native void jni_cc_multichannelEndpointFind(long ptr, int deviceId, int instanceId, int generic, int specific, Object callbackArg);
    private native void jni_cc_multichannelEndpointGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_multichannelCapabilitiesGet(long ptr, int deviceId, int instanceId, int endpoint, Object callbackArg);
    private native void jni_cc_multichannelAggregatedMembersGet(long ptr, int deviceId, int instanceId, int endpoint, Object callbackArg);
    private native void jni_cc_nodeNamingGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_nodeNamingGetName(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_nodeNamingGetLocation(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_nodeNamingSetName(long ptr, int deviceId, int instanceId, String name, Object callbackArg);
    private native void jni_cc_nodeNamingSetLocation(long ptr, int deviceId, int instanceId, String location, Object callbackArg);
    private native void jni_cc_thermostatSetpointGet(long ptr, int deviceId, int instanceId, int mode, Object callbackArg);
    private native void jni_cc_thermostatSetpointSet(long ptr, int deviceId, int instanceId, int mode, float value, Object callbackArg);
    private native void jni_cc_thermostatModeGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_thermostatModeSet(long ptr, int deviceId, int instanceId, int mode, Object callbackArg);
    private native void jni_cc_thermostatModeSetManufacturerSpecific(long ptr, int deviceId, int instanceId, int[] manufacturerSpecifc, Object callbackArg);
    private native void jni_cc_thermostatFanModeGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_thermostatFanModeSet(long ptr, int deviceId, int instanceId, boolean on, int mode, Object callbackArg);
    private native void jni_cc_thermostatFanStateGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_thermostatOperatingStateGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_thermostatOperatingStateLoggingGet(long ptr, int deviceId, int instanceId, int state, Object callbackArg);
    private native void jni_cc_thermostatSetbackGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_thermostatSetbackSet(long ptr, int deviceId, int instanceId, int overrideType, int state, float value, Object callbackArg);
    private native void jni_cc_alarmSensorSupportedGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_alarmSensorGet(long ptr, int deviceId, int instanceId, int type, Object callbackArg);
    private native void jni_cc_doorLockGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_doorLockConfigurationGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_doorLockSet(long ptr, int deviceId, int instanceId, int mode, Object callbackArg);
    private native void jni_cc_doorLockConfigurationSet(long ptr, int deviceId, int instanceId, int operationType, int outsideHandlesEnabled, int insideHandlesEnabled, int lockTimeout, int autolockTime, int holdReleaseTime, boolean twistAssist, boolean blockToBlock, Object callbackArg);
    private native void jni_cc_doorLockLoggingGet(long ptr, int deviceId, int instanceId, int record, Object callbackArg);
    private native void jni_cc_userCodeGet(long ptr, int deviceId, int instanceId, int user, Object callbackArg);
    private native void jni_cc_userCodeGetAll(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_userCodeSet(long ptr, int deviceId, int instanceId, int user, String code, int status, Object callbackArg);
    private native void jni_cc_userCodeSetRaw(long ptr, int deviceId, int instanceId, int user, int[] code, int status, Object callbackArg);
    private native void jni_cc_userCodeMasterCodeGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_userCodeMasterCodeSet(long ptr, int deviceId, int instanceId, String code, Object callbackArg);
    private native void jni_cc_userCodeMasterCodeSetRaw(long ptr, int deviceId, int instanceId, int[] code, Object callbackArg);
    private native void jni_cc_userCodeKeypadModeGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_userCodeKeypadModeSet(long ptr, int deviceId, int instanceId, int mode, Object callbackArg);
    private native void jni_cc_userCodeChecksumGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_timeTimeGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_timeDateGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_timeOffsetGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_timeParametersGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_timeParametersSet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_clockGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_clockSet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_sceneActivationSet(long ptr, int deviceId, int instanceId, int sceneId, int dimmingDuration, Object callbackArg);
    private native void jni_cc_sceneControllerConfGet(long ptr, int deviceId, int instanceId, int group, Object callbackArg);
    private native void jni_cc_sceneControllerConfSet(long ptr, int deviceId, int instanceId, int group, int scene, int duration, Object callbackArg);
    private native void jni_cc_sceneActuatorConfGet(long ptr, int deviceId, int instanceId, int scene, Object callbackArg);
    private native void jni_cc_sceneActuatorConfSet(long ptr, int deviceId, int instanceId, int scene, int level, int dimming, boolean override, Object callbackArg);
    private native void jni_cc_indicatorGet(long ptr, int deviceId, int instanceId, int indicator_id, Object callbackArg);
    private native void jni_cc_indicatorSet(long ptr, int deviceId, int instanceId, int indicator_id, int property_id, int state, Object callbackArg);
    private native void jni_cc_indicatorSetMultiple(long ptr, int deviceId, int instanceId, int indicator_id, int[] property_ids, int[] states, Object callbackArg);
    private native void jni_cc_indicatorIdentify(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_protectionGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_protectionSet(long ptr, int deviceId, int instanceId, int state, int rfState, Object callbackArg);
    private native void jni_cc_protectionExclusiveGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_protectionExclusiveSet(long ptr, int deviceId, int instanceId, int controlNodeId, Object callbackArg);
    private native void jni_cc_protectionTimeoutGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_protectionTimeoutSet(long ptr, int deviceId, int instanceId, int timeout, Object callbackArg);
    private native void jni_cc_scheduleEntryLockEnable(long ptr, int deviceId, int instanceId, int user, boolean enable, Object callbackArg);
    private native void jni_cc_scheduleEntryLockWeekdayGet(long ptr, int deviceId, int instanceId, int user, int slot, Object callbackArg);
    private native void jni_cc_scheduleEntryLockWeekdaySet(long ptr, int deviceId, int instanceId, boolean modifyErase, int user, int slot, int dayOfWeek, int startHour, int startMinute, int stopHour, int stopMinute, Object callbackArg);
    private native void jni_cc_scheduleEntryLockYearGet(long ptr, int deviceId, int instanceId, int user, int slot, Object callbackArg);
    private native void jni_cc_scheduleEntryLockYearSet(long ptr, int deviceId, int instanceId, boolean modifyErase, int user, int slot, int startYear, int startMonth, int startDay, int startHour, int startMinute, int stopYear, int stopMonth, int stopDay, int stopHour, int stopMinute, Object callbackArg);
    private native void jni_cc_climateControlScheduleOverrideGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_climateControlScheduleOverrideSet(long ptr, int deviceId, int instanceId, int overrideType, int overrideState, Object callbackArg);
    private native void jni_cc_meterTableMonitorStatusDateGet(long ptr, int deviceId, int instanceId, int maxResults, long startDate, long endDate, Object callbackArg);
    private native void jni_cc_meterTableMonitorStatusDepthGet(long ptr, int deviceId, int instanceId, int maxResults, Object callbackArg);
    private native void jni_cc_meterTableMonitorCurrentDataGet(long ptr, int deviceId, int instanceId, int setId, Object callbackArg);
    private native void jni_cc_meterTableMonitorHistoricalDataGet(long ptr, int deviceId, int instanceId, int setId, int maxResults, long startDate, long endDate, Object callbackArg);
    private native void jni_cc_alarmGet(long ptr, int deviceId, int instanceId, int type, int event, Object callbackArg);
    private native void jni_cc_alarmSet(long ptr, int deviceId, int instanceId, int type, int level, Object callbackArg);
    private native void jni_cc_powerLevelGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_powerLevelSet(long ptr, int deviceId, int instanceId, int level, int timeout, Object callbackArg);
    private native void jni_cc_powerLevelTestNodeGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_powerLevelTestNodeSet(long ptr, int deviceId, int instanceId, int testNodeId, int level, int frameCount, Object callbackArg);
    private native void jni_cc_zwavePlusInfoGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_firmwareUpdateGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_firmwareUpdatePerform(long ptr, int deviceId, int instanceId, int manufacturerId, int firmwareId, int firmwareTarget, int[] data, boolean delayActivation, Object callbackArg);
    private native void jni_cc_firmwareUpdateActivation(long ptr, int deviceId, int instanceId, int manufacturerId, int firmwareId, int firmwareTarget, int[] data, Object callbackArg);
    private native void jni_cc_firmwareUpdatePrepare(long ptr, int deviceId, int instanceId, int manufacturerId, int firmwareId, int firmwareTarget, Object callbackArg);
    private native void jni_cc_associationGroupInformationGetInfo(long ptr, int deviceId, int instanceId, int groupId, Object callbackArg);
    private native void jni_cc_associationGroupInformationGetName(long ptr, int deviceId, int instanceId, int groupId, Object callbackArg);
    private native void jni_cc_associationGroupInformationGetCommands(long ptr, int deviceId, int instanceId, int groupId, Object callbackArg);
    private native void jni_cc_switchColorGet(long ptr, int deviceId, int instanceId, int capabilityId, Object callbackArg);
    private native void jni_cc_switchColorSet(long ptr, int deviceId, int instanceId, int capabilityId, int state, int duration, Object callbackArg);
    private native void jni_cc_switchColorSetMultiple(long ptr, int deviceId, int instanceId, int[] capabilityIds, int[] states, int duration, Object callbackArg);
    private native void jni_cc_switchColorStartStateChange(long ptr, int deviceId, int instanceId, int capabilityId, int dir, int duration, boolean ignoreStartLevel, int startLevel, Object callbackArg);
    private native void jni_cc_switchColorStopStateChange(long ptr, int deviceId, int instanceId, int capabilityId, Object callbackArg);
    private native void jni_cc_soundSwitchTonePlayGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_soundSwitchTonePlaySet(long ptr, int deviceId, int instanceId, int toneId, int volume, Object callbackArg);
    private native void jni_cc_soundSwitchConfigurationGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_soundSwitchConfigurationSet(long ptr, int deviceId, int instanceId, int toneId, int volume, Object callbackArg);
    private native void jni_cc_barrierOperatorGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_barrierOperatorSet(long ptr, int deviceId, int instanceId, int state, Object callbackArg);
    private native void jni_cc_barrierOperatorSignalGet(long ptr, int deviceId, int instanceId, int signalType, Object callbackArg);
    private native void jni_cc_barrierOperatorSignalSet(long ptr, int deviceId, int instanceId, int signalType, int state, Object callbackArg);
    private native void jni_cc_simpleAvControlSet(long ptr, int deviceId, int instanceId, int keyAttribute, int avCommand, Object callbackArg);
    private native void jni_cc_securityInject(long ptr, int deviceId, int instanceId, int[] data, Object callbackArg);
    private native void jni_cc_securityS2Inject(long ptr, int deviceId, int instanceId, int[] data, int keyClass, Object callbackArg);
    private native void jni_cc_entryControlConfigurationGet(long ptr, int deviceId, int instanceId, Object callbackArg);
    private native void jni_cc_entryControlConfigurationSet(long ptr, int deviceId, int instanceId, int keyCacheSize, int keyCacheTimeout, Object callbackArg);
     /* END AUTOGENERATED CC CODE */

    /* BEGIN AUTOGENERATED FC TEMPLATE
    public void %function_command_camel_name%(%params_java_declarations%Object callbackArg) {
        jni_fc_%function_short_camel_case_name%(jzway, %params_java%callbackArg);
    }
    public void %function_command_camel_name%(%params_java_declarations_no_comma%) {
        jni_fc_%function_short_camel_case_name%(jzway, %params_java%0);
    }
    
     * END AUTOGENERATED FC TEMPLATE */
    /* BEGIN AUTOGENERATED FC CODE */
    public void getSerialApiCapabilities(Object callbackArg) {
        jni_fc_getSerialApiCapabilities(jzway, callbackArg);
    }
    public void getSerialApiCapabilities() {
        jni_fc_getSerialApiCapabilities(jzway, 0);
    }
    
    public void serialApiSetTimeouts(int ackTimeout, int byteTimeout, Object callbackArg) {
        jni_fc_serialApiSetTimeouts(jzway, ackTimeout, byteTimeout, callbackArg);
    }
    public void serialApiSetTimeouts(int ackTimeout, int byteTimeout) {
        jni_fc_serialApiSetTimeouts(jzway, ackTimeout, byteTimeout, 0);
    }
    
    public void serialApiGetInitData(Object callbackArg) {
        jni_fc_serialApiGetInitData(jzway, callbackArg);
    }
    public void serialApiGetInitData() {
        jni_fc_serialApiGetInitData(jzway, 0);
    }
    
    public void serialApiApplicationNodeInfo(boolean listening, boolean optional, boolean flirs1000, boolean flirs250, int generic_class, int specific_class, int[] nif, Object callbackArg) {
        jni_fc_serialApiApplicationNodeInfo(jzway, listening, optional, flirs1000, flirs250, generic_class, specific_class, nif, callbackArg);
    }
    public void serialApiApplicationNodeInfo(boolean listening, boolean optional, boolean flirs1000, boolean flirs250, int generic_class, int specific_class, int[] nif) {
        jni_fc_serialApiApplicationNodeInfo(jzway, listening, optional, flirs1000, flirs250, generic_class, specific_class, nif, 0);
    }
    
    public void watchdogStart(Object callbackArg) {
        jni_fc_watchdogStart(jzway, callbackArg);
    }
    public void watchdogStart() {
        jni_fc_watchdogStart(jzway, 0);
    }
    
    public void watchdogStop(Object callbackArg) {
        jni_fc_watchdogStop(jzway, callbackArg);
    }
    public void watchdogStop() {
        jni_fc_watchdogStop(jzway, 0);
    }
    
    public void getHomeId(Object callbackArg) {
        jni_fc_getHomeId(jzway, callbackArg);
    }
    public void getHomeId() {
        jni_fc_getHomeId(jzway, 0);
    }
    
    public void getControllerCapabilities(Object callbackArg) {
        jni_fc_getControllerCapabilities(jzway, callbackArg);
    }
    public void getControllerCapabilities() {
        jni_fc_getControllerCapabilities(jzway, 0);
    }
    
    public void getVersion(Object callbackArg) {
        jni_fc_getVersion(jzway, callbackArg);
    }
    public void getVersion() {
        jni_fc_getVersion(jzway, 0);
    }
    
    public void getSucNodeId(Object callbackArg) {
        jni_fc_getSucNodeId(jzway, callbackArg);
    }
    public void getSucNodeId() {
        jni_fc_getSucNodeId(jzway, 0);
    }
    
    public void enableSuc(boolean enable, boolean sis, Object callbackArg) {
        jni_fc_enableSuc(jzway, enable, sis, callbackArg);
    }
    public void enableSuc(boolean enable, boolean sis) {
        jni_fc_enableSuc(jzway, enable, sis, 0);
    }
    
    public void setSucNodeId(int node_id, boolean enable, boolean sis, Object callbackArg) {
        jni_fc_setSucNodeId(jzway, node_id, enable, sis, callbackArg);
    }
    public void setSucNodeId(int node_id, boolean enable, boolean sis) {
        jni_fc_setSucNodeId(jzway, node_id, enable, sis, 0);
    }
    
    public void memoryGetByte(int offset, Object callbackArg) {
        jni_fc_memoryGetByte(jzway, offset, callbackArg);
    }
    public void memoryGetByte(int offset) {
        jni_fc_memoryGetByte(jzway, offset, 0);
    }
    
    public void memoryGetBuffer(int offset, int length, Object callbackArg) {
        jni_fc_memoryGetBuffer(jzway, offset, length, callbackArg);
    }
    public void memoryGetBuffer(int offset, int length) {
        jni_fc_memoryGetBuffer(jzway, offset, length, 0);
    }
    
    public void memoryPutByte(int offset, int data, Object callbackArg) {
        jni_fc_memoryPutByte(jzway, offset, data, callbackArg);
    }
    public void memoryPutByte(int offset, int data) {
        jni_fc_memoryPutByte(jzway, offset, data, 0);
    }
    
    public void memoryPutBuffer(int offset, int[] data, Object callbackArg) {
        jni_fc_memoryPutBuffer(jzway, offset, data, callbackArg);
    }
    public void memoryPutBuffer(int offset, int[] data) {
        jni_fc_memoryPutBuffer(jzway, offset, data, 0);
    }
    
    public void nvmGetId(Object callbackArg) {
        jni_fc_nvmGetId(jzway, callbackArg);
    }
    public void nvmGetId() {
        jni_fc_nvmGetId(jzway, 0);
    }
    
    public void nvmExtReadLongByte(int offset, Object callbackArg) {
        jni_fc_nvmExtReadLongByte(jzway, offset, callbackArg);
    }
    public void nvmExtReadLongByte(int offset) {
        jni_fc_nvmExtReadLongByte(jzway, offset, 0);
    }
    
    public void nvmExtReadLongBuffer(int offset, int length, Object callbackArg) {
        jni_fc_nvmExtReadLongBuffer(jzway, offset, length, callbackArg);
    }
    public void nvmExtReadLongBuffer(int offset, int length) {
        jni_fc_nvmExtReadLongBuffer(jzway, offset, length, 0);
    }
    
    public void nvmExtWriteLongByte(int offset, int data, Object callbackArg) {
        jni_fc_nvmExtWriteLongByte(jzway, offset, data, callbackArg);
    }
    public void nvmExtWriteLongByte(int offset, int data) {
        jni_fc_nvmExtWriteLongByte(jzway, offset, data, 0);
    }
    
    public void nvmExtWriteLongBuffer(int offset, int[] data, Object callbackArg) {
        jni_fc_nvmExtWriteLongBuffer(jzway, offset, data, callbackArg);
    }
    public void nvmExtWriteLongBuffer(int offset, int[] data) {
        jni_fc_nvmExtWriteLongBuffer(jzway, offset, data, 0);
    }
    
    public void isFailedNode(int node_id, Object callbackArg) {
        jni_fc_isFailedNode(jzway, node_id, callbackArg);
    }
    public void isFailedNode(int node_id) {
        jni_fc_isFailedNode(jzway, node_id, 0);
    }
    
    public void sendDataAbort(Object callbackArg) {
        jni_fc_sendDataAbort(jzway, callbackArg);
    }
    public void sendDataAbort() {
        jni_fc_sendDataAbort(jzway, 0);
    }
    
    public void serialApiSoftReset(Object callbackArg) {
        jni_fc_serialApiSoftReset(jzway, callbackArg);
    }
    public void serialApiSoftReset() {
        jni_fc_serialApiSoftReset(jzway, 0);
    }
    
    public void sendData(int node_id, int[] data, String description, Object callbackArg) {
        jni_fc_sendData(jzway, node_id, data, description, callbackArg);
    }
    public void sendData(int node_id, int[] data, String description) {
        jni_fc_sendData(jzway, node_id, data, description, 0);
    }
    
    public void sendDataSecure(int node_id, int[] data, String description, int key_class, Object callbackArg) {
        jni_fc_sendDataSecure(jzway, node_id, data, description, key_class, callbackArg);
    }
    public void sendDataSecure(int node_id, int[] data, String description, int key_class) {
        jni_fc_sendDataSecure(jzway, node_id, data, description, key_class, 0);
    }
    
    public void getNodeProtocolInfo(int node_id, Object callbackArg) {
        jni_fc_getNodeProtocolInfo(jzway, node_id, callbackArg);
    }
    public void getNodeProtocolInfo(int node_id) {
        jni_fc_getNodeProtocolInfo(jzway, node_id, 0);
    }
    
    public void getRoutingTableLine(int node_id, boolean remove_bad, boolean remove_repeaters, Object callbackArg) {
        jni_fc_getRoutingTableLine(jzway, node_id, remove_bad, remove_repeaters, callbackArg);
    }
    public void getRoutingTableLine(int node_id, boolean remove_bad, boolean remove_repeaters) {
        jni_fc_getRoutingTableLine(jzway, node_id, remove_bad, remove_repeaters, 0);
    }
    
    public void assignReturnRoute(int node_id, int dest_id, Object callbackArg) {
        jni_fc_assignReturnRoute(jzway, node_id, dest_id, callbackArg);
    }
    public void assignReturnRoute(int node_id, int dest_id) {
        jni_fc_assignReturnRoute(jzway, node_id, dest_id, 0);
    }
    
    public void assignSucReturnRoute(int node_id, Object callbackArg) {
        jni_fc_assignSucReturnRoute(jzway, node_id, callbackArg);
    }
    public void assignSucReturnRoute(int node_id) {
        jni_fc_assignSucReturnRoute(jzway, node_id, 0);
    }
    
    public void deleteReturnRoute(int node_id, Object callbackArg) {
        jni_fc_deleteReturnRoute(jzway, node_id, callbackArg);
    }
    public void deleteReturnRoute(int node_id) {
        jni_fc_deleteReturnRoute(jzway, node_id, 0);
    }
    
    public void deleteSucReturnRoute(int node_id, Object callbackArg) {
        jni_fc_deleteSucReturnRoute(jzway, node_id, callbackArg);
    }
    public void deleteSucReturnRoute(int node_id) {
        jni_fc_deleteSucReturnRoute(jzway, node_id, 0);
    }
    
    public void setDefault(Object callbackArg) {
        jni_fc_setDefault(jzway, callbackArg);
    }
    public void setDefault() {
        jni_fc_setDefault(jzway, 0);
    }
    
    public void sendSucNodeId(int node_id, Object callbackArg) {
        jni_fc_sendSucNodeId(jzway, node_id, callbackArg);
    }
    public void sendSucNodeId(int node_id) {
        jni_fc_sendSucNodeId(jzway, node_id, 0);
    }
    
    public void sendNodeInformation(int node_id, Object callbackArg) {
        jni_fc_sendNodeInformation(jzway, node_id, callbackArg);
    }
    public void sendNodeInformation(int node_id) {
        jni_fc_sendNodeInformation(jzway, node_id, 0);
    }
    
    public void requestNodeInformation(int node_id, Object callbackArg) {
        jni_fc_requestNodeInformation(jzway, node_id, callbackArg);
    }
    public void requestNodeInformation(int node_id) {
        jni_fc_requestNodeInformation(jzway, node_id, 0);
    }
    
    public void removeFailedNode(int node_id, Object callbackArg) {
        jni_fc_removeFailedNode(jzway, node_id, callbackArg);
    }
    public void removeFailedNode(int node_id) {
        jni_fc_removeFailedNode(jzway, node_id, 0);
    }
    
    public void replaceFailedNode(int node_id, Object callbackArg) {
        jni_fc_replaceFailedNode(jzway, node_id, callbackArg);
    }
    public void replaceFailedNode(int node_id) {
        jni_fc_replaceFailedNode(jzway, node_id, 0);
    }
    
    public void requestNetworkUpdate(Object callbackArg) {
        jni_fc_requestNetworkUpdate(jzway, callbackArg);
    }
    public void requestNetworkUpdate() {
        jni_fc_requestNetworkUpdate(jzway, 0);
    }
    
    public void requestNodeNeighbourUpdate(int node_id, Object callbackArg) {
        jni_fc_requestNodeNeighbourUpdate(jzway, node_id, callbackArg);
    }
    public void requestNodeNeighbourUpdate(int node_id) {
        jni_fc_requestNodeNeighbourUpdate(jzway, node_id, 0);
    }
    
    public void setLearnMode(boolean startStop, Object callbackArg) {
        jni_fc_setLearnMode(jzway, startStop, callbackArg);
    }
    public void setLearnMode(boolean startStop) {
        jni_fc_setLearnMode(jzway, startStop, 0);
    }
    
    public void addNodeToNetwork(boolean startStop, boolean highPower, Object callbackArg) {
        jni_fc_addNodeToNetwork(jzway, startStop, highPower, callbackArg);
    }
    public void addNodeToNetwork(boolean startStop, boolean highPower) {
        jni_fc_addNodeToNetwork(jzway, startStop, highPower, 0);
    }
    
    public void smartStartEnable(Object callbackArg) {
        jni_fc_smartStartEnable(jzway, callbackArg);
    }
    public void smartStartEnable() {
        jni_fc_smartStartEnable(jzway, 0);
    }
    
    public void removeNodeFromNetwork(boolean startStop, boolean highPower, Object callbackArg) {
        jni_fc_removeNodeFromNetwork(jzway, startStop, highPower, callbackArg);
    }
    public void removeNodeFromNetwork(boolean startStop, boolean highPower) {
        jni_fc_removeNodeFromNetwork(jzway, startStop, highPower, 0);
    }
    
    public void removeNodeIdFromNetwork(int nodeId, boolean startStop, boolean highPower, Object callbackArg) {
        jni_fc_removeNodeIdFromNetwork(jzway, nodeId, startStop, highPower, callbackArg);
    }
    public void removeNodeIdFromNetwork(int nodeId, boolean startStop, boolean highPower) {
        jni_fc_removeNodeIdFromNetwork(jzway, nodeId, startStop, highPower, 0);
    }
    
    public void controllerChange(boolean startStop, boolean highPower, Object callbackArg) {
        jni_fc_controllerChange(jzway, startStop, highPower, callbackArg);
    }
    public void controllerChange(boolean startStop, boolean highPower) {
        jni_fc_controllerChange(jzway, startStop, highPower, 0);
    }
    
    public void createNewPrimary(boolean startStop, Object callbackArg) {
        jni_fc_createNewPrimary(jzway, startStop, callbackArg);
    }
    public void createNewPrimary(boolean startStop) {
        jni_fc_createNewPrimary(jzway, startStop, 0);
    }
    
    public void zmeFreqChange(int freq, Object callbackArg) {
        jni_fc_zmeFreqChange(jzway, freq, callbackArg);
    }
    public void zmeFreqChange(int freq) {
        jni_fc_zmeFreqChange(jzway, freq, 0);
    }
    
    public void rfPowerLevelSet(int level, Object callbackArg) {
        jni_fc_rfPowerLevelSet(jzway, level, callbackArg);
    }
    public void rfPowerLevelSet(int level) {
        jni_fc_rfPowerLevelSet(jzway, level, 0);
    }
    
    public void rfPowerLevelGet(Object callbackArg) {
        jni_fc_rfPowerLevelGet(jzway, callbackArg);
    }
    public void rfPowerLevelGet() {
        jni_fc_rfPowerLevelGet(jzway, 0);
    }
    
    public void sendTestFrame(int node_id, int level, Object callbackArg) {
        jni_fc_sendTestFrame(jzway, node_id, level, callbackArg);
    }
    public void sendTestFrame(int node_id, int level) {
        jni_fc_sendTestFrame(jzway, node_id, level, 0);
    }
    
    public void flashAutoProgSet(Object callbackArg) {
        jni_fc_flashAutoProgSet(jzway, callbackArg);
    }
    public void flashAutoProgSet() {
        jni_fc_flashAutoProgSet(jzway, 0);
    }
    
    public void exploreRequestInclusion(Object callbackArg) {
        jni_fc_exploreRequestInclusion(jzway, callbackArg);
    }
    public void exploreRequestInclusion() {
        jni_fc_exploreRequestInclusion(jzway, 0);
    }
    
    public void exploreRequestExclusion(Object callbackArg) {
        jni_fc_exploreRequestExclusion(jzway, callbackArg);
    }
    public void exploreRequestExclusion() {
        jni_fc_exploreRequestExclusion(jzway, 0);
    }
    
    public void zmeBootloaderFlash(int seg, Object callbackArg) {
        jni_fc_zmeBootloaderFlash(jzway, seg, callbackArg);
    }
    public void zmeBootloaderFlash(int seg) {
        jni_fc_zmeBootloaderFlash(jzway, seg, 0);
    }
    
    public void zmeBootloaderLoadFlash(int[] data, Object callbackArg) {
        jni_fc_zmeBootloaderLoadFlash(jzway, data, callbackArg);
    }
    public void zmeBootloaderLoadFlash(int[] data) {
        jni_fc_zmeBootloaderLoadFlash(jzway, data, 0);
    }
    
    public void zmeCapabilities(int[] data, Object callbackArg) {
        jni_fc_zmeCapabilities(jzway, data, callbackArg);
    }
    public void zmeCapabilities(int[] data) {
        jni_fc_zmeCapabilities(jzway, data, 0);
    }
    
    public void zmeLicenseSet(int[] data, Object callbackArg) {
        jni_fc_zmeLicenseSet(jzway, data, callbackArg);
    }
    public void zmeLicenseSet(int[] data) {
        jni_fc_zmeLicenseSet(jzway, data, 0);
    }
    
    public void applicationCommandHandlerInject(int node_id, int[] data, Object callbackArg) {
        jni_fc_applicationCommandHandlerInject(jzway, node_id, data, callbackArg);
    }
    public void applicationCommandHandlerInject(int node_id, int[] data) {
        jni_fc_applicationCommandHandlerInject(jzway, node_id, data, 0);
    }
    
    public void getBackgroundRssi(Object callbackArg) {
        jni_fc_getBackgroundRssi(jzway, callbackArg);
    }
    public void getBackgroundRssi() {
        jni_fc_getBackgroundRssi(jzway, 0);
    }
    
    public void serialApiSetupSetIma(boolean enable, Object callbackArg) {
        jni_fc_serialApiSetupSetIma(jzway, enable, callbackArg);
    }
    public void serialApiSetupSetIma(boolean enable) {
        jni_fc_serialApiSetupSetIma(jzway, enable, 0);
    }
    
    public void serialApiSetupSetTxPower(int power, Object callbackArg) {
        jni_fc_serialApiSetupSetTxPower(jzway, power, callbackArg);
    }
    public void serialApiSetupSetTxPower(int power) {
        jni_fc_serialApiSetupSetTxPower(jzway, power, 0);
    }
    
    public void clearNetworkStats(Object callbackArg) {
        jni_fc_clearNetworkStats(jzway, callbackArg);
    }
    public void clearNetworkStats() {
        jni_fc_clearNetworkStats(jzway, 0);
    }
    
    public void getNetworkStats(Object callbackArg) {
        jni_fc_getNetworkStats(jzway, callbackArg);
    }
    public void getNetworkStats() {
        jni_fc_getNetworkStats(jzway, 0);
    }
    
    public void getPriorityRoute(int node_id, Object callbackArg) {
        jni_fc_getPriorityRoute(jzway, node_id, callbackArg);
    }
    public void getPriorityRoute(int node_id) {
        jni_fc_getPriorityRoute(jzway, node_id, 0);
    }
    
    public void setPriorityRoute(int node_id, int repeater1, int repeater2, int repeater3, int repeater4, int route_speed, Object callbackArg) {
        jni_fc_setPriorityRoute(jzway, node_id, repeater1, repeater2, repeater3, repeater4, route_speed, callbackArg);
    }
    public void setPriorityRoute(int node_id, int repeater1, int repeater2, int repeater3, int repeater4, int route_speed) {
        jni_fc_setPriorityRoute(jzway, node_id, repeater1, repeater2, repeater3, repeater4, route_speed, 0);
    }
    
    public void setPromiscuousMode(boolean enable, Object callbackArg) {
        jni_fc_setPromiscuousMode(jzway, enable, callbackArg);
    }
    public void setPromiscuousMode(boolean enable) {
        jni_fc_setPromiscuousMode(jzway, enable, 0);
    }
    
    public void assignPriorityReturnRoute(int node_id, int dest_id, int repeater1, int repeater2, int repeater3, int repeater4, int route_speed, Object callbackArg) {
        jni_fc_assignPriorityReturnRoute(jzway, node_id, dest_id, repeater1, repeater2, repeater3, repeater4, route_speed, callbackArg);
    }
    public void assignPriorityReturnRoute(int node_id, int dest_id, int repeater1, int repeater2, int repeater3, int repeater4, int route_speed) {
        jni_fc_assignPriorityReturnRoute(jzway, node_id, dest_id, repeater1, repeater2, repeater3, repeater4, route_speed, 0);
    }
    
    public void assignPrioritySucReturnRoute(int node_id, int repeater1, int repeater2, int repeater3, int repeater4, int route_speed, Object callbackArg) {
        jni_fc_assignPrioritySucReturnRoute(jzway, node_id, repeater1, repeater2, repeater3, repeater4, route_speed, callbackArg);
    }
    public void assignPrioritySucReturnRoute(int node_id, int repeater1, int repeater2, int repeater3, int repeater4, int route_speed) {
        jni_fc_assignPrioritySucReturnRoute(jzway, node_id, repeater1, repeater2, repeater3, repeater4, route_speed, 0);
    }
    
    public void zmeSerialapiOptionsJammingDetectionSet(int ch1_threshold, int ch2_threshold, int ch3_threshold, Object callbackArg) {
        jni_fc_zmeSerialapiOptionsJammingDetectionSet(jzway, ch1_threshold, ch2_threshold, ch3_threshold, callbackArg);
    }
    public void zmeSerialapiOptionsJammingDetectionSet(int ch1_threshold, int ch2_threshold, int ch3_threshold) {
        jni_fc_zmeSerialapiOptionsJammingDetectionSet(jzway, ch1_threshold, ch2_threshold, ch3_threshold, 0);
    }
    
    public void zmeSerialapiOptionsAdvancedImaSet(boolean enable, Object callbackArg) {
        jni_fc_zmeSerialapiOptionsAdvancedImaSet(jzway, enable, callbackArg);
    }
    public void zmeSerialapiOptionsAdvancedImaSet(boolean enable) {
        jni_fc_zmeSerialapiOptionsAdvancedImaSet(jzway, enable, 0);
    }
    
    public void zmeSerialapiOptionsStaticApiSet(boolean enable, Object callbackArg) {
        jni_fc_zmeSerialapiOptionsStaticApiSet(jzway, enable, callbackArg);
    }
    public void zmeSerialapiOptionsStaticApiSet(boolean enable) {
        jni_fc_zmeSerialapiOptionsStaticApiSet(jzway, enable, 0);
    }
    
    public void zmeSerialapiOptionsUartSpeedSet(int speed, Object callbackArg) {
        jni_fc_zmeSerialapiOptionsUartSpeedSet(jzway, speed, callbackArg);
    }
    public void zmeSerialapiOptionsUartSpeedSet(int speed) {
        jni_fc_zmeSerialapiOptionsUartSpeedSet(jzway, speed, 0);
    }
    
    public void getLongRangeChannel(Object callbackArg) {
        jni_fc_getLongRangeChannel(jzway, callbackArg);
    }
    public void getLongRangeChannel() {
        jni_fc_getLongRangeChannel(jzway, 0);
    }
    
    public void setLongRangeChannel(int channel, Object callbackArg) {
        jni_fc_setLongRangeChannel(jzway, channel, callbackArg);
    }
    public void setLongRangeChannel(int channel) {
        jni_fc_setLongRangeChannel(jzway, channel, 0);
    }
    
    public void getLongRangeNodes(Object callbackArg) {
        jni_fc_getLongRangeNodes(jzway, callbackArg);
    }
    public void getLongRangeNodes() {
        jni_fc_getLongRangeNodes(jzway, 0);
    }
    
     /* END AUTOGENERATED FC CODE */


    public interface DataCallback {
        public void dataCallback(Data data, Integer type);
    }

    public interface DeviceCallback {
        public void deviceCallback(Integer type, Integer deviceId, Integer instanceId, Integer commandClassId);
    }

    public interface StatusCallback {
        public void statusCallback(boolean result, Object obj);
    }

    public interface TerminateCallback {
        public void terminateCallback();
    }
    
    public final class Data {
        public static final int phantomUpdate = 0x40;
        public static final int childEvent = 0x80;
        public static final int updated = 0x01;
        public static final int invalidated = 0x02;
        public static final int deleted = 0x03;
        public static final int childCreated = 0x04;
        
        public static final int Empty = 0;
        public static final int Bool = 1;
        public static final int Int = 2;
        public static final int Float = 3;
        public static final int String = 4;
        public static final int Binary = 5;
        public static final int IntArray = 6;
        public static final int FloatArray = 7;
        public static final int StringArray = 8;

        private Object value;
        private Type valueType;
        private String valueTypeStr;

        private long dh;

        public final String name;
        public final String path;

        private long updateTime;
        private long invalidateTime;

        private Boolean isAlive;
        
        private Set<DataCallback> callbacks;
        
        // exceptions
        
        public final class NotAlive extends Throwable {
            public final Data data;
            
            public NotAlive(Data data) {
                super();
                this.data = data;
            }
        }

        public final class NotFound extends Throwable {
            public NotFound() {
                super();
            }
        }

        // constructors
        
        public Data(String path, long dhParent) throws NotFound, Exception {
            this(jni_zdataFind(dhParent, path, jzway));
        }

        public Data(String path) throws NotFound, Exception {
            this(jni_zdataControllerFind(path, jzway));
        }

        public Data(String path, int deviceId) throws NotFound, Exception {
            this(jni_zdataDeviceFind(path, deviceId, jzway));
        }

        public Data(String path, int deviceId, int instanceId) throws NotFound, Exception {
            this(jni_zdataInstanceFind(path, deviceId, instanceId, jzway));
        }

        public Data(String path, int deviceId, int instanceId, int commandClassId) throws NotFound, Exception {
            this(jni_zdataCommandClassFind(path, deviceId, instanceId, commandClassId, jzway));
        }

        private Data(long dh) throws NotFound, Exception {
            if (dh == 0) {
                throw new NotFound();
            }
            this.dh = dh;
            name = jni_zdataGetName(dh);
            path = jni_zdataGetPath(dh);
            isAlive = true;
            callbacks = new HashSet<>();
            updateTime = jni_zdataGetUpdateTime(dh);
            invalidateTime = jni_zdataGetInvalidateTime(dh);
            jni_zdataAddCallback(dh);
            getValue();
        }

        @Override
        public void finalize() {
            jni_zdataRemoveCallback(dh);
        }

        private void isAlive() throws NotAlive {
            if (!isAlive) {
                throw new NotAlive(this);
            }
        }
        
        public void bind(DataCallback func) throws NotAlive, Exception {
            isAlive();
            
            callbacks.add(func);
        }
        
        public void unbind(DataCallback func) throws NotAlive, Exception {
            isAlive();
            
            callbacks.remove(func);
        }

        public long getUpdateTime() {
            return updateTime;
        }

        public long getInvalidateTime() {
            return invalidateTime;
        }

        private void dataCallback(int type) throws Exception {
            // get type of the event
            boolean isPhantom = (phantomUpdate & type) > 0;
            boolean isChild = (childEvent & type) > 0;
            type = type & (~phantomUpdate) & (~childEvent);
            if (type == updated) {
                getValue();
            } else if (type == invalidated) {
                jni_zdataGetInvalidateTime(dh);
            } else if (type == deleted) {
                isAlive = false;
            } else if (type == childCreated) {
                // nothing to do
            } else {
                throw new Exception("Type of the event is an invalid integer.");
            }

            if (isPhantom) {
                jni_zdataGetUpdateTime(dh);
            }
            
            for (DataCallback dc : callbacks) {
                dc.dataCallback(this, type);
            }
        }

        private void getValue() throws Exception {
            int dataType = jni_zdataGetType(dh);
            if (dataType == Bool) {
                valueType = Boolean.class;
                valueTypeStr = "Boolean";
                value = jni_zdataGetBoolean(dh);
            } else if (dataType == Int) {
                valueType = Integer.class;
                valueTypeStr = "Integer";
                value = jni_zdataGetInteger(dh);
            } else if (dataType == Float) {
                valueType = Float.class;
                valueTypeStr = "Float";
                value = jni_zdataGetFloat(dh);
            } else if (dataType == String) {
                valueType = String.class;
                valueTypeStr = "String";
                value = jni_zdataGetString(dh);
            } else if (dataType == Binary) {
                valueType = Byte[].class;
                valueTypeStr = "Byte[]";
                int[] val = jni_zdataGetBinary(dh);
                value = Arrays.stream(val).boxed().toArray( Integer[]::new );
            } else if (dataType == IntArray) {
                valueType = Integer[].class;
                valueTypeStr = "Integer[]";
                int[] val = jni_zdataGetIntArray(dh);
                value = Arrays.stream(val).boxed().toArray( Integer[]::new );
            } else if (dataType == FloatArray) {
                valueType = Float[].class;
                valueTypeStr = "Float[]";
                float[] val = jni_zdataGetFloatArray(dh);
                int len = val.length;
                Float[] newValue = new Float[len];
                for (int i = 0; i < len; i++) {
                    newValue[i] = val[i];
                }
                value = newValue;
            } else if (dataType == StringArray) {
                valueType = String[].class;
                valueTypeStr = "String[]";
                value = jni_zdataGetStringArray(dh);
            } else if (dataType == Empty) {
                valueType = Object.class;
                valueTypeStr = "Null";
                value = null;
            } else {
                throw new Exception("Type of the value in data holder is an invalid integer.");
            }
        }

        public Data[] getChildren() throws NotAlive, Exception {
            isAlive();
            
            long[] list = jni_zdataGetChildren(dh);
            int length = list.length;
            Data[] children = new Data[length];
            for (int i = 0; i < length; i++) {
                try {
                    children[i] = new Data(list[i]);
                } catch (NotFound e) {
                    // should never happen
                    throw new RuntimeException();
                }
            }
            return children;
        }
        
        public Data get(String path) throws NotAlive, NotFound, Exception {
            isAlive();
            
            return new Data(path, this.dh);
        }

        public Type getValueType() throws NotAlive {
            isAlive();
            
            return valueType;
        }

        public String getValueTypeStr() throws NotAlive {
            isAlive();
            
            return valueTypeStr;
        }

        public void setBool(Boolean data) throws NotAlive {
            isAlive();
            
            jni_zdataSetBoolean(dh, data);
        }

        public void setInt(Integer data) throws NotAlive {
            isAlive();
            
            jni_zdataSetInteger(dh, data);
        }

        public void setFloat(Float data) throws NotAlive {
            isAlive();
            
            jni_zdataSetFloat(dh, data);
        }

        public void setString(String data) throws NotAlive {
            isAlive();
            
            jni_zdataSetString(dh, data);
        }

        public void setByteList(Integer[] data) throws NotAlive {
            isAlive();
            
            int size = data.length;
            int[] rdata = new int[size];
            for (int i = 0; i < size; i++) {
                rdata[i] = data[i];
            }
            jni_zdataSetBinary(dh, rdata);
        }

        public void setIntList(Integer[] data) throws NotAlive {
            isAlive();
            
            int size = data.length;
            int[] rdata = new int[size];
            for (int i = 0; i < size; i++) {
                rdata[i] = data[i];
            }
            jni_zdataSetIntArray(dh, rdata);
        }

        public void setFloatList(Float[] data) throws NotAlive {
            isAlive();
            
            int size = data.length;
            float[] rdata = new float[size];
            for (int i = 0; i < size; i++) {
                rdata[i] = data[i];
            }
            jni_zdataSetFloatArray(dh, rdata);
        }

        public void setStringList(String[] data) throws NotAlive {
            isAlive();

            jni_zdataSetStringArray(dh, data);
        }

        public void setEmpty() throws NotAlive {
            isAlive();
            
            jni_zdataSetEmpty(dh);
        }

        public Boolean getBool() throws NotAlive {
            isAlive();
            
            if (valueType == Boolean.class && valueTypeStr.equals("Boolean")) {
                return (Boolean) value;
            } else {
                throw new ClassCastException("Illegal call: value is not Boolean");
            }
        }

        public Integer getInt() throws NotAlive {
            isAlive();
            
            if (valueType == Integer.class && valueTypeStr.equals("Integer")) {
                return (Integer) value;
            } else {
                throw new ClassCastException("Illegal call: value is not Integer");
            }
        }

        public Float getFloat() throws NotAlive {
            isAlive();
            
            if (valueType == Float.class && valueTypeStr.equals("Float")) {
                return (Float) value;
            } else {
                throw new ClassCastException("Illegal call: value is not Float");
            }
        }

        public String getString() throws NotAlive {
            isAlive();
            
            if (valueType == String.class && valueTypeStr.equals("String")) {
                return (String) value;
            } else {
                throw new ClassCastException("Illegal call: value is not String");
            }
        }

        public Integer[] getByteList() throws NotAlive {
            isAlive();
            
            if (valueType == Byte[].class && valueTypeStr.equals("Byte[]")) {
                return (Integer[]) value;
            } else {
                throw new ClassCastException("Illegal call: value is not Integer[]");
            }
        }

        public Integer[] getIntList() throws NotAlive {
            isAlive();
            
            if (valueType == Integer[].class && valueTypeStr.equals("Integer[]")) {
                return (Integer[]) value;
            } else {
                throw new ClassCastException("Illegal call: value is not Integer[]");
            }
        }

        public Float[] getFloatList() throws NotAlive {
            isAlive();
            
            if (valueType == Float[].class && valueTypeStr.equals("Float[]")) {
                return (Float[]) value;
            } else {
                throw new ClassCastException("Illegal call: value is not Float[]");
            }
        }

        public String[] getStringList() throws NotAlive {
            isAlive();
            
            if (valueType == String[].class && valueTypeStr.equals("String[]")) {
                return (String[]) value;
            } else {
                throw new ClassCastException("Illegal call: value is not String[]");
            }
        }

        // JNI functions
        private native void jni_zdataAddCallback(long dh);
        private native void jni_zdataRemoveCallback(long dh);
        private native String jni_zdataGetName(long data);
        private native int jni_zdataGetType(long dh);
        private native boolean jni_zdataGetBoolean(long dh);
        private native int jni_zdataGetInteger(long dh);
        private native float jni_zdataGetFloat(long dh);
        private native String jni_zdataGetString(long dh);
        private native int[] jni_zdataGetBinary(long dh);
        private native int[] jni_zdataGetIntArray(long dh);
        private native float[] jni_zdataGetFloatArray(long dh);
        private native String[] jni_zdataGetStringArray(long dh);
        private native void jni_zdataSetEmpty(long dh);
        private native void jni_zdataSetBoolean(long dh, boolean data);
        private native void jni_zdataSetInteger(long dh, int data);
        private native void jni_zdataSetFloat(long dh, float data);
        private native void jni_zdataSetString(long dh, String data);
        private native void jni_zdataSetBinary(long dh, int[] data);
        private native void jni_zdataSetIntArray(long dh, int[] data);
        private native void jni_zdataSetFloatArray(long dh, float[] data);
        private native void jni_zdataSetStringArray(long dh, String[] data);
        private native long[] jni_zdataGetChildren(long dh);
        private native String jni_zdataGetPath(long dh);
        private native long jni_zdataGetUpdateTime(long dh);
        private native long jni_zdataGetInvalidateTime(long dh);
    }

    public final class Controller {
        public final Data data;

        public Controller() throws Exception {
            try {
                data = new Data("");
            } catch (Data.NotFound e) {
                // should never happen
                throw new RuntimeException();
            }
        }
        
        public void addNodeToNetwork(boolean startStop) {
            jni_addNodeToNetwork(jzway, startStop);
        }

        public void removeNodeFromNetwork(boolean startStop) {
            jni_removeNodeFromNetwork(jzway, startStop);
        }

        public void change(boolean startStop) {
            jni_controllerChange(jzway, startStop);
        }

        public void getSUCNodeId() {
            jni_fc_getSucNodeId(jzway, 0);
        }

        public void setSUCNodeId(int nodeId) {
            jni_setSUCNodeId(jzway, nodeId);
        }

        public void setSISNodeId(int nodeId) {
            jni_setSISNodeId(jzway, nodeId);
        }

        public void disableSUCNodeId(int nodeId) {
            jni_disableSUCNodeId(jzway, nodeId);
        }

        public void sendNodeInformation(int nodeId) {
            jni_fc_sendNodeInformation(jzway, nodeId, 0);
        }

        public void setDefault() {
            jni_setDefault(jzway);
        }

        public void requestNetworkUpdate() {
            jni_requestNetworkUpdate(jzway);
        }

        public void setLearnMode(boolean startStop) {
            jni_setLearnMode(jzway, startStop);
        }

        public int[] backup() {
            return jni_backup(jzway);
        }

        public void restore(int[] data, boolean full) {
            jni_restore(jzway, data, full);
        }

        public void nodeProvisioningDSKAdd(int[] dsk) {
            jni_nodeProvisioningDSKAdd(jzway, dsk);
        }

        public void nodeProvisioningDSKRemove(int[] dsk) {
            jni_nodeProvisioningDSKRemove(jzway, dsk);
        }

        public void ZDDXSaveToXML() { jni_ZDDXSaveToXML(jzway); }
    }

    public final class Device {
        public final Integer id;
        public final Data data;

        public Map<Integer, Instance> instances;

        public Device(ZWay zway, Integer device_id) throws Exception {
            id = device_id;
            try {
                data = new Data("", device_id);
            } catch (Data.NotFound e) {
                // should never happen
                throw new RuntimeException();
            }
            instances = new HashMap<>();
        }

        public void deviceSendNOP(Integer nodeId, Object arg) { jni_deviceSendNOP(jzway, nodeId, arg); }
        public void deviceAwakeQueue(Integer nodeId) { jni_deviceAwakeQueue(jzway, nodeId); }
        public void deviceInterviewForce(Integer deviceId) { jni_deviceInterviewForce(jzway, deviceId); }
        public boolean deviceIsInterviewDone(Integer deviceId) { return jni_deviceIsInterviewDone(jzway, deviceId); }
        public void deviceDelayCommunication(Integer deviceId, Integer delay) { jni_deviceDelayCommunication(jzway, deviceId, delay); }
        public void deviceAssignReturnRoute(Integer deviceId, Integer nodeId) { jni_deviceAssignReturnRoute(jzway, deviceId, nodeId); }
        public void deviceAssignPriorityReturnRoute(Integer deviceId, Integer nodeId, Integer repeater1, Integer repeater2, Integer repeater3, Integer repeater4) { jni_deviceAssignPriorityReturnRoute(jzway, deviceId, nodeId, repeater1, repeater2, repeater3, repeater4); }
        public void deviceDeleteReturnRoute(Integer deviceId) { jni_deviceDeleteReturnRoute(jzway, deviceId); }
        public void deviceAssignSUCReturnRoute(Integer deviceId) { jni_deviceAssignSUCReturnRoute(jzway, deviceId); }
        public void deviceAssignPrioritySUCReturnRoute(Integer deviceId, Integer repeater1, Integer repeater2, Integer repeater3, Integer repeater4) { jni_deviceAssignPrioritySUCReturnRoute(jzway, deviceId, repeater1, repeater2, repeater3, repeater4); }
        public void deviceDeleteSUCReturnRoute(Integer deviceId) { jni_deviceDeleteSUCReturnRoute(jzway, deviceId); }

        public final class Instance {
            private final Device device;

            public final Integer id;
            public final Data data;

            public Map<Integer, CommandClass> commandClasses;
            public Map<String, CommandClass> commandClassesByName;
            
            public Instance(ZWay zway, Device dev, Integer instance_id) throws Exception {
                device = dev;
                id = instance_id;
                try {
                    data = new Data("", dev.id, instance_id);
                } catch (Data.NotFound e) {
                    // should never happen
                    throw new RuntimeException();
                }
                commandClasses = new HashMap<>();
                commandClassesByName = new HashMap<>();
            }

            public class CommandClass {
                protected final Instance instance;

                public final Data data = null;
                
                public CommandClass(ZWay zway, Instance inst) throws Exception {
                    instance = inst;
                }

                public void interview(int deviceId, int instanceId, int ccId) { jni_commandInterview(jzway, deviceId, instanceId, ccId); }
            }

            /* BEGIN AUTOGENERATED CC TEMPLATE
            public final class %cc_capitalized_name% extends CommandClass {
                public final static int id = %cc_id%;
                public final Data data;

                public %cc_capitalized_name%(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, %cc_id%);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

             * BEGIN AUTOGENERATED CMD TEMPLATE
                public void %function_command_camel_name%(%params_java_declarations%Object callbackArg) {
                    jni_cc_%function_short_camel_case_name%(jzway, instance.device.id, instance.id, %params_java%callbackArg);
                }
                public void %function_command_camel_name%(%params_java_declarations_no_comma%) {
                    jni_cc_%function_short_camel_case_name%(jzway, instance.device.id, instance.id, %params_java%0);
                }
                
             * END AUTOGENERATED CMD TEMPLATE
            }
            
             * END AUTOGENERATED CC TEMPLATE */
            /* BEGIN AUTOGENERATED CC CODE */
            public final class Basic extends CommandClass {
                public final static int id = 0x20;
                public final Data data;

                public Basic(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x20);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_basicGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_basicGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(int value, Object callbackArg) {
                    jni_cc_basicSet(jzway, instance.device.id, instance.id, value, callbackArg);
                }
                public void set(int value) {
                    jni_cc_basicSet(jzway, instance.device.id, instance.id, value, 0);
                }
                
            }
            
            public final class Wakeup extends CommandClass {
                public final static int id = 0x84;
                public final Data data;

                public Wakeup(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x84);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_wakeupGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_wakeupGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void capabilitiesGet(Object callbackArg) {
                    jni_cc_wakeupCapabilitiesGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void capabilitiesGet() {
                    jni_cc_wakeupCapabilitiesGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void sleep(Object callbackArg) {
                    jni_cc_wakeupSleep(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void sleep() {
                    jni_cc_wakeupSleep(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(int interval, int notification_node_id, Object callbackArg) {
                    jni_cc_wakeupSet(jzway, instance.device.id, instance.id, interval, notification_node_id, callbackArg);
                }
                public void set(int interval, int notification_node_id) {
                    jni_cc_wakeupSet(jzway, instance.device.id, instance.id, interval, notification_node_id, 0);
                }
                
            }
            
            public final class NoOperation extends CommandClass {
                public final static int id = 0x0;
                public final Data data;

                public NoOperation(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x0);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

            }
            
            public final class Battery extends CommandClass {
                public final static int id = 0x80;
                public final Data data;

                public Battery(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x80);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_batteryGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_batteryGet(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class ManufacturerSpecific extends CommandClass {
                public final static int id = 0x72;
                public final Data data;

                public ManufacturerSpecific(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x72);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_manufacturerSpecificGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_manufacturerSpecificGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void deviceIdGet(int type, Object callbackArg) {
                    jni_cc_manufacturerSpecificDeviceIdGet(jzway, instance.device.id, instance.id, type, callbackArg);
                }
                public void deviceIdGet(int type) {
                    jni_cc_manufacturerSpecificDeviceIdGet(jzway, instance.device.id, instance.id, type, 0);
                }
                
            }
            
            public final class Proprietary extends CommandClass {
                public final static int id = 0x88;
                public final Data data;

                public Proprietary(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x88);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_proprietaryGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_proprietaryGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(int[] data, Object callbackArg) {
                    jni_cc_proprietarySet(jzway, instance.device.id, instance.id, data, callbackArg);
                }
                public void set(int[] data) {
                    jni_cc_proprietarySet(jzway, instance.device.id, instance.id, data, 0);
                }
                
            }
            
            public final class Configuration extends CommandClass {
                public final static int id = 0x70;
                public final Data data;

                public Configuration(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x70);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(int parameter, Object callbackArg) {
                    jni_cc_configurationGet(jzway, instance.device.id, instance.id, parameter, callbackArg);
                }
                public void get(int parameter) {
                    jni_cc_configurationGet(jzway, instance.device.id, instance.id, parameter, 0);
                }
                
                public void set(int parameter, int value, int size, Object callbackArg) {
                    jni_cc_configurationSet(jzway, instance.device.id, instance.id, parameter, value, size, callbackArg);
                }
                public void set(int parameter, int value, int size) {
                    jni_cc_configurationSet(jzway, instance.device.id, instance.id, parameter, value, size, 0);
                }
                
                public void setDefault(int parameter, Object callbackArg) {
                    jni_cc_configurationSetDefault(jzway, instance.device.id, instance.id, parameter, callbackArg);
                }
                public void setDefault(int parameter) {
                    jni_cc_configurationSetDefault(jzway, instance.device.id, instance.id, parameter, 0);
                }
                
                public void defaultReset(Object callbackArg) {
                    jni_cc_configurationDefaultReset(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void defaultReset() {
                    jni_cc_configurationDefaultReset(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class SensorBinary extends CommandClass {
                public final static int id = 0x30;
                public final Data data;

                public SensorBinary(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x30);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(int sensorType, Object callbackArg) {
                    jni_cc_sensorBinaryGet(jzway, instance.device.id, instance.id, sensorType, callbackArg);
                }
                public void get(int sensorType) {
                    jni_cc_sensorBinaryGet(jzway, instance.device.id, instance.id, sensorType, 0);
                }
                
            }
            
            public final class Association extends CommandClass {
                public final static int id = 0x85;
                public final Data data;

                public Association(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x85);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(int group_id, Object callbackArg) {
                    jni_cc_associationGet(jzway, instance.device.id, instance.id, group_id, callbackArg);
                }
                public void get(int group_id) {
                    jni_cc_associationGet(jzway, instance.device.id, instance.id, group_id, 0);
                }
                
                public void set(int group_id, int include_node, Object callbackArg) {
                    jni_cc_associationSet(jzway, instance.device.id, instance.id, group_id, include_node, callbackArg);
                }
                public void set(int group_id, int include_node) {
                    jni_cc_associationSet(jzway, instance.device.id, instance.id, group_id, include_node, 0);
                }
                
                public void remove(int group_id, int exclude_node, Object callbackArg) {
                    jni_cc_associationRemove(jzway, instance.device.id, instance.id, group_id, exclude_node, callbackArg);
                }
                public void remove(int group_id, int exclude_node) {
                    jni_cc_associationRemove(jzway, instance.device.id, instance.id, group_id, exclude_node, 0);
                }
                
                public void groupingsGet(Object callbackArg) {
                    jni_cc_associationGroupingsGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void groupingsGet() {
                    jni_cc_associationGroupingsGet(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class Meter extends CommandClass {
                public final static int id = 0x32;
                public final Data data;

                public Meter(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x32);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(int scale, Object callbackArg) {
                    jni_cc_meterGet(jzway, instance.device.id, instance.id, scale, callbackArg);
                }
                public void get(int scale) {
                    jni_cc_meterGet(jzway, instance.device.id, instance.id, scale, 0);
                }
                
                public void reset(Object callbackArg) {
                    jni_cc_meterReset(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void reset() {
                    jni_cc_meterReset(jzway, instance.device.id, instance.id, 0);
                }
                
                public void supported(Object callbackArg) {
                    jni_cc_meterSupported(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void supported() {
                    jni_cc_meterSupported(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class MeterPulse extends CommandClass {
                public final static int id = 0x35;
                public final Data data;

                public MeterPulse(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x35);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_meterPulseGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_meterPulseGet(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class SensorMultilevel extends CommandClass {
                public final static int id = 0x31;
                public final Data data;

                public SensorMultilevel(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x31);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(int sensor_type, Object callbackArg) {
                    jni_cc_sensorMultilevelGet(jzway, instance.device.id, instance.id, sensor_type, callbackArg);
                }
                public void get(int sensor_type) {
                    jni_cc_sensorMultilevelGet(jzway, instance.device.id, instance.id, sensor_type, 0);
                }
                
            }
            
            public final class SensorConfiguration extends CommandClass {
                public final static int id = 0x9e;
                public final Data data;

                public SensorConfiguration(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x9e);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_sensorConfigurationGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_sensorConfigurationGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(int mode, float value, Object callbackArg) {
                    jni_cc_sensorConfigurationSet(jzway, instance.device.id, instance.id, mode, value, callbackArg);
                }
                public void set(int mode, float value) {
                    jni_cc_sensorConfigurationSet(jzway, instance.device.id, instance.id, mode, value, 0);
                }
                
            }
            
            public final class SwitchAll extends CommandClass {
                public final static int id = 0x27;
                public final Data data;

                public SwitchAll(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x27);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_switchAllGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_switchAllGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(int mode, Object callbackArg) {
                    jni_cc_switchAllSet(jzway, instance.device.id, instance.id, mode, callbackArg);
                }
                public void set(int mode) {
                    jni_cc_switchAllSet(jzway, instance.device.id, instance.id, mode, 0);
                }
                
                public void setOn(Object callbackArg) {
                    jni_cc_switchAllSetOn(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void setOn() {
                    jni_cc_switchAllSetOn(jzway, instance.device.id, instance.id, 0);
                }
                
                public void setOff(Object callbackArg) {
                    jni_cc_switchAllSetOff(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void setOff() {
                    jni_cc_switchAllSetOff(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class SwitchBinary extends CommandClass {
                public final static int id = 0x25;
                public final Data data;

                public SwitchBinary(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x25);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_switchBinaryGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_switchBinaryGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(boolean value, int duration, Object callbackArg) {
                    jni_cc_switchBinarySet(jzway, instance.device.id, instance.id, value, duration, callbackArg);
                }
                public void set(boolean value, int duration) {
                    jni_cc_switchBinarySet(jzway, instance.device.id, instance.id, value, duration, 0);
                }
                
            }
            
            public final class SwitchMultilevel extends CommandClass {
                public final static int id = 0x26;
                public final Data data;

                public SwitchMultilevel(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x26);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_switchMultilevelGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_switchMultilevelGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(int level, int duration, Object callbackArg) {
                    jni_cc_switchMultilevelSet(jzway, instance.device.id, instance.id, level, duration, callbackArg);
                }
                public void set(int level, int duration) {
                    jni_cc_switchMultilevelSet(jzway, instance.device.id, instance.id, level, duration, 0);
                }
                
                public void startLevelChange(int dir, int duration, boolean ignoreStartLevel, int startLevel, int incdec, int step, Object callbackArg) {
                    jni_cc_switchMultilevelStartLevelChange(jzway, instance.device.id, instance.id, dir, duration, ignoreStartLevel, startLevel, incdec, step, callbackArg);
                }
                public void startLevelChange(int dir, int duration, boolean ignoreStartLevel, int startLevel, int incdec, int step) {
                    jni_cc_switchMultilevelStartLevelChange(jzway, instance.device.id, instance.id, dir, duration, ignoreStartLevel, startLevel, incdec, step, 0);
                }
                
                public void stopLevelChange(Object callbackArg) {
                    jni_cc_switchMultilevelStopLevelChange(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void stopLevelChange() {
                    jni_cc_switchMultilevelStopLevelChange(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class MultiChannelAssociation extends CommandClass {
                public final static int id = 0x8e;
                public final Data data;

                public MultiChannelAssociation(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x8e);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void multichannelAssociationGet(int group_id, Object callbackArg) {
                    jni_cc_multichannelAssociationGet(jzway, instance.device.id, instance.id, group_id, callbackArg);
                }
                public void multichannelAssociationGet(int group_id) {
                    jni_cc_multichannelAssociationGet(jzway, instance.device.id, instance.id, group_id, 0);
                }
                
                public void multichannelAssociationSet(int group_id, int include_node, int include_instance, Object callbackArg) {
                    jni_cc_multichannelAssociationSet(jzway, instance.device.id, instance.id, group_id, include_node, include_instance, callbackArg);
                }
                public void multichannelAssociationSet(int group_id, int include_node, int include_instance) {
                    jni_cc_multichannelAssociationSet(jzway, instance.device.id, instance.id, group_id, include_node, include_instance, 0);
                }
                
                public void multichannelAssociationRemove(int group_id, int exclude_node, int exclude_instance, Object callbackArg) {
                    jni_cc_multichannelAssociationRemove(jzway, instance.device.id, instance.id, group_id, exclude_node, exclude_instance, callbackArg);
                }
                public void multichannelAssociationRemove(int group_id, int exclude_node, int exclude_instance) {
                    jni_cc_multichannelAssociationRemove(jzway, instance.device.id, instance.id, group_id, exclude_node, exclude_instance, 0);
                }
                
                public void multichannelAssociationGroupingsGet(Object callbackArg) {
                    jni_cc_multichannelAssociationGroupingsGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void multichannelAssociationGroupingsGet() {
                    jni_cc_multichannelAssociationGroupingsGet(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class MultiChannel extends CommandClass {
                public final static int id = 0x60;
                public final Data data;

                public MultiChannel(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x60);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void multichannelGet(int cc_id, Object callbackArg) {
                    jni_cc_multichannelGet(jzway, instance.device.id, instance.id, cc_id, callbackArg);
                }
                public void multichannelGet(int cc_id) {
                    jni_cc_multichannelGet(jzway, instance.device.id, instance.id, cc_id, 0);
                }
                
                public void multichannelEndpointFind(int generic, int specific, Object callbackArg) {
                    jni_cc_multichannelEndpointFind(jzway, instance.device.id, instance.id, generic, specific, callbackArg);
                }
                public void multichannelEndpointFind(int generic, int specific) {
                    jni_cc_multichannelEndpointFind(jzway, instance.device.id, instance.id, generic, specific, 0);
                }
                
                public void multichannelEndpointGet(Object callbackArg) {
                    jni_cc_multichannelEndpointGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void multichannelEndpointGet() {
                    jni_cc_multichannelEndpointGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void multichannelCapabilitiesGet(int endpoint, Object callbackArg) {
                    jni_cc_multichannelCapabilitiesGet(jzway, instance.device.id, instance.id, endpoint, callbackArg);
                }
                public void multichannelCapabilitiesGet(int endpoint) {
                    jni_cc_multichannelCapabilitiesGet(jzway, instance.device.id, instance.id, endpoint, 0);
                }
                
                public void multichannelAggregatedMembersGet(int endpoint, Object callbackArg) {
                    jni_cc_multichannelAggregatedMembersGet(jzway, instance.device.id, instance.id, endpoint, callbackArg);
                }
                public void multichannelAggregatedMembersGet(int endpoint) {
                    jni_cc_multichannelAggregatedMembersGet(jzway, instance.device.id, instance.id, endpoint, 0);
                }
                
            }
            
            public final class NodeNaming extends CommandClass {
                public final static int id = 0x77;
                public final Data data;

                public NodeNaming(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x77);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_nodeNamingGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_nodeNamingGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void getName(Object callbackArg) {
                    jni_cc_nodeNamingGetName(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void getName() {
                    jni_cc_nodeNamingGetName(jzway, instance.device.id, instance.id, 0);
                }
                
                public void getLocation(Object callbackArg) {
                    jni_cc_nodeNamingGetLocation(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void getLocation() {
                    jni_cc_nodeNamingGetLocation(jzway, instance.device.id, instance.id, 0);
                }
                
                public void setName(String name, Object callbackArg) {
                    jni_cc_nodeNamingSetName(jzway, instance.device.id, instance.id, name, callbackArg);
                }
                public void setName(String name) {
                    jni_cc_nodeNamingSetName(jzway, instance.device.id, instance.id, name, 0);
                }
                
                public void setLocation(String location, Object callbackArg) {
                    jni_cc_nodeNamingSetLocation(jzway, instance.device.id, instance.id, location, callbackArg);
                }
                public void setLocation(String location) {
                    jni_cc_nodeNamingSetLocation(jzway, instance.device.id, instance.id, location, 0);
                }
                
            }
            
            public final class ThermostatSetPoint extends CommandClass {
                public final static int id = 0x43;
                public final Data data;

                public ThermostatSetPoint(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x43);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void thermostatSetpointGet(int mode, Object callbackArg) {
                    jni_cc_thermostatSetpointGet(jzway, instance.device.id, instance.id, mode, callbackArg);
                }
                public void thermostatSetpointGet(int mode) {
                    jni_cc_thermostatSetpointGet(jzway, instance.device.id, instance.id, mode, 0);
                }
                
                public void thermostatSetpointSet(int mode, float value, Object callbackArg) {
                    jni_cc_thermostatSetpointSet(jzway, instance.device.id, instance.id, mode, value, callbackArg);
                }
                public void thermostatSetpointSet(int mode, float value) {
                    jni_cc_thermostatSetpointSet(jzway, instance.device.id, instance.id, mode, value, 0);
                }
                
            }
            
            public final class ThermostatMode extends CommandClass {
                public final static int id = 0x40;
                public final Data data;

                public ThermostatMode(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x40);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_thermostatModeGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_thermostatModeGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(int mode, Object callbackArg) {
                    jni_cc_thermostatModeSet(jzway, instance.device.id, instance.id, mode, callbackArg);
                }
                public void set(int mode) {
                    jni_cc_thermostatModeSet(jzway, instance.device.id, instance.id, mode, 0);
                }
                
                public void setManufacturerSpecific(int[] manufacturerSpecifc, Object callbackArg) {
                    jni_cc_thermostatModeSetManufacturerSpecific(jzway, instance.device.id, instance.id, manufacturerSpecifc, callbackArg);
                }
                public void setManufacturerSpecific(int[] manufacturerSpecifc) {
                    jni_cc_thermostatModeSetManufacturerSpecific(jzway, instance.device.id, instance.id, manufacturerSpecifc, 0);
                }
                
            }
            
            public final class ThermostatFanMode extends CommandClass {
                public final static int id = 0x44;
                public final Data data;

                public ThermostatFanMode(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x44);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_thermostatFanModeGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_thermostatFanModeGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(boolean on, int mode, Object callbackArg) {
                    jni_cc_thermostatFanModeSet(jzway, instance.device.id, instance.id, on, mode, callbackArg);
                }
                public void set(boolean on, int mode) {
                    jni_cc_thermostatFanModeSet(jzway, instance.device.id, instance.id, on, mode, 0);
                }
                
            }
            
            public final class ThermostatFanState extends CommandClass {
                public final static int id = 0x45;
                public final Data data;

                public ThermostatFanState(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x45);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_thermostatFanStateGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_thermostatFanStateGet(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class ThermostatOperatingState extends CommandClass {
                public final static int id = 0x42;
                public final Data data;

                public ThermostatOperatingState(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x42);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_thermostatOperatingStateGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_thermostatOperatingStateGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void loggingGet(int state, Object callbackArg) {
                    jni_cc_thermostatOperatingStateLoggingGet(jzway, instance.device.id, instance.id, state, callbackArg);
                }
                public void loggingGet(int state) {
                    jni_cc_thermostatOperatingStateLoggingGet(jzway, instance.device.id, instance.id, state, 0);
                }
                
            }
            
            public final class ThermostatSetBack extends CommandClass {
                public final static int id = 0x47;
                public final Data data;

                public ThermostatSetBack(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x47);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void thermostatSetbackGet(Object callbackArg) {
                    jni_cc_thermostatSetbackGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void thermostatSetbackGet() {
                    jni_cc_thermostatSetbackGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void thermostatSetbackSet(int overrideType, int state, float value, Object callbackArg) {
                    jni_cc_thermostatSetbackSet(jzway, instance.device.id, instance.id, overrideType, state, value, callbackArg);
                }
                public void thermostatSetbackSet(int overrideType, int state, float value) {
                    jni_cc_thermostatSetbackSet(jzway, instance.device.id, instance.id, overrideType, state, value, 0);
                }
                
            }
            
            public final class AlarmSensor extends CommandClass {
                public final static int id = 0x9c;
                public final Data data;

                public AlarmSensor(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x9c);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void supportedGet(Object callbackArg) {
                    jni_cc_alarmSensorSupportedGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void supportedGet() {
                    jni_cc_alarmSensorSupportedGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void get(int type, Object callbackArg) {
                    jni_cc_alarmSensorGet(jzway, instance.device.id, instance.id, type, callbackArg);
                }
                public void get(int type) {
                    jni_cc_alarmSensorGet(jzway, instance.device.id, instance.id, type, 0);
                }
                
            }
            
            public final class DoorLock extends CommandClass {
                public final static int id = 0x62;
                public final Data data;

                public DoorLock(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x62);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_doorLockGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_doorLockGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void configurationGet(Object callbackArg) {
                    jni_cc_doorLockConfigurationGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void configurationGet() {
                    jni_cc_doorLockConfigurationGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(int mode, Object callbackArg) {
                    jni_cc_doorLockSet(jzway, instance.device.id, instance.id, mode, callbackArg);
                }
                public void set(int mode) {
                    jni_cc_doorLockSet(jzway, instance.device.id, instance.id, mode, 0);
                }
                
                public void configurationSet(int operationType, int outsideHandlesEnabled, int insideHandlesEnabled, int lockTimeout, int autolockTime, int holdReleaseTime, boolean twistAssist, boolean blockToBlock, Object callbackArg) {
                    jni_cc_doorLockConfigurationSet(jzway, instance.device.id, instance.id, operationType, outsideHandlesEnabled, insideHandlesEnabled, lockTimeout, autolockTime, holdReleaseTime, twistAssist, blockToBlock, callbackArg);
                }
                public void configurationSet(int operationType, int outsideHandlesEnabled, int insideHandlesEnabled, int lockTimeout, int autolockTime, int holdReleaseTime, boolean twistAssist, boolean blockToBlock) {
                    jni_cc_doorLockConfigurationSet(jzway, instance.device.id, instance.id, operationType, outsideHandlesEnabled, insideHandlesEnabled, lockTimeout, autolockTime, holdReleaseTime, twistAssist, blockToBlock, 0);
                }
                
            }
            
            public final class DoorLockLogging extends CommandClass {
                public final static int id = 0x4c;
                public final Data data;

                public DoorLockLogging(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x4c);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(int record, Object callbackArg) {
                    jni_cc_doorLockLoggingGet(jzway, instance.device.id, instance.id, record, callbackArg);
                }
                public void get(int record) {
                    jni_cc_doorLockLoggingGet(jzway, instance.device.id, instance.id, record, 0);
                }
                
            }
            
            public final class UserCode extends CommandClass {
                public final static int id = 0x63;
                public final Data data;

                public UserCode(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x63);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(int user, Object callbackArg) {
                    jni_cc_userCodeGet(jzway, instance.device.id, instance.id, user, callbackArg);
                }
                public void get(int user) {
                    jni_cc_userCodeGet(jzway, instance.device.id, instance.id, user, 0);
                }
                
                public void getAll(Object callbackArg) {
                    jni_cc_userCodeGetAll(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void getAll() {
                    jni_cc_userCodeGetAll(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(int user, String code, int status, Object callbackArg) {
                    jni_cc_userCodeSet(jzway, instance.device.id, instance.id, user, code, status, callbackArg);
                }
                public void set(int user, String code, int status) {
                    jni_cc_userCodeSet(jzway, instance.device.id, instance.id, user, code, status, 0);
                }
                
                public void setRaw(int user, int[] code, int status, Object callbackArg) {
                    jni_cc_userCodeSetRaw(jzway, instance.device.id, instance.id, user, code, status, callbackArg);
                }
                public void setRaw(int user, int[] code, int status) {
                    jni_cc_userCodeSetRaw(jzway, instance.device.id, instance.id, user, code, status, 0);
                }
                
                public void masterCodeGet(Object callbackArg) {
                    jni_cc_userCodeMasterCodeGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void masterCodeGet() {
                    jni_cc_userCodeMasterCodeGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void masterCodeSet(String code, Object callbackArg) {
                    jni_cc_userCodeMasterCodeSet(jzway, instance.device.id, instance.id, code, callbackArg);
                }
                public void masterCodeSet(String code) {
                    jni_cc_userCodeMasterCodeSet(jzway, instance.device.id, instance.id, code, 0);
                }
                
                public void masterCodeSetRaw(int[] code, Object callbackArg) {
                    jni_cc_userCodeMasterCodeSetRaw(jzway, instance.device.id, instance.id, code, callbackArg);
                }
                public void masterCodeSetRaw(int[] code) {
                    jni_cc_userCodeMasterCodeSetRaw(jzway, instance.device.id, instance.id, code, 0);
                }
                
                public void keypadModeGet(Object callbackArg) {
                    jni_cc_userCodeKeypadModeGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void keypadModeGet() {
                    jni_cc_userCodeKeypadModeGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void keypadModeSet(int mode, Object callbackArg) {
                    jni_cc_userCodeKeypadModeSet(jzway, instance.device.id, instance.id, mode, callbackArg);
                }
                public void keypadModeSet(int mode) {
                    jni_cc_userCodeKeypadModeSet(jzway, instance.device.id, instance.id, mode, 0);
                }
                
                public void checksumGet(Object callbackArg) {
                    jni_cc_userCodeChecksumGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void checksumGet() {
                    jni_cc_userCodeChecksumGet(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class Time extends CommandClass {
                public final static int id = 0x8a;
                public final Data data;

                public Time(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x8a);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_timeTimeGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_timeTimeGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void dateGet(Object callbackArg) {
                    jni_cc_timeDateGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void dateGet() {
                    jni_cc_timeDateGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void offsetGet(Object callbackArg) {
                    jni_cc_timeOffsetGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void offsetGet() {
                    jni_cc_timeOffsetGet(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class TimeParameters extends CommandClass {
                public final static int id = 0x8b;
                public final Data data;

                public TimeParameters(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x8b);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_timeParametersGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_timeParametersGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(Object callbackArg) {
                    jni_cc_timeParametersSet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void set() {
                    jni_cc_timeParametersSet(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class Clock extends CommandClass {
                public final static int id = 0x81;
                public final Data data;

                public Clock(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x81);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_clockGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_clockGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(Object callbackArg) {
                    jni_cc_clockSet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void set() {
                    jni_cc_clockSet(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class SceneActivation extends CommandClass {
                public final static int id = 0x2b;
                public final Data data;

                public SceneActivation(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x2b);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void set(int sceneId, int dimmingDuration, Object callbackArg) {
                    jni_cc_sceneActivationSet(jzway, instance.device.id, instance.id, sceneId, dimmingDuration, callbackArg);
                }
                public void set(int sceneId, int dimmingDuration) {
                    jni_cc_sceneActivationSet(jzway, instance.device.id, instance.id, sceneId, dimmingDuration, 0);
                }
                
            }
            
            public final class SceneControllerConf extends CommandClass {
                public final static int id = 0x2d;
                public final Data data;

                public SceneControllerConf(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x2d);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(int group, Object callbackArg) {
                    jni_cc_sceneControllerConfGet(jzway, instance.device.id, instance.id, group, callbackArg);
                }
                public void get(int group) {
                    jni_cc_sceneControllerConfGet(jzway, instance.device.id, instance.id, group, 0);
                }
                
                public void set(int group, int scene, int duration, Object callbackArg) {
                    jni_cc_sceneControllerConfSet(jzway, instance.device.id, instance.id, group, scene, duration, callbackArg);
                }
                public void set(int group, int scene, int duration) {
                    jni_cc_sceneControllerConfSet(jzway, instance.device.id, instance.id, group, scene, duration, 0);
                }
                
            }
            
            public final class SceneActuatorConf extends CommandClass {
                public final static int id = 0x2c;
                public final Data data;

                public SceneActuatorConf(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x2c);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(int scene, Object callbackArg) {
                    jni_cc_sceneActuatorConfGet(jzway, instance.device.id, instance.id, scene, callbackArg);
                }
                public void get(int scene) {
                    jni_cc_sceneActuatorConfGet(jzway, instance.device.id, instance.id, scene, 0);
                }
                
                public void set(int scene, int level, int dimming, boolean override, Object callbackArg) {
                    jni_cc_sceneActuatorConfSet(jzway, instance.device.id, instance.id, scene, level, dimming, override, callbackArg);
                }
                public void set(int scene, int level, int dimming, boolean override) {
                    jni_cc_sceneActuatorConfSet(jzway, instance.device.id, instance.id, scene, level, dimming, override, 0);
                }
                
            }
            
            public final class Indicator extends CommandClass {
                public final static int id = 0x87;
                public final Data data;

                public Indicator(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x87);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(int indicator_id, Object callbackArg) {
                    jni_cc_indicatorGet(jzway, instance.device.id, instance.id, indicator_id, callbackArg);
                }
                public void get(int indicator_id) {
                    jni_cc_indicatorGet(jzway, instance.device.id, instance.id, indicator_id, 0);
                }
                
                public void set(int indicator_id, int property_id, int state, Object callbackArg) {
                    jni_cc_indicatorSet(jzway, instance.device.id, instance.id, indicator_id, property_id, state, callbackArg);
                }
                public void set(int indicator_id, int property_id, int state) {
                    jni_cc_indicatorSet(jzway, instance.device.id, instance.id, indicator_id, property_id, state, 0);
                }
                
                public void setMultiple(int indicator_id, int[] property_ids, int[] states, Object callbackArg) {
                    jni_cc_indicatorSetMultiple(jzway, instance.device.id, instance.id, indicator_id, property_ids, states, callbackArg);
                }
                public void setMultiple(int indicator_id, int[] property_ids, int[] states) {
                    jni_cc_indicatorSetMultiple(jzway, instance.device.id, instance.id, indicator_id, property_ids, states, 0);
                }
                
                public void identify(Object callbackArg) {
                    jni_cc_indicatorIdentify(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void identify() {
                    jni_cc_indicatorIdentify(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class Protection extends CommandClass {
                public final static int id = 0x75;
                public final Data data;

                public Protection(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x75);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_protectionGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_protectionGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(int state, int rfState, Object callbackArg) {
                    jni_cc_protectionSet(jzway, instance.device.id, instance.id, state, rfState, callbackArg);
                }
                public void set(int state, int rfState) {
                    jni_cc_protectionSet(jzway, instance.device.id, instance.id, state, rfState, 0);
                }
                
                public void exclusiveGet(Object callbackArg) {
                    jni_cc_protectionExclusiveGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void exclusiveGet() {
                    jni_cc_protectionExclusiveGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void exclusiveSet(int controlNodeId, Object callbackArg) {
                    jni_cc_protectionExclusiveSet(jzway, instance.device.id, instance.id, controlNodeId, callbackArg);
                }
                public void exclusiveSet(int controlNodeId) {
                    jni_cc_protectionExclusiveSet(jzway, instance.device.id, instance.id, controlNodeId, 0);
                }
                
                public void timeoutGet(Object callbackArg) {
                    jni_cc_protectionTimeoutGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void timeoutGet() {
                    jni_cc_protectionTimeoutGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void timeoutSet(int timeout, Object callbackArg) {
                    jni_cc_protectionTimeoutSet(jzway, instance.device.id, instance.id, timeout, callbackArg);
                }
                public void timeoutSet(int timeout) {
                    jni_cc_protectionTimeoutSet(jzway, instance.device.id, instance.id, timeout, 0);
                }
                
            }
            
            public final class ScheduleEntryLock extends CommandClass {
                public final static int id = 0x4e;
                public final Data data;

                public ScheduleEntryLock(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x4e);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void enable(int user, boolean enable, Object callbackArg) {
                    jni_cc_scheduleEntryLockEnable(jzway, instance.device.id, instance.id, user, enable, callbackArg);
                }
                public void enable(int user, boolean enable) {
                    jni_cc_scheduleEntryLockEnable(jzway, instance.device.id, instance.id, user, enable, 0);
                }
                
                public void weekdayGet(int user, int slot, Object callbackArg) {
                    jni_cc_scheduleEntryLockWeekdayGet(jzway, instance.device.id, instance.id, user, slot, callbackArg);
                }
                public void weekdayGet(int user, int slot) {
                    jni_cc_scheduleEntryLockWeekdayGet(jzway, instance.device.id, instance.id, user, slot, 0);
                }
                
                public void weekdaySet(boolean modifyErase, int user, int slot, int dayOfWeek, int startHour, int startMinute, int stopHour, int stopMinute, Object callbackArg) {
                    jni_cc_scheduleEntryLockWeekdaySet(jzway, instance.device.id, instance.id, modifyErase, user, slot, dayOfWeek, startHour, startMinute, stopHour, stopMinute, callbackArg);
                }
                public void weekdaySet(boolean modifyErase, int user, int slot, int dayOfWeek, int startHour, int startMinute, int stopHour, int stopMinute) {
                    jni_cc_scheduleEntryLockWeekdaySet(jzway, instance.device.id, instance.id, modifyErase, user, slot, dayOfWeek, startHour, startMinute, stopHour, stopMinute, 0);
                }
                
                public void yearGet(int user, int slot, Object callbackArg) {
                    jni_cc_scheduleEntryLockYearGet(jzway, instance.device.id, instance.id, user, slot, callbackArg);
                }
                public void yearGet(int user, int slot) {
                    jni_cc_scheduleEntryLockYearGet(jzway, instance.device.id, instance.id, user, slot, 0);
                }
                
                public void yearSet(boolean modifyErase, int user, int slot, int startYear, int startMonth, int startDay, int startHour, int startMinute, int stopYear, int stopMonth, int stopDay, int stopHour, int stopMinute, Object callbackArg) {
                    jni_cc_scheduleEntryLockYearSet(jzway, instance.device.id, instance.id, modifyErase, user, slot, startYear, startMonth, startDay, startHour, startMinute, stopYear, stopMonth, stopDay, stopHour, stopMinute, callbackArg);
                }
                public void yearSet(boolean modifyErase, int user, int slot, int startYear, int startMonth, int startDay, int startHour, int startMinute, int stopYear, int stopMonth, int stopDay, int stopHour, int stopMinute) {
                    jni_cc_scheduleEntryLockYearSet(jzway, instance.device.id, instance.id, modifyErase, user, slot, startYear, startMonth, startDay, startHour, startMinute, stopYear, stopMonth, stopDay, stopHour, stopMinute, 0);
                }
                
            }
            
            public final class ClimateControlSchedule extends CommandClass {
                public final static int id = 0x46;
                public final Data data;

                public ClimateControlSchedule(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x46);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void overrideGet(Object callbackArg) {
                    jni_cc_climateControlScheduleOverrideGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void overrideGet() {
                    jni_cc_climateControlScheduleOverrideGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void overrideSet(int overrideType, int overrideState, Object callbackArg) {
                    jni_cc_climateControlScheduleOverrideSet(jzway, instance.device.id, instance.id, overrideType, overrideState, callbackArg);
                }
                public void overrideSet(int overrideType, int overrideState) {
                    jni_cc_climateControlScheduleOverrideSet(jzway, instance.device.id, instance.id, overrideType, overrideState, 0);
                }
                
            }
            
            public final class MeterTableMonitor extends CommandClass {
                public final static int id = 0x3d;
                public final Data data;

                public MeterTableMonitor(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x3d);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void statusDateGet(int maxResults, long startDate, long endDate, Object callbackArg) {
                    jni_cc_meterTableMonitorStatusDateGet(jzway, instance.device.id, instance.id, maxResults, startDate, endDate, callbackArg);
                }
                public void statusDateGet(int maxResults, long startDate, long endDate) {
                    jni_cc_meterTableMonitorStatusDateGet(jzway, instance.device.id, instance.id, maxResults, startDate, endDate, 0);
                }
                
                public void statusDepthGet(int maxResults, Object callbackArg) {
                    jni_cc_meterTableMonitorStatusDepthGet(jzway, instance.device.id, instance.id, maxResults, callbackArg);
                }
                public void statusDepthGet(int maxResults) {
                    jni_cc_meterTableMonitorStatusDepthGet(jzway, instance.device.id, instance.id, maxResults, 0);
                }
                
                public void currentDataGet(int setId, Object callbackArg) {
                    jni_cc_meterTableMonitorCurrentDataGet(jzway, instance.device.id, instance.id, setId, callbackArg);
                }
                public void currentDataGet(int setId) {
                    jni_cc_meterTableMonitorCurrentDataGet(jzway, instance.device.id, instance.id, setId, 0);
                }
                
                public void historicalDataGet(int setId, int maxResults, long startDate, long endDate, Object callbackArg) {
                    jni_cc_meterTableMonitorHistoricalDataGet(jzway, instance.device.id, instance.id, setId, maxResults, startDate, endDate, callbackArg);
                }
                public void historicalDataGet(int setId, int maxResults, long startDate, long endDate) {
                    jni_cc_meterTableMonitorHistoricalDataGet(jzway, instance.device.id, instance.id, setId, maxResults, startDate, endDate, 0);
                }
                
            }
            
            public final class Alarm extends CommandClass {
                public final static int id = 0x71;
                public final Data data;

                public Alarm(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x71);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(int type, int event, Object callbackArg) {
                    jni_cc_alarmGet(jzway, instance.device.id, instance.id, type, event, callbackArg);
                }
                public void get(int type, int event) {
                    jni_cc_alarmGet(jzway, instance.device.id, instance.id, type, event, 0);
                }
                
                public void set(int type, int level, Object callbackArg) {
                    jni_cc_alarmSet(jzway, instance.device.id, instance.id, type, level, callbackArg);
                }
                public void set(int type, int level) {
                    jni_cc_alarmSet(jzway, instance.device.id, instance.id, type, level, 0);
                }
                
            }
            
            public final class PowerLevel extends CommandClass {
                public final static int id = 0x73;
                public final Data data;

                public PowerLevel(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x73);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_powerLevelGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_powerLevelGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(int level, int timeout, Object callbackArg) {
                    jni_cc_powerLevelSet(jzway, instance.device.id, instance.id, level, timeout, callbackArg);
                }
                public void set(int level, int timeout) {
                    jni_cc_powerLevelSet(jzway, instance.device.id, instance.id, level, timeout, 0);
                }
                
                public void testNodeGet(Object callbackArg) {
                    jni_cc_powerLevelTestNodeGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void testNodeGet() {
                    jni_cc_powerLevelTestNodeGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void testNodeSet(int testNodeId, int level, int frameCount, Object callbackArg) {
                    jni_cc_powerLevelTestNodeSet(jzway, instance.device.id, instance.id, testNodeId, level, frameCount, callbackArg);
                }
                public void testNodeSet(int testNodeId, int level, int frameCount) {
                    jni_cc_powerLevelTestNodeSet(jzway, instance.device.id, instance.id, testNodeId, level, frameCount, 0);
                }
                
            }
            
            public final class ZWavePlusInfo extends CommandClass {
                public final static int id = 0x5e;
                public final Data data;

                public ZWavePlusInfo(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x5e);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void zwavePlusInfoGet(Object callbackArg) {
                    jni_cc_zwavePlusInfoGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void zwavePlusInfoGet() {
                    jni_cc_zwavePlusInfoGet(jzway, instance.device.id, instance.id, 0);
                }
                
            }
            
            public final class FirmwareUpdate extends CommandClass {
                public final static int id = 0x7a;
                public final Data data;

                public FirmwareUpdate(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x7a);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_firmwareUpdateGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_firmwareUpdateGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void perform(int manufacturerId, int firmwareId, int firmwareTarget, int[] data, boolean delayActivation, Object callbackArg) {
                    jni_cc_firmwareUpdatePerform(jzway, instance.device.id, instance.id, manufacturerId, firmwareId, firmwareTarget, data, delayActivation, callbackArg);
                }
                public void perform(int manufacturerId, int firmwareId, int firmwareTarget, int[] data, boolean delayActivation) {
                    jni_cc_firmwareUpdatePerform(jzway, instance.device.id, instance.id, manufacturerId, firmwareId, firmwareTarget, data, delayActivation, 0);
                }
                
                public void activation(int manufacturerId, int firmwareId, int firmwareTarget, int[] data, Object callbackArg) {
                    jni_cc_firmwareUpdateActivation(jzway, instance.device.id, instance.id, manufacturerId, firmwareId, firmwareTarget, data, callbackArg);
                }
                public void activation(int manufacturerId, int firmwareId, int firmwareTarget, int[] data) {
                    jni_cc_firmwareUpdateActivation(jzway, instance.device.id, instance.id, manufacturerId, firmwareId, firmwareTarget, data, 0);
                }
                
                public void prepare(int manufacturerId, int firmwareId, int firmwareTarget, Object callbackArg) {
                    jni_cc_firmwareUpdatePrepare(jzway, instance.device.id, instance.id, manufacturerId, firmwareId, firmwareTarget, callbackArg);
                }
                public void prepare(int manufacturerId, int firmwareId, int firmwareTarget) {
                    jni_cc_firmwareUpdatePrepare(jzway, instance.device.id, instance.id, manufacturerId, firmwareId, firmwareTarget, 0);
                }
                
            }
            
            public final class AssociationGroupInformation extends CommandClass {
                public final static int id = 0x59;
                public final Data data;

                public AssociationGroupInformation(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x59);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void getInfo(int groupId, Object callbackArg) {
                    jni_cc_associationGroupInformationGetInfo(jzway, instance.device.id, instance.id, groupId, callbackArg);
                }
                public void getInfo(int groupId) {
                    jni_cc_associationGroupInformationGetInfo(jzway, instance.device.id, instance.id, groupId, 0);
                }
                
                public void getName(int groupId, Object callbackArg) {
                    jni_cc_associationGroupInformationGetName(jzway, instance.device.id, instance.id, groupId, callbackArg);
                }
                public void getName(int groupId) {
                    jni_cc_associationGroupInformationGetName(jzway, instance.device.id, instance.id, groupId, 0);
                }
                
                public void getCommands(int groupId, Object callbackArg) {
                    jni_cc_associationGroupInformationGetCommands(jzway, instance.device.id, instance.id, groupId, callbackArg);
                }
                public void getCommands(int groupId) {
                    jni_cc_associationGroupInformationGetCommands(jzway, instance.device.id, instance.id, groupId, 0);
                }
                
            }
            
            public final class SwitchColor extends CommandClass {
                public final static int id = 0x33;
                public final Data data;

                public SwitchColor(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x33);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(int capabilityId, Object callbackArg) {
                    jni_cc_switchColorGet(jzway, instance.device.id, instance.id, capabilityId, callbackArg);
                }
                public void get(int capabilityId) {
                    jni_cc_switchColorGet(jzway, instance.device.id, instance.id, capabilityId, 0);
                }
                
                public void set(int capabilityId, int state, int duration, Object callbackArg) {
                    jni_cc_switchColorSet(jzway, instance.device.id, instance.id, capabilityId, state, duration, callbackArg);
                }
                public void set(int capabilityId, int state, int duration) {
                    jni_cc_switchColorSet(jzway, instance.device.id, instance.id, capabilityId, state, duration, 0);
                }
                
                public void setMultiple(int[] capabilityIds, int[] states, int duration, Object callbackArg) {
                    jni_cc_switchColorSetMultiple(jzway, instance.device.id, instance.id, capabilityIds, states, duration, callbackArg);
                }
                public void setMultiple(int[] capabilityIds, int[] states, int duration) {
                    jni_cc_switchColorSetMultiple(jzway, instance.device.id, instance.id, capabilityIds, states, duration, 0);
                }
                
                public void startStateChange(int capabilityId, int dir, int duration, boolean ignoreStartLevel, int startLevel, Object callbackArg) {
                    jni_cc_switchColorStartStateChange(jzway, instance.device.id, instance.id, capabilityId, dir, duration, ignoreStartLevel, startLevel, callbackArg);
                }
                public void startStateChange(int capabilityId, int dir, int duration, boolean ignoreStartLevel, int startLevel) {
                    jni_cc_switchColorStartStateChange(jzway, instance.device.id, instance.id, capabilityId, dir, duration, ignoreStartLevel, startLevel, 0);
                }
                
                public void stopStateChange(int capabilityId, Object callbackArg) {
                    jni_cc_switchColorStopStateChange(jzway, instance.device.id, instance.id, capabilityId, callbackArg);
                }
                public void stopStateChange(int capabilityId) {
                    jni_cc_switchColorStopStateChange(jzway, instance.device.id, instance.id, capabilityId, 0);
                }
                
            }
            
            public final class SoundSwitch extends CommandClass {
                public final static int id = 0x79;
                public final Data data;

                public SoundSwitch(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x79);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void tonePlayGet(Object callbackArg) {
                    jni_cc_soundSwitchTonePlayGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void tonePlayGet() {
                    jni_cc_soundSwitchTonePlayGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void tonePlaySet(int toneId, int volume, Object callbackArg) {
                    jni_cc_soundSwitchTonePlaySet(jzway, instance.device.id, instance.id, toneId, volume, callbackArg);
                }
                public void tonePlaySet(int toneId, int volume) {
                    jni_cc_soundSwitchTonePlaySet(jzway, instance.device.id, instance.id, toneId, volume, 0);
                }
                
                public void configurationGet(Object callbackArg) {
                    jni_cc_soundSwitchConfigurationGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void configurationGet() {
                    jni_cc_soundSwitchConfigurationGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void configurationSet(int toneId, int volume, Object callbackArg) {
                    jni_cc_soundSwitchConfigurationSet(jzway, instance.device.id, instance.id, toneId, volume, callbackArg);
                }
                public void configurationSet(int toneId, int volume) {
                    jni_cc_soundSwitchConfigurationSet(jzway, instance.device.id, instance.id, toneId, volume, 0);
                }
                
            }
            
            public final class BarrierOperator extends CommandClass {
                public final static int id = 0x66;
                public final Data data;

                public BarrierOperator(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x66);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void get(Object callbackArg) {
                    jni_cc_barrierOperatorGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void get() {
                    jni_cc_barrierOperatorGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void set(int state, Object callbackArg) {
                    jni_cc_barrierOperatorSet(jzway, instance.device.id, instance.id, state, callbackArg);
                }
                public void set(int state) {
                    jni_cc_barrierOperatorSet(jzway, instance.device.id, instance.id, state, 0);
                }
                
                public void signalGet(int signalType, Object callbackArg) {
                    jni_cc_barrierOperatorSignalGet(jzway, instance.device.id, instance.id, signalType, callbackArg);
                }
                public void signalGet(int signalType) {
                    jni_cc_barrierOperatorSignalGet(jzway, instance.device.id, instance.id, signalType, 0);
                }
                
                public void signalSet(int signalType, int state, Object callbackArg) {
                    jni_cc_barrierOperatorSignalSet(jzway, instance.device.id, instance.id, signalType, state, callbackArg);
                }
                public void signalSet(int signalType, int state) {
                    jni_cc_barrierOperatorSignalSet(jzway, instance.device.id, instance.id, signalType, state, 0);
                }
                
            }
            
            public final class SimpleAVControl extends CommandClass {
                public final static int id = 0x94;
                public final Data data;

                public SimpleAVControl(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x94);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void simpleAvControlSet(int keyAttribute, int avCommand, Object callbackArg) {
                    jni_cc_simpleAvControlSet(jzway, instance.device.id, instance.id, keyAttribute, avCommand, callbackArg);
                }
                public void simpleAvControlSet(int keyAttribute, int avCommand) {
                    jni_cc_simpleAvControlSet(jzway, instance.device.id, instance.id, keyAttribute, avCommand, 0);
                }
                
            }
            
            public final class Security extends CommandClass {
                public final static int id = 0x98;
                public final Data data;

                public Security(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x98);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void inject(int[] data, Object callbackArg) {
                    jni_cc_securityInject(jzway, instance.device.id, instance.id, data, callbackArg);
                }
                public void inject(int[] data) {
                    jni_cc_securityInject(jzway, instance.device.id, instance.id, data, 0);
                }
                
            }
            
            public final class SecurityS2 extends CommandClass {
                public final static int id = 0x9f;
                public final Data data;

                public SecurityS2(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x9f);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void inject(int[] data, int keyClass, Object callbackArg) {
                    jni_cc_securityS2Inject(jzway, instance.device.id, instance.id, data, keyClass, callbackArg);
                }
                public void inject(int[] data, int keyClass) {
                    jni_cc_securityS2Inject(jzway, instance.device.id, instance.id, data, keyClass, 0);
                }
                
            }
            
            public final class EntryControl extends CommandClass {
                public final static int id = 0x6f;
                public final Data data;

                public EntryControl(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x6f);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

                public void configurationGet(Object callbackArg) {
                    jni_cc_entryControlConfigurationGet(jzway, instance.device.id, instance.id, callbackArg);
                }
                public void configurationGet() {
                    jni_cc_entryControlConfigurationGet(jzway, instance.device.id, instance.id, 0);
                }
                
                public void configurationSet(int keyCacheSize, int keyCacheTimeout, Object callbackArg) {
                    jni_cc_entryControlConfigurationSet(jzway, instance.device.id, instance.id, keyCacheSize, keyCacheTimeout, callbackArg);
                }
                public void configurationSet(int keyCacheSize, int keyCacheTimeout) {
                    jni_cc_entryControlConfigurationSet(jzway, instance.device.id, instance.id, keyCacheSize, keyCacheTimeout, 0);
                }
                
            }
            
            public final class Hail extends CommandClass {
                public final static int id = 0x82;
                public final Data data;

                public Hail(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x82);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

            }
            
            public final class CRC16 extends CommandClass {
                public final static int id = 0x56;
                public final Data data;

                public CRC16(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x56);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

            }
            
            public final class MultiCmd extends CommandClass {
                public final static int id = 0x8f;
                public final Data data;

                public MultiCmd(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x8f);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

            }
            
            public final class Supervision extends CommandClass {
                public final static int id = 0x6c;
                public final Data data;

                public Supervision(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x6c);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

            }
            
            public final class ApplicationStatus extends CommandClass {
                public final static int id = 0x22;
                public final Data data;

                public ApplicationStatus(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x22);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

            }
            
            public final class Version extends CommandClass {
                public final static int id = 0x86;
                public final Data data;

                public Version(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x86);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

            }
            
            public final class DeviceResetLocally extends CommandClass {
                public final static int id = 0x5a;
                public final Data data;

                public DeviceResetLocally(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x5a);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

            }
            
            public final class CentralScene extends CommandClass {
                public final static int id = 0x5b;
                public final Data data;

                public CentralScene(ZWay zway, Instance instance) throws Exception {
                    super(zway, instance);
                    try {
                        data = new Data("", instance.device.id, instance.id, 0x5b);
                    } catch (Data.NotFound e) {
                        // should never happen
                        throw new RuntimeException();
                    }
                }

            }
            
             /* END AUTOGENERATED CC CODE */
        }
    }
}
