/*
 * Decompiled with CFR 0.152.
 */
package mil.nga.mgrs;

import java.text.ParseException;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import mil.nga.mgrs.MGRSUtils;
import mil.nga.mgrs.features.Bounds;
import mil.nga.mgrs.features.Line;
import mil.nga.mgrs.features.Point;
import mil.nga.mgrs.grid.GridType;
import mil.nga.mgrs.gzd.GridZone;
import mil.nga.mgrs.gzd.GridZones;
import mil.nga.mgrs.utm.Hemisphere;
import mil.nga.mgrs.utm.UTM;

public class MGRS {
    private static final String[] columnLetters = new String[]{"ABCDEFGH", "JKLMNPQR", "STUVWXYZ"};
    private static final String[] rowLetters = new String[]{"ABCDEFGHJKLMNPQRSTUV", "FGHJKLMNPQRSTUVABCDE"};
    private static final Pattern mgrsPattern = Pattern.compile("^(\\d{1,2})([C-HJ-NP-X])(?:([A-HJ-NP-Z][A-HJ-NP-V])((\\d{2}){0,5}))?$", 2);
    private static final Pattern mgrsInvalidPattern = Pattern.compile("^3[246]X.*$", 2);
    private int zone;
    private char band;
    private char column;
    private char row;
    private long easting;
    private long northing;

    public static MGRS create(int zone, char band, char column, char row, long easting, long northing) {
        return new MGRS(zone, band, column, row, easting, northing);
    }

    public static MGRS create(int zone, char band, long easting, long northing) {
        return new MGRS(zone, band, easting, northing);
    }

    public MGRS(int zone, char band, char column, char row, long easting, long northing) {
        this.zone = zone;
        this.band = band;
        this.column = column;
        this.row = row;
        this.easting = easting;
        this.northing = northing;
    }

    public MGRS(int zone, char band, long easting, long northing) {
        this(zone, band, MGRS.getColumnLetter(zone, easting), MGRS.getRowLetter(zone, northing), easting, northing);
    }

    public int getZone() {
        return this.zone;
    }

    public char getBand() {
        return this.band;
    }

    public char getColumn() {
        return this.column;
    }

    public char getRow() {
        return this.row;
    }

    public long getEasting() {
        return this.easting;
    }

    public long getNorthing() {
        return this.northing;
    }

    public Hemisphere getHemisphere() {
        return Hemisphere.fromBandLetter(this.band);
    }

    public String coordinate() {
        return this.coordinate(GridType.METER);
    }

    public String coordinate(GridType type) {
        StringBuilder mgrs = new StringBuilder();
        if (type != null) {
            mgrs.append(this.zone);
            mgrs.append(this.band);
            if (type != GridType.GZD) {
                mgrs.append(this.column);
                mgrs.append(this.row);
                if (type != GridType.HUNDRED_KILOMETER) {
                    mgrs.append(this.getEastingAndNorthing(type));
                }
            }
        }
        return mgrs.toString();
    }

    public String getEastingAndNorthing(GridType type) {
        int accuracy = 5 - (int)Math.log10(type.getPrecision());
        String easting = String.format(Locale.getDefault(), "%05d", this.easting);
        String northing = String.format(Locale.getDefault(), "%05d", this.northing);
        return easting.substring(0, accuracy) + northing.substring(0, accuracy);
    }

    public String coordinate(int accuracy) {
        return this.coordinate(GridType.withAccuracy(accuracy));
    }

    public GridType precision() {
        return GridType.withAccuracy(this.accuracy());
    }

    public int accuracy() {
        int accuracy = 5;
        for (int accuracyLevel = 10; accuracyLevel <= 100000 && this.easting % (long)accuracyLevel == 0L && this.northing % (long)accuracyLevel == 0L; accuracyLevel *= 10) {
            --accuracy;
        }
        return accuracy;
    }

    public String getColumnRowId() {
        return String.valueOf(this.column) + this.row;
    }

    public GridZone getGridZone() {
        return GridZones.getGridZone(this);
    }

    public Point toPoint() {
        return Point.from(this);
    }

    public UTM toUTM() {
        double easting = this.getUTMEasting();
        double northing = this.getUTMNorthing();
        Hemisphere hemisphere = this.getHemisphere();
        return UTM.create(this.zone, hemisphere, easting, northing);
    }

    public double getUTMEasting() {
        String columnLetters = MGRS.getColumnLetters(this.zone);
        int columnIndex = columnLetters.indexOf(this.column) + 1;
        double e100kNum = (double)columnIndex * 100000.0;
        return e100kNum + (double)this.easting;
    }

    public double getUTMNorthing() {
        String rowLetters = MGRS.getRowLetters(this.zone);
        int rowIndex = rowLetters.indexOf(this.row);
        double n100kNum = (double)rowIndex * 100000.0;
        double latBand = GridZones.getSouthLatitude(this.band);
        double latBandNorthing = Point.degrees(0.0, latBand).toUTM().getNorthing();
        double nBand = Math.floor(latBandNorthing / 100000.0) * 100000.0;
        double n2M = 0.0;
        while (n2M + n100kNum + (double)this.northing < nBand) {
            n2M += 2000000.0;
        }
        return n2M + n100kNum + (double)this.northing;
    }

    public String toString() {
        return this.coordinate();
    }

    public static boolean isMGRS(String mgrs) {
        return mgrsPattern.matcher(mgrs = MGRS.removeSpaces(mgrs)).matches() && !mgrsInvalidPattern.matcher(mgrs).matches();
    }

    private static String removeSpaces(String value) {
        return value.replaceAll("\\s", "");
    }

    public static MGRS from(Point latLng) {
        latLng = latLng.toDegrees();
        UTM utm = latLng.toUTM();
        char bandLetter = latLng.getBandLetter();
        char columnLetter = MGRS.getColumnLetter(utm);
        char rowLetter = MGRS.getRowLetter(utm);
        long easting = (long)(utm.getEasting() % 100000.0);
        long northing = (long)(utm.getNorthing() % 100000.0);
        return MGRS.create(utm.getZone(), bandLetter, columnLetter, rowLetter, easting, northing);
    }

    public static MGRS parse(String mgrs) throws ParseException {
        char band;
        Matcher matcher = mgrsPattern.matcher(MGRS.removeSpaces(mgrs));
        if (!matcher.matches()) {
            throw new ParseException("Invalid MGRS: " + mgrs, 0);
        }
        int zone = Integer.parseInt(matcher.group(1));
        GridZone gridZone = GridZones.getGridZone(zone, band = matcher.group(2).toUpperCase().charAt(0));
        if (gridZone == null) {
            throw new ParseException("Invalid MGRS: " + mgrs, 0);
        }
        MGRS mgrsValue = null;
        String columnRow = matcher.group(3);
        if (columnRow != null) {
            columnRow = columnRow.toUpperCase();
            char column = columnRow.charAt(0);
            char row = columnRow.charAt(1);
            long easting = 0L;
            long northing = 0L;
            String location = matcher.group(4);
            if (!location.isEmpty()) {
                int precision = location.length() / 2;
                double multiplier = Math.pow(10.0, 5 - precision);
                easting = (long)(Double.parseDouble(location.substring(0, precision)) * multiplier);
                northing = (long)(Double.parseDouble(location.substring(precision)) * multiplier);
            }
            mgrsValue = MGRS.create(zone, band, column, row, easting, northing);
            if (location.isEmpty()) {
                boolean southBounds;
                Point point = mgrsValue.toPoint().toDegrees();
                Bounds gridBounds = gridZone.getBounds();
                Point gridSouthwest = gridBounds.getSouthwest().toDegrees();
                boolean westBounds = point.getLongitude() < gridSouthwest.getLongitude();
                boolean bl = southBounds = point.getLatitude() < gridSouthwest.getLatitude();
                if (westBounds || southBounds) {
                    Point north;
                    if (westBounds && southBounds) {
                        Point northeast = MGRS.create(zone, band, column, row, GridType.HUNDRED_KILOMETER.getPrecision(), GridType.HUNDRED_KILOMETER.getPrecision()).toPoint();
                        if (gridBounds.contains(northeast)) {
                            mgrsValue = Point.degrees(gridSouthwest.getLongitude(), gridSouthwest.getLatitude()).toMGRS();
                        }
                    } else if (westBounds) {
                        Point east = MGRS.create(zone, band, column, row, GridType.HUNDRED_KILOMETER.getPrecision(), northing).toPoint();
                        if (gridBounds.contains(east)) {
                            Point intersection = MGRS.getWesternBoundsPoint(gridZone, point, east);
                            mgrsValue = intersection.toMGRS();
                        }
                    } else if (southBounds && gridBounds.contains(north = MGRS.create(zone, band, column, row, easting, GridType.HUNDRED_KILOMETER.getPrecision()).toPoint())) {
                        Point intersection = MGRS.getSouthernBoundsPoint(gridZone, point, north);
                        mgrsValue = intersection.toMGRS();
                    }
                }
            }
        } else {
            mgrsValue = gridZone.getBounds().getSouthwest().toMGRS();
        }
        return mgrsValue;
    }

    private static Point getWesternBoundsPoint(GridZone gridZone, Point west, Point east) {
        UTM eastUTM = east.toUTM();
        double northing = eastUTM.getNorthing();
        int zoneNumber = gridZone.getNumber();
        Hemisphere hemisphere = gridZone.getHemisphere();
        Line line = Line.line(west, east);
        Line boundsLine = gridZone.getBounds().getWestLine();
        Point intersection = MGRSUtils.intersection(line, boundsLine);
        UTM intersectionUTM = UTM.from(intersection, zoneNumber, hemisphere);
        double intersectionEasting = intersectionUTM.getEasting();
        double boundsEasting = Math.ceil(intersectionEasting);
        Point boundsPoint = Point.create(zoneNumber, hemisphere, boundsEasting, northing);
        boundsPoint.setLongitude(boundsLine.getPoint1().getLongitude());
        return boundsPoint;
    }

    private static Point getSouthernBoundsPoint(GridZone gridZone, Point south, Point north) {
        UTM northUTM = north.toUTM();
        double easting = northUTM.getEasting();
        int zoneNumber = gridZone.getNumber();
        Hemisphere hemisphere = gridZone.getHemisphere();
        Line line = Line.line(south, north);
        Line boundsLine = gridZone.getBounds().getSouthLine();
        Point intersection = MGRSUtils.intersection(line, boundsLine);
        UTM intersectionUTM = UTM.from(intersection, zoneNumber, hemisphere);
        double intersectionNorthing = intersectionUTM.getNorthing();
        double boundsNorthing = Math.ceil(intersectionNorthing);
        Point boundsPoint = Point.create(zoneNumber, hemisphere, easting, boundsNorthing);
        boundsPoint.setLatitude(boundsLine.getPoint1().getLatitude());
        return boundsPoint;
    }

    public static GridType precision(String mgrs) throws ParseException {
        String location;
        Matcher matcher = mgrsPattern.matcher(MGRS.removeSpaces(mgrs));
        if (!matcher.matches()) {
            throw new ParseException("Invalid MGRS: " + mgrs, 0);
        }
        GridType precision = null;
        precision = matcher.group(3) != null ? (!(location = matcher.group(4)).isEmpty() ? GridType.withAccuracy(location.length() / 2) : GridType.HUNDRED_KILOMETER) : GridType.GZD;
        return precision;
    }

    public static int accuracy(String mgrs) throws ParseException {
        return MGRS.precision(mgrs).getAccuracy();
    }

    public static String getColumnRowId(double easting, double northing, int zoneNumber) {
        char columnLetter = MGRS.getColumnLetter(zoneNumber, easting);
        char rowLetter = MGRS.getRowLetter(zoneNumber, northing);
        return String.valueOf(columnLetter) + rowLetter;
    }

    public static char getColumnLetter(UTM utm) {
        return MGRS.getColumnLetter(utm.getZone(), utm.getEasting());
    }

    public static char getColumnLetter(int zoneNumber, double easting) {
        int column = (int)Math.floor(easting / 100000.0);
        String columnLetters = MGRS.getColumnLetters(zoneNumber);
        return columnLetters.charAt(column - 1);
    }

    public static char getRowLetter(UTM utm) {
        return MGRS.getRowLetter(utm.getZone(), utm.getNorthing());
    }

    public static char getRowLetter(int zoneNumber, double northing) {
        int row = (int)Math.floor(northing / 100000.0) % 20;
        String rowLetters = MGRS.getRowLetters(zoneNumber);
        return rowLetters.charAt(row);
    }

    private static String getColumnLetters(int zoneNumber) {
        return columnLetters[(zoneNumber - 1) % 3];
    }

    private static String getRowLetters(int zoneNumber) {
        return rowLetters[(zoneNumber - 1) % 2];
    }
}

