package dc.android.libs.swipe;

import android.content.Context;
import android.content.res.TypedArray;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewConfiguration;
import android.view.ViewGroup;
import android.widget.Scroller;
import android.widget.TextView;
import dc.android.libs.swipe.indicator.SwipeIndicator;
import dc.android.libs.swipe.listener.ISwipeListener;
import dc.android.libs.swipe.listener.ISwipeUIListener;
import dc.common.Logger;

/**
 * 自定义下拉刷新控件
 * 拆分两块，上面header+下面content/container
 *
 * @author senrsl
 * @ClassName: SwipeFrameLayout
 * @Package: dc.android.libs.swipe
 * @CreateTime: 2019/8/16 4:23 PM
 */
public class SwipeFrameLayout extends ViewGroup {

    // status enum
    public final static byte SWIPE_STATUS_INIT = 1;
    private byte mStatus = SWIPE_STATUS_INIT;
    public final static byte SWIPE_STATUS_PREPARE = 2;
    public final static byte SWIPE_STATUS_LOADING = 3;
    public final static byte SWIPE_STATUS_COMPLETE = 4;
    private static final boolean DEBUG_LAYOUT = true;
    public static boolean DEBUG = SwipeContext.isDebugSwipe;
    private static int ID = 1;
    // auto refresh status
    private final static byte FLAG_AUTO_REFRESH_AT_ONCE = 0x01;
    private final static byte FLAG_AUTO_REFRESH_BUT_LATER = 0x01 << 1;
    private final static byte FLAG_ENABLE_NEXT_SWIPE_AT_ONCE = 0x01 << 2;
    private final static byte FLAG_PIN_CONTENT = 0x01 << 3;
    private final static byte MASK_AUTO_REFRESH = 0x03;
    protected View mContent;
    // optional config for define header and content in xml file
    private int mHeaderId = 0;
    private int mContainerId = 0;
    // config
    private int mDurationToClose = 200;
    private int mDurationToCloseHeader = 1000;
    private boolean mKeepHeaderWhenRefresh = true;
    private boolean mPullToRefresh = false;
    private View mHeaderView;
    private SwipeUIListenerImpl mSwipeUIListenerImpl = SwipeUIListenerImpl.create();
    private ISwipeListener listener;
    // working parameters
    private SwipeFrameLayout.ScrollChecker mScrollChecker;
    //private int mPagingTouchSlop;
    private int mHeaderHeight;
    private boolean mDisableWhenHorizontalMove = false;
    private int mFlag = 0x00;

    // disable when detect moving horizontally
    private boolean mPreventForHorizontal = false;

    private MotionEvent mLastMoveEvent;

    private SwipeUIRunnable runnable;

    private int mLoadingMinTime = 500;
    private long mLoadingStartTime = 0;
    private SwipeIndicator indicator;
    private boolean mHasSendCancelEvent = false;
    private Runnable mPerformRefreshCompleteDelay = new Runnable() {
        @Override
        public void run() {
            performRefreshComplete();
        }
    };

    public SwipeFrameLayout(Context context) {
        this(context, null);
    }

    public SwipeFrameLayout(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public SwipeFrameLayout(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);

        indicator = new SwipeIndicator();

        TypedArray arr = context.obtainStyledAttributes(attrs, R.styleable.SwipeFrameLayout, 0, 0);
        if (arr != null) {

            mHeaderId = arr.getResourceId(R.styleable.SwipeFrameLayout_swipe_header, mHeaderId);    //headerId
            mContainerId = arr.getResourceId(R.styleable.SwipeFrameLayout_swipe_content, mContainerId);//containerId

            indicator.setResistance(arr.getFloat(R.styleable.SwipeFrameLayout_swipe_resistance, indicator.getResistance()));//阻尼系数，默认1.7f

            mDurationToClose = arr.getInt(R.styleable.SwipeFrameLayout_swipe_duration_to_close, mDurationToClose);//回弹延时，默认 200ms，回弹到刷新高度所用时间
            mDurationToCloseHeader = arr.getInt(R.styleable.SwipeFrameLayout_swipe_duration_to_close_header, mDurationToCloseHeader);//头部回弹时间，默认 1000ms

            float ratio = indicator.getRatioOfHeaderToHeightRefresh();
            ratio = arr.getFloat(R.styleable.SwipeFrameLayout_swipe_ratio_of_header_height_to_refresh, ratio);//触发刷新时移动的位置比例，默认，1.2f倍
            indicator.setRatioOfHeaderHeightToRefresh(ratio);

            mKeepHeaderWhenRefresh = arr.getBoolean(R.styleable.SwipeFrameLayout_swipe_keep_header_when_refresh, mKeepHeaderWhenRefresh);//刷新是否保持头部，默认值 true

            mPullToRefresh = arr.getBoolean(R.styleable.SwipeFrameLayout_swipe_pull_to_refresh, mPullToRefresh);//下拉刷新 / 释放刷新，默认为释放刷新
            arr.recycle();
        }

        mScrollChecker = new SwipeFrameLayout.ScrollChecker();

//        final ViewConfiguration conf = ViewConfiguration.get(getContext());
//        mPagingTouchSlop = conf.getScaledTouchSlop() * 2;
    }

    //重写 onFinishInflate 方法来确定 Header 和 Content
    @Override
    protected void onFinishInflate() {
        final int childCount = getChildCount();
        if (childCount > 2) {
            throw new IllegalStateException("SwipeFrameLayout can only contains 2 children");
        } else if (childCount == 2) {
            if (mHeaderId != 0 && mHeaderView == null) {
                mHeaderView = findViewById(mHeaderId);
            }
            if (mContainerId != 0 && mContent == null) {
                mContent = findViewById(mContainerId);
            }

            // not specify header or content
            if (mContent == null || mHeaderView == null) {

                View child1 = getChildAt(0);
                View child2 = getChildAt(1);
                if (child1 instanceof ISwipeUIListener) {
                    mHeaderView = child1;
                    mContent = child2;
                } else if (child2 instanceof ISwipeUIListener) {
                    mHeaderView = child2;
                    mContent = child1;
                } else {
                    // both are not specified
                    if (mContent == null && mHeaderView == null) {
                        mHeaderView = child1;
                        mContent = child2;
                    }
                    // only one is specified
                    else {
                        if (mHeaderView == null) {
                            mHeaderView = mContent == child1 ? child2 : child1;
                        } else {
                            mContent = mHeaderView == child1 ? child2 : child1;
                        }
                    }
                }
            }
        } else if (childCount == 1) {
            mContent = getChildAt(0);
        } else {
            TextView errorView = new TextView(getContext());
            errorView.setClickable(true);
            errorView.setTextColor(0xffff6600);
            errorView.setGravity(Gravity.CENTER);
            errorView.setTextSize(20);
            errorView.setText("The content view in SwipeFrameLayout is empty. Do you forget to specify its id in xml layout file?");
            mContent = errorView;
            addView(mContent);
        }
        if (mHeaderView != null) {
            mHeaderView.bringToFront();
        }
        super.onFinishInflate();
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        if (mScrollChecker != null) {
            mScrollChecker.destroy();
        }

        if (mPerformRefreshCompleteDelay != null) {
            removeCallbacks(mPerformRefreshCompleteDelay);
        }
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);

        if (isDebug()) {
            Logger.w(getClass().getName(), String.format("onMeasure frame: width: %s, height: %s, padding: %s %s %s %s",
                    getMeasuredHeight(), getMeasuredWidth(),
                    getPaddingLeft(), getPaddingRight(), getPaddingTop(), getPaddingBottom()));

        }

        //测量header
        if (mHeaderView != null) {
            measureChildWithMargins(mHeaderView, widthMeasureSpec, 0, heightMeasureSpec, 0);
            MarginLayoutParams lp = (MarginLayoutParams) mHeaderView.getLayoutParams();
            mHeaderHeight = mHeaderView.getMeasuredHeight() + lp.topMargin + lp.bottomMargin;
            indicator.setHeaderHeight(mHeaderHeight);
        }

        //测量content
        if (mContent != null) {
            measureContentView(mContent, widthMeasureSpec, heightMeasureSpec);
            if (isDebug()) {
                ViewGroup.MarginLayoutParams lp = (MarginLayoutParams) mContent.getLayoutParams();
                Logger.w(getClass().getName(), String.format("onMeasure content, width: %s, height: %s, margin: %s %s %s %s",
                        getMeasuredWidth(), getMeasuredHeight(),
                        lp.leftMargin, lp.topMargin, lp.rightMargin, lp.bottomMargin));
                Logger.w(getClass().getName(), "onMeasure, currentPos: %s, lastPos: %s, top: %s",
                        indicator.getCurrentPosY(), indicator.getLastPosY(), mContent.getTop());
            }
        }
    }

    private void measureContentView(View child,
                                    int parentWidthMeasureSpec,
                                    int parentHeightMeasureSpec) {
        final MarginLayoutParams lp = (MarginLayoutParams) child.getLayoutParams();

        final int childWidthMeasureSpec = getChildMeasureSpec(parentWidthMeasureSpec,
                getPaddingLeft() + getPaddingRight() + lp.leftMargin + lp.rightMargin, lp.width);
        final int childHeightMeasureSpec = getChildMeasureSpec(parentHeightMeasureSpec,
                getPaddingTop() + getPaddingBottom() + lp.topMargin, lp.height);

        child.measure(childWidthMeasureSpec, childHeightMeasureSpec);
    }

    @Override
    protected void onLayout(boolean flag, int i, int j, int k, int l) {
        layoutChildren();
    }

    private void layoutChildren() {
        int offset = indicator.getCurrentPosY();
        int paddingLeft = getPaddingLeft();
        int paddingTop = getPaddingTop();

        if (mHeaderView != null) {
            MarginLayoutParams lp = (MarginLayoutParams) mHeaderView.getLayoutParams();
            final int left = paddingLeft + lp.leftMargin;
            // enhance readability(header is layout above screen when first init)
            final int top = -(mHeaderHeight - paddingTop - lp.topMargin - offset);
            final int right = left + mHeaderView.getMeasuredWidth();
            final int bottom = top + mHeaderView.getMeasuredHeight();
            mHeaderView.layout(left, top, right, bottom);
            if (isDebug()) {
                Logger.w(getClass().getName(), String.format("onLayout header: %s %s %s %s", left, top, right, bottom));
            }
        }
        if (mContent != null) {
            if (isPinContent()) {
                offset = 0;
            }
            MarginLayoutParams lp = (MarginLayoutParams) mContent.getLayoutParams();
            final int left = paddingLeft + lp.leftMargin;
            final int top = paddingTop + lp.topMargin + offset;
            final int right = left + mContent.getMeasuredWidth();
            final int bottom = top + mContent.getMeasuredHeight();
            if (isDebug()) {
                Logger.w(getClass().getName(), String.format("onLayout content: %s %s %s %s", left, top, right, bottom));
            }
            mContent.layout(left, top, right, bottom);
        }
    }

    @SuppressWarnings({"PointlessBooleanExpression", "ConstantConditions"})
    private boolean isDebug() {
        return DEBUG && DEBUG_LAYOUT;
    }

    public boolean dispatchTouchEventSupper(MotionEvent e) {
        return super.dispatchTouchEvent(e);
    }

    @Override
    public boolean dispatchTouchEvent(MotionEvent e) {
        if (!isEnabled() || mContent == null || mHeaderView == null) {
            return dispatchTouchEventSupper(e);
        }
        int action = e.getAction();
        switch (action) {
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                indicator.onRelease();
                if (indicator.hasLeftStartPosition()) {
                    if (DEBUG) {
                        Logger.w("call onRelease when user release");
                    }
                    onRelease(false);
                    if (indicator.hasMovedAfterPressedDown()) {
                        sendCancelEvent();
                        return true;
                    }
                    return dispatchTouchEventSupper(e);
                } else {
                    return dispatchTouchEventSupper(e);
                }

            case MotionEvent.ACTION_DOWN:
                mHasSendCancelEvent = false;
                indicator.onPressDown(e.getX(), e.getY());

                mScrollChecker.abortIfWorking();

                mPreventForHorizontal = false;
                // The cancel event will be sent once the position is moved.
                // So let the event pass to children.
                // fix #93, #102
                dispatchTouchEventSupper(e);
                return true;

            case MotionEvent.ACTION_MOVE:
                mLastMoveEvent = e;
                indicator.onMove(e.getX(), e.getY());
                float offsetX = indicator.getOffsetX();
                float offsetY = indicator.getOffsetY();

//                if (mDisableWhenHorizontalMove && !mPreventForHorizontal && (Math.abs(offsetX) > mPagingTouchSlop && Math.abs(offsetX) > Math.abs(offsetY))) {
//                    if (indicator.isInStartPosition()) {
//                        mPreventForHorizontal = true;
//                    }
//                }
                if (mDisableWhenHorizontalMove && !mPreventForHorizontal && Math.abs(offsetX) > Math.abs(offsetY)) {
                    if (indicator.isInStartPosition()) {
                        mPreventForHorizontal = true;
                    }
                }
                if (mPreventForHorizontal) {
                    return dispatchTouchEventSupper(e);
                }

                boolean moveDown = offsetY > 0;
                boolean moveUp = !moveDown;
                boolean canMoveUp = indicator.hasLeftStartPosition();

                if (DEBUG) {
                    boolean canMoveDown = listener != null && listener.checkCanDoRefresh(this, mContent, mHeaderView);
                    Logger.w(getClass().getName(),
                            String.format("ACTION_MOVE: offsetY:%s, currentPos: %s, moveUp: %s, canMoveUp: %s, moveDown: %s: canMoveDown: %s", offsetY, indicator.getCurrentPosY(), moveUp, canMoveUp, moveDown, canMoveDown));
                }

                // disable move when header not reach top
                if (moveDown && listener != null && !listener.checkCanDoRefresh(this, mContent, mHeaderView)) {
                    return dispatchTouchEventSupper(e);
                }

                if ((moveUp && canMoveUp) || moveDown) {
                    movePos(offsetY);
                    return true;
                }
        }
        return dispatchTouchEventSupper(e);
    }

    /**
     * if deltaY > 0, move the content down
     *
     * @param deltaY
     */
    private void movePos(float deltaY) {
        // has reached the top
        if ((deltaY < 0 && indicator.isInStartPosition())) {
            if (DEBUG) {
                Logger.w(String.format("has reached the top"));
            }
            return;
        }

        int to = indicator.getCurrentPosY() + (int) deltaY;

        // over top
        if (indicator.willOverTop(to)) {
            if (DEBUG) {
                Logger.w(String.format("over top"));
            }
            to = SwipeIndicator.POS_START;
        }

        indicator.setCurrentPos(to);
        int change = to - indicator.getLastPosY();
        updatePos(change);
    }

    private void updatePos(int change) {
        if (change == 0) {
            return;
        }

        boolean isUnderTouch = indicator.isUnderTouch();

        // once moved, cancel event will be sent to child
        if (isUnderTouch && !mHasSendCancelEvent && indicator.hasMovedAfterPressedDown()) {
            mHasSendCancelEvent = true;
            sendCancelEvent();
        }

        // leave initiated position or just refresh complete
        if ((indicator.hasJustLeftStartPosition() && mStatus == SWIPE_STATUS_INIT) ||
                (indicator.goDownCrossFinishPosition() && mStatus == SWIPE_STATUS_COMPLETE && isEnabledNextSwipeAtOnce())) {

            mStatus = SWIPE_STATUS_PREPARE;
            mSwipeUIListenerImpl.onUIRefreshPrepare(this);
            if (DEBUG) {
                Logger.w(getClass().getName(), String.format("SwipeUIListener: onUIRefreshPrepare, mFlag %s", mFlag));
            }
        }

        // back to initiated position
        if (indicator.hasJustBackToStartPosition()) {
            tryToNotifyReset();

            // recover event to children
            if (isUnderTouch) {
                sendDownEvent();
            }
        }

        // Pull to Refresh
        if (mStatus == SWIPE_STATUS_PREPARE) {
            // reach fresh height while moving from top to bottom
            if (isUnderTouch && !isAutoRefresh() && mPullToRefresh
                    && indicator.crossRefreshLineFromTopToBottom()) {
                tryToPerformRefresh();
            }
            // reach header height while auto refresh
            if (performAutoRefreshButLater() && indicator.hasJustReachedHeaderHeightFromTopToBottom()) {
                tryToPerformRefresh();
            }
        }

        if (DEBUG) {
            Logger.w(getClass().getName(), String.format("updatePos: change: %s, current: %s last: %s, top: %s, headerHeight: %s",
                    change, indicator.getCurrentPosY(), indicator.getLastPosY(), mContent.getTop(), mHeaderHeight));
        }

        mHeaderView.offsetTopAndBottom(change);
        if (!isPinContent()) {
            mContent.offsetTopAndBottom(change);
        }
        invalidate();

        if (mSwipeUIListenerImpl.hasHandler()) {
            mSwipeUIListenerImpl.onUIPositionChange(this, isUnderTouch, mStatus, indicator);
        }
        onPositionChange(isUnderTouch, mStatus, indicator);
    }

    protected void onPositionChange(boolean isInTouching, byte status, SwipeIndicator indicator) {
    }

    @SuppressWarnings("unused")
    public int getHeaderHeight() {
        return mHeaderHeight;
    }

    private void onRelease(boolean stayForLoading) {

        tryToPerformRefresh();

        if (mStatus == SWIPE_STATUS_LOADING) {
            // keep header for fresh
            if (mKeepHeaderWhenRefresh) {
                // scroll header back
                if (indicator.isOverOffsetToKeepHeaderWhileLoading() && !stayForLoading) {
                    mScrollChecker.tryToScrollTo(indicator.getOffsetToKeepHeaderWhileLoading(), mDurationToClose);
                } else {
                    // do nothing
                }
            } else {
                tryScrollBackToTopWhileLoading();
            }
        } else {
            if (mStatus == SWIPE_STATUS_COMPLETE) {
                notifyUIRefreshComplete(false);
            } else {
                tryScrollBackToTopAbortRefresh();
            }
        }
    }

    /**
     * please DO REMEMBER resume the hook
     *
     * @param swipeUIRunnable
     */

    public void setRefreshCompleteHook(SwipeUIRunnable swipeUIRunnable) {
        runnable = swipeUIRunnable;
        swipeUIRunnable.setResumeAction(new Runnable() {
            @Override
            public void run() {
                if (DEBUG) {
                    Logger.w("runnable resume.");
                }
                notifyUIRefreshComplete(true);
            }
        });
    }

    /**
     * Scroll back to to if is not under touch
     */
    private void tryScrollBackToTop() {
        if (!indicator.isUnderTouch()) {
            mScrollChecker.tryToScrollTo(SwipeIndicator.POS_START, mDurationToCloseHeader);
        }
    }

    /**
     * just make easier to understand
     */
    private void tryScrollBackToTopWhileLoading() {
        tryScrollBackToTop();
    }

    /**
     * just make easier to understand
     */
    private void tryScrollBackToTopAfterComplete() {
        tryScrollBackToTop();
    }

    /**
     * just make easier to understand
     */
    private void tryScrollBackToTopAbortRefresh() {
        tryScrollBackToTop();
    }

    private boolean tryToPerformRefresh() {
        if (mStatus != SWIPE_STATUS_PREPARE) {
            return false;
        }

        //
        if ((indicator.isOverOffsetToKeepHeaderWhileLoading() && isAutoRefresh()) || indicator.isOverOffsetToRefresh()) {
            mStatus = SWIPE_STATUS_LOADING;
            performRefresh();
        }
        return false;
    }

    private void performRefresh() {
        mLoadingStartTime = System.currentTimeMillis();
        if (mSwipeUIListenerImpl.hasHandler()) {
            mSwipeUIListenerImpl.onUIRefreshBegin(this);
            if (DEBUG) {
                Logger.w("SwipeUIListener: onUIRefreshBegin");
            }
        }
        if (listener != null) {
            listener.onRefreshBegin(this);
        }
    }

    /**
     * If at the top and not in loading, reset
     */
    private boolean tryToNotifyReset() {
        if ((mStatus == SWIPE_STATUS_COMPLETE || mStatus == SWIPE_STATUS_PREPARE) && indicator.isInStartPosition()) {
            if (mSwipeUIListenerImpl.hasHandler()) {
                mSwipeUIListenerImpl.onUIReset(this);
                if (DEBUG) {
                    Logger.w("SwipeUIListener: onUIReset");
                }
            }
            mStatus = SWIPE_STATUS_INIT;
            clearFlag();
            return true;
        }
        return false;
    }

    protected void onScrollAbort() {
        if (indicator.hasLeftStartPosition() && isAutoRefresh()) {
            if (DEBUG) {
                Logger.w("call onRelease after scroll abort");
            }
            onRelease(true);
        }
    }

    protected void onScrollFinish() {
        if (indicator.hasLeftStartPosition() && isAutoRefresh()) {
            if (DEBUG) {
                Logger.w("call onRelease after scroll finish");
            }
            onRelease(true);
        }
    }

    /**
     * Detect whether is refreshing.
     *
     * @return
     */
    public boolean isRefreshing() {
        return mStatus == SWIPE_STATUS_LOADING;
    }

    /**
     * Call this when data is loaded.
     * The UI will perform complete at once or after a delay, depends on the time elapsed is greater then {@link #mLoadingMinTime} or not.
     */
    final public void refreshComplete() {
        if (DEBUG) {
            Logger.w("refreshComplete");
        }

        if (runnable != null) {
            runnable.reset();
        }

        int delay = (int) (mLoadingMinTime - (System.currentTimeMillis() - mLoadingStartTime));
        if (delay <= 0) {
            if (DEBUG) {
                Logger.w("performRefreshComplete at once");
            }
            performRefreshComplete();
        } else {
            postDelayed(mPerformRefreshCompleteDelay, delay);
            if (DEBUG) {
                Logger.w("performRefreshComplete after delay: %s", delay);
            }
        }
    }

    /**
     * Do refresh complete work when time elapsed is greater than {@link #mLoadingMinTime}
     */
    private void performRefreshComplete() {
        mStatus = SWIPE_STATUS_COMPLETE;

        // if is auto refresh do nothing, wait scroller stop
        if (mScrollChecker.mIsRunning && isAutoRefresh()) {
            // do nothing
            if (DEBUG) {
                Logger.w(getClass().getName(), String.format("performRefreshComplete do nothing, scrolling: %s, auto refresh: %s",
                        mScrollChecker.mIsRunning, mFlag));
            }
            return;
        }

        notifyUIRefreshComplete(false);
    }

    /**
     * Do real refresh work. If there is a hook, execute the hook first.
     *
     * @param ignoreHook
     */
    private void notifyUIRefreshComplete(boolean ignoreHook) {
        /**
         * After hook operation is done, {@link #notifyUIRefreshComplete} will be call in resume action to ignore hook.
         */
        if (indicator.hasLeftStartPosition() && !ignoreHook && runnable != null) {
            if (DEBUG) {
                Logger.w("notifyUIRefreshComplete runnable run.");
            }

            runnable.takeOver();
            return;
        }
        if (mSwipeUIListenerImpl.hasHandler()) {
            if (DEBUG) {
                Logger.w("SwipeUIListener: onUIRefreshComplete");
            }
            mSwipeUIListenerImpl.onUIRefreshComplete(this);
        }
        indicator.onUIRefreshComplete();
        tryScrollBackToTopAfterComplete();
        tryToNotifyReset();
    }

    public void autoRefresh() {
        autoRefresh(true, mDurationToCloseHeader);
    }

    public void autoRefresh(boolean atOnce) {
        autoRefresh(atOnce, mDurationToCloseHeader);
    }

    private void clearFlag() {
        // remove auto fresh flag
        mFlag = mFlag & ~MASK_AUTO_REFRESH;
    }

    public void autoRefresh(boolean atOnce, int duration) {

        if (mStatus != SWIPE_STATUS_INIT) {
            return;
        }

        mFlag |= atOnce ? FLAG_AUTO_REFRESH_AT_ONCE : FLAG_AUTO_REFRESH_BUT_LATER;

        mStatus = SWIPE_STATUS_PREPARE;
        if (mSwipeUIListenerImpl.hasHandler()) {
            mSwipeUIListenerImpl.onUIRefreshPrepare(this);
            if (DEBUG) {
                Logger.w(getClass().getName(), String.format("SwipeUIListener: onUIRefreshPrepare, mFlag %s", mFlag));
            }
        }
        mScrollChecker.tryToScrollTo(indicator.getOffsetToRefresh(), duration);
        if (atOnce) {
            mStatus = SWIPE_STATUS_LOADING;
            performRefresh();
        }
    }

    public boolean isAutoRefresh() {
        return (mFlag & MASK_AUTO_REFRESH) > 0;
    }

    private boolean performAutoRefreshButLater() {
        return (mFlag & MASK_AUTO_REFRESH) == FLAG_AUTO_REFRESH_BUT_LATER;
    }

    public boolean isEnabledNextSwipeAtOnce() {
        return (mFlag & FLAG_ENABLE_NEXT_SWIPE_AT_ONCE) > 0;
    }

    /**
     * @param enable
     */
    public void setEnabledNextSwipeAtOnce(boolean enable) {
        if (enable) {
            mFlag = mFlag | FLAG_ENABLE_NEXT_SWIPE_AT_ONCE;
        } else {
            mFlag = mFlag & ~FLAG_ENABLE_NEXT_SWIPE_AT_ONCE;
        }
    }

    public boolean isPinContent() {
        return (mFlag & FLAG_PIN_CONTENT) > 0;
    }

    /**
     * The content view will now move when {@param pinContent} set to true.
     *
     * @param pinContent
     */
    public void setPinContent(boolean pinContent) {
        if (pinContent) {
            mFlag = mFlag | FLAG_PIN_CONTENT;
        } else {
            mFlag = mFlag & ~FLAG_PIN_CONTENT;
        }
    }

    /**
     * It's useful when working with viewpager.
     *
     * @param disable
     */
    public void disableWhenHorizontalMove(boolean disable) {
        mDisableWhenHorizontalMove = disable;
    }

    /**
     * loading will last at least for so long
     *
     * @param time
     */
    public void setLoadingMinTime(int time) {
        mLoadingMinTime = time;
    }

    /**
     * Not necessary any longer. Once moved, cancel event will be sent to child.
     *
     * @param yes
     */
    @Deprecated
    public void setInterceptEventWhileWorking(boolean yes) {
    }

    @SuppressWarnings({"unused"})
    public View getContentView() {
        return mContent;
    }

    public void setSwipeListener(ISwipeListener swipeListener) {
        listener = swipeListener;
    }

    public void addSwipeUIListener(ISwipeUIListener listener) {
        SwipeUIListenerImpl.addHandler(mSwipeUIListenerImpl, listener);
    }

    @SuppressWarnings({"unused"})
    public void removeSwipeUIListener(ISwipeUIListener listener) {
        mSwipeUIListenerImpl = SwipeUIListenerImpl.removeHandler(mSwipeUIListenerImpl, listener);
    }

    public void setSwipeIndicator(SwipeIndicator slider) {
        if (indicator != null && indicator != slider) {
            slider.convertFrom(indicator);
        }
        indicator = slider;
    }

    @SuppressWarnings({"unused"})
    public float getResistance() {
        return indicator.getResistance();
    }

    public void setResistance(float resistance) {
        indicator.setResistance(resistance);
    }

    @SuppressWarnings({"unused"})
    public float getDurationToClose() {
        return mDurationToClose;
    }

    /**
     * The duration to return back to the refresh position
     *
     * @param duration
     */
    public void setDurationToClose(int duration) {
        mDurationToClose = duration;
    }

    @SuppressWarnings({"unused"})
    public long getDurationToCloseHeader() {
        return mDurationToCloseHeader;
    }

    /**
     * The duration to close time
     *
     * @param duration
     */
    public void setDurationToCloseHeader(int duration) {
        mDurationToCloseHeader = duration;
    }

    public void setRatioOfHeaderHeightToRefresh(float ratio) {
        indicator.setRatioOfHeaderHeightToRefresh(ratio);
    }

    public int getOffsetToRefresh() {
        return indicator.getOffsetToRefresh();
    }

    @SuppressWarnings({"unused"})
    public void setOffsetToRefresh(int offset) {
        indicator.setOffsetToRefresh(offset);
    }

    @SuppressWarnings({"unused"})
    public float getRatioOfHeaderToHeightRefresh() {
        return indicator.getRatioOfHeaderToHeightRefresh();
    }

    @SuppressWarnings({"unused"})
    public int getOffsetToKeepHeaderWhileLoading() {
        return indicator.getOffsetToKeepHeaderWhileLoading();
    }

    @SuppressWarnings({"unused"})
    public void setOffsetToKeepHeaderWhileLoading(int offset) {
        indicator.setOffsetToKeepHeaderWhileLoading(offset);
    }

    @SuppressWarnings({"unused"})
    public boolean isKeepHeaderWhenRefresh() {
        return mKeepHeaderWhenRefresh;
    }

    public void setKeepHeaderWhenRefresh(boolean keepOrNot) {
        mKeepHeaderWhenRefresh = keepOrNot;
    }

    public boolean isPullToRefresh() {
        return mPullToRefresh;
    }

    public void setPullToRefresh(boolean pullToRefresh) {
        mPullToRefresh = pullToRefresh;
    }

    @SuppressWarnings({"unused"})
    public View getHeaderView() {
        return mHeaderView;
    }

    public void setHeaderView(View header) {
        if (mHeaderView != null && header != null && mHeaderView != header) {
            removeView(mHeaderView);
        }
        ViewGroup.LayoutParams lp = header.getLayoutParams();
        if (lp == null) {
            lp = new SwipeFrameLayout.LayoutParams(-1, -2);
            header.setLayoutParams(lp);
        }
        mHeaderView = header;
        addView(header);
    }

    @Override
    protected boolean checkLayoutParams(ViewGroup.LayoutParams p) {
        return p != null && p instanceof SwipeFrameLayout.LayoutParams;
    }

    @Override
    protected ViewGroup.LayoutParams generateDefaultLayoutParams() {
        return new SwipeFrameLayout.LayoutParams(SwipeFrameLayout.LayoutParams.MATCH_PARENT, SwipeFrameLayout.LayoutParams.MATCH_PARENT);
    }

    @Override
    protected ViewGroup.LayoutParams generateLayoutParams(ViewGroup.LayoutParams p) {
        return new SwipeFrameLayout.LayoutParams(p);
    }

    @Override
    public ViewGroup.LayoutParams generateLayoutParams(AttributeSet attrs) {
        return new SwipeFrameLayout.LayoutParams(getContext(), attrs);
    }

    private void sendCancelEvent() {
        if (DEBUG) {
            Logger.w("send cancel event");
        }
        // The ScrollChecker will update position and lead to send cancel event when mLastMoveEvent is null.
        // fix #104, #80, #92
        if (mLastMoveEvent == null) {
            return;
        }
        MotionEvent last = mLastMoveEvent;
        MotionEvent e = MotionEvent.obtain(last.getDownTime(), last.getEventTime() + ViewConfiguration.getLongPressTimeout(), MotionEvent.ACTION_CANCEL, last.getX(), last.getY(), last.getMetaState());
        dispatchTouchEventSupper(e);
    }

    private void sendDownEvent() {
        if (DEBUG) {
            Logger.w("send down event");
        }
        final MotionEvent last = mLastMoveEvent;
        MotionEvent e = MotionEvent.obtain(last.getDownTime(), last.getEventTime(), MotionEvent.ACTION_DOWN, last.getX(), last.getY(), last.getMetaState());
        dispatchTouchEventSupper(e);
    }

    public static class LayoutParams extends MarginLayoutParams {

        public LayoutParams(Context c, AttributeSet attrs) {
            super(c, attrs);
        }

        public LayoutParams(int width, int height) {
            super(width, height);
        }

        @SuppressWarnings({"unused"})
        public LayoutParams(MarginLayoutParams source) {
            super(source);
        }

        public LayoutParams(ViewGroup.LayoutParams source) {
            super(source);
        }
    }

    class ScrollChecker implements Runnable {

        private int mLastFlingY;
        private Scroller mScroller;
        private boolean mIsRunning = false;
        private int mStart;
        private int mTo;

        public ScrollChecker() {
            mScroller = new Scroller(getContext());
        }

        public void run() {
            boolean finish = !mScroller.computeScrollOffset() || mScroller.isFinished();
            int curY = mScroller.getCurrY();
            int deltaY = curY - mLastFlingY;
            if (DEBUG) {
                if (deltaY != 0) {
                    Logger.w(getClass().getName(), String.format("scroll: %s, start: %s, to: %s, currentPos: %s, current :%s, last: %s, delta: %s",
                            finish, mStart, mTo, indicator.getCurrentPosY(), curY, mLastFlingY, deltaY));
                }
            }
            if (!finish) {
                mLastFlingY = curY;
                movePos(deltaY);
                post(this);
            } else {
                finish();
            }
        }

        private void finish() {
            if (DEBUG) {
                Logger.w(getClass().getName(), String.format("finish, currentPos:%s", indicator.getCurrentPosY()));
            }
            reset();
            onScrollFinish();
        }

        private void reset() {
            mIsRunning = false;
            mLastFlingY = 0;
            removeCallbacks(this);
        }

        private void destroy() {
            reset();
            if (!mScroller.isFinished()) {
                mScroller.forceFinished(true);
            }
        }

        public void abortIfWorking() {
            if (mIsRunning) {
                if (!mScroller.isFinished()) {
                    mScroller.forceFinished(true);
                }
                onScrollAbort();
                reset();
            }
        }

        public void tryToScrollTo(int to, int duration) {
            if (indicator.isAlreadyHere(to)) {
                return;
            }
            mStart = indicator.getCurrentPosY();
            mTo = to;
            int distance = to - mStart;
            if (DEBUG) {
                Logger.w(getClass().getName(), String.format("tryToScrollTo: start: %s, distance:%s, to:%s", mStart, distance, to));
            }
            removeCallbacks(this);

            mLastFlingY = 0;

            // fix #47: Scroller should be reused
            if (!mScroller.isFinished()) {
                mScroller.forceFinished(true);
            }
            mScroller.startScroll(0, 0, 0, distance, duration);
            post(this);
            mIsRunning = true;
        }
    }
}
