package ml.karmaconfigs.api.bukkit.region;

/*
 * This file is part of KarmaAPI, licensed under the MIT License.
 *
 *  Copyright (c) karma (KarmaDev) <karmaconfigs@gmail.com>
 *  Copyright (c) contributors
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in all
 *  copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 *  SOFTWARE.
 */

import com.bergerkiller.bukkit.common.events.EntityMoveEvent;
import ml.karmaconfigs.api.bukkit.region.event.InteractAction;
import ml.karmaconfigs.api.bukkit.region.event.death.Forensics;
import ml.karmaconfigs.api.bukkit.region.event.entity.*;
import ml.karmaconfigs.api.bukkit.region.event.player.PlayerActionWithRegionEvent;
import ml.karmaconfigs.api.bukkit.region.event.player.PlayerInteractAtRegionEvent;
import org.bukkit.Bukkit;
import org.bukkit.block.Block;
import org.bukkit.block.BlockState;
import org.bukkit.block.data.Ageable;
import org.bukkit.entity.Entity;
import org.bukkit.entity.LivingEntity;
import org.bukkit.entity.Player;
import org.bukkit.event.Event;
import org.bukkit.event.EventHandler;
import org.bukkit.event.EventPriority;
import org.bukkit.event.Listener;
import org.bukkit.event.entity.*;
import org.bukkit.event.player.PlayerInteractEntityEvent;
import org.bukkit.event.player.PlayerInteractEvent;
import org.bukkit.event.player.PlayerJoinEvent;
import org.bukkit.event.player.PlayerMoveEvent;
import org.bukkit.material.Crops;
import org.bukkit.plugin.Plugin;
import org.codehaus.plexus.util.CachedMap;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Dummy listener for regions entity join/leave
 * event
 */
@SuppressWarnings({"unchecked", "unused"})
class DummyListener implements Listener {

    private final static CachedMap entity_cache = new CachedMap();

    private final Plugin plugin;

    /**
     * Initialize the dummy listener
     *
     * @param channel the dummy plugin channel
     */
    DummyListener(final Plugin channel) {
        plugin = channel;
    }

    /**
     * Event listener
     *
     * @param e the event
     */
    @EventHandler(priority = EventPriority.LOWEST)
    public void onMove(EntityMoveEvent e) {
        Entity entity = e.getEntity();

        Cuboid.getRegions().forEach((region) -> {
            Set<Entity> cached = (Set<Entity>) entity_cache.getOrDefault(region, Collections.newSetFromMap(new ConcurrentHashMap<>()));
            Set<Entity> insert = new HashSet<>();
            Set<Entity> remove = new HashSet<>();

            boolean changes = false;
            if (region.isInside(entity)) {
                if (!cached.contains(entity)) {
                    EntityJoinRegionEvent event = new EntityJoinRegionEvent(entity, region);
                    Bukkit.getServer().getPluginManager().callEvent(event);

                    insert.add(entity);

                    changes = true;
                }
            } else {
                if (cached.contains(entity)) {
                    EntityLeaveRegionEvent event = new EntityLeaveRegionEvent(entity, region);
                    Bukkit.getServer().getPluginManager().callEvent(event);

                    remove.add(entity);

                    changes = true;
                }
            }

            if (changes) {
                cached.removeAll(remove);
                cached.addAll(insert);

                entity_cache.put(region, cached);
            }
        });
    }

    /**
     * Event listener
     *
     * @param e the event
     */
    @EventHandler(priority = EventPriority.LOWEST)
    public void onMove(PlayerMoveEvent e) {
        if (!e.isCancelled()) {
            Player entity = e.getPlayer();

            Cuboid.getRegions().forEach((region) -> {
                    Set<Entity> cached = (Set<Entity>) entity_cache.getOrDefault(region, Collections.newSetFromMap(new ConcurrentHashMap<>()));
                    Set<Entity> insert = new HashSet<>();
                    Set<Entity> remove = new HashSet<>();

                boolean changes = false;
                if (region.isInside(entity)) {
                    if (!cached.contains(entity)) {
                        EntityJoinRegionEvent event = new EntityJoinRegionEvent(entity, region);
                        Bukkit.getServer().getPluginManager().callEvent(event);

                        insert.add(entity);

                        changes = true;
                    }
                } else {
                    if (cached.contains(entity)) {
                        EntityLeaveRegionEvent event = new EntityLeaveRegionEvent(entity, region);
                        Bukkit.getServer().getPluginManager().callEvent(event);

                        remove.add(entity);

                        changes = true;
                    }
                }

                if (changes) {
                    cached.removeAll(remove);
                    cached.addAll(insert);

                    entity_cache.put(region, cached);
                }
            });
        }
    }

    /**
     * Event listener
     *
     * @param e the event
     */
    @EventHandler(priority = EventPriority.LOWEST)
    public void onInteractAtEntity(PlayerInteractEntityEvent e) {
        if (!e.isCancelled()) {
            Player player = e.getPlayer();
            Entity target = e.getRightClicked();

            Cuboid.getRegions().forEach((region) -> {
                if (region.isInside(target)) {
                    PlayerInteractAtRegionEvent event = new PlayerInteractAtRegionEvent(target, player, region);
                    Bukkit.getServer().getPluginManager().callEvent(event);

                    e.setCancelled(event.isCancelled());
                }
            });
        }
    }

    /**
     * Event listener
     *
     * @param e the event
     */
    @EventHandler(priority = EventPriority.LOWEST)
    @SuppressWarnings("deprecation")
    public void onInteractPlayer(PlayerInteractEvent e) {
        Player player = e.getPlayer();
        Block block = e.getClickedBlock();

        if (block != null) {
            Cuboid.getRegions().forEach((region) -> {
                if (region.isInside(block)) {
                    Event.Result blockUse = e.useInteractedBlock();
                    Event.Result itemUse = e.useItemInHand();

                    if (blockUse.equals(Event.Result.ALLOW) && itemUse.equals(Event.Result.ALLOW)) {
                        InteractAction action = InteractAction.UNKNOWN;

                        switch (e.getAction()) {
                            case RIGHT_CLICK_BLOCK:
                                action = InteractAction.RIGHT_CLICK_BLOCK;
                                break;
                            case LEFT_CLICK_BLOCK:
                                action = InteractAction.LEFT_CLICK_BLOCK;
                                break;
                            case LEFT_CLICK_AIR:
                                action = InteractAction.LEFT_CLICK_AIR;
                                break;
                            case RIGHT_CLICK_AIR:
                                action = InteractAction.RIGHT_CLICK_AIR;
                                break;
                            case PHYSICAL:
                                BlockState state = block.getState();

                                //If we can get the crop states, then it's a soil item
                                try {
                                    Crops crop = (Crops) state.getData();
                                    action = InteractAction.JUMP_SOIL;
                                } catch (Throwable ex) {
                                    //Newer minecraft versions...
                                    if (state.getBlockData() instanceof Ageable) {
                                        Ageable ageable = (Ageable) state.getBlockData();
                                        action = InteractAction.JUMP_SOIL;
                                    }
                                }

                                if (action.equals(InteractAction.UNKNOWN)) {
                                    if (block.getType().name().contains("PLATE")) {
                                        action = InteractAction.PRESSURE_PLATE;
                                    } else {
                                        if (block.getType().name().contains("REDSTONE_ORE")) {
                                            action = InteractAction.REDSTONE_ORE;
                                        } else {
                                            action = InteractAction.TRIPWIRE;
                                        }
                                    }
                                }
                                break;
                        }

                        InteractAction finalAction = action;

                        PlayerActionWithRegionEvent event = new PlayerActionWithRegionEvent(player, block, finalAction, region);
                        Bukkit.getServer().getPluginManager().callEvent(event);

                        e.setCancelled(event.isCancelled());
                    }
                }
            });
        }
    }

    /**
     * Event listener
     *
     * @param e the event
     */
    @EventHandler(priority = EventPriority.HIGHEST)
    public void entityInteract(EntityInteractEvent e) {
        if (!e.isCancelled()) {
            Entity entity = e.getEntity();
            Block block = e.getBlock();

            Cuboid.getRegions().forEach((region) -> {
                if (region.isInside(block)) {
                    EntityInteractWithRegionEvent event = new EntityInteractWithRegionEvent(entity, block, region);
                    Bukkit.getServer().getPluginManager().callEvent(event);

                    e.setCancelled(event.isCancelled());
                }
            });
        }
    }

    /**
     * Event listener
     *
     * @param e the event
     */
    @EventHandler(priority = EventPriority.LOWEST)
    public void entitySpawn(EntitySpawnEvent e) {
        if (!e.isCancelled()) {
            Entity entity = e.getEntity();

            Cuboid.getRegions().forEach((region) -> {
                if (region.isInside(entity)) {
                    EntitySpawnAtRegionEvent event = new EntitySpawnAtRegionEvent(entity, region);
                    Bukkit.getServer().getPluginManager().callEvent(event);

                    e.setCancelled(event.isCancelled());

                    Set<Entity> cached = (Set<Entity>) entity_cache.getOrDefault(region, Collections.newSetFromMap(new ConcurrentHashMap<>()));
                    cached.add(entity);

                    entity_cache.put(region, cached);
                }
            });
        }
    }

    /**
     * Event listener
     *
     * @param e the event
     */
    @EventHandler(priority = EventPriority.LOWEST)
    public void playerJoin(PlayerJoinEvent e) {
        Player player = e.getPlayer();

        Cuboid.getRegions().forEach((region) -> {
            if (region.isInside(player)) {
                EntitySpawnAtRegionEvent event = new EntitySpawnAtRegionEvent(player, region);
                Bukkit.getServer().getPluginManager().callEvent(event);

                Set<Entity> cached = (Set<Entity>) entity_cache.getOrDefault(region, Collections.newSetFromMap(new ConcurrentHashMap<>()));
                cached.add(player);

                entity_cache.put(region, cached);
            }
        });
    }

    /**
     * Event listener
     *
     * @param e the event
     */
    @EventHandler(priority = EventPriority.LOWEST)
    public void entityDie(EntityDamageByEntityEvent e) {
        if (!e.isCancelled()) {
            Entity issuer = e.getDamager();
            Entity tmp = e.getEntity();

            EntityDamageEvent.DamageCause cause = e.getCause();
            double damage = e.getFinalDamage();

            Cuboid.getRegions().forEach((region) -> {
                if (region.isInside(issuer) || region.isInside(tmp)) {
                    if (tmp instanceof LivingEntity) {
                        LivingEntity entity = (LivingEntity) tmp;

                        if (damage >= entity.getHealth()) {
                            //We can assume the entity died as the damage caused is more than the entity's health
                            Forensics forensics = new Forensics(issuer, null, cause, damage);
                            EntityDieAtRegionEvent event = new EntityDieAtRegionEvent(entity, forensics, region);

                            if (event.isCancelled()) {
                                e.setCancelled(true);
                            } else {
                                Set<Entity> cached = (Set<Entity>) entity_cache.getOrDefault(region, Collections.newSetFromMap(new ConcurrentHashMap<>()));
                                cached.remove(entity);

                                entity_cache.put(region, cached);
                            }
                        }
                    } else {
                        plugin.getServer().getScheduler().runTaskLater(plugin, () -> {
                            if (tmp.isDead()) {
                                Forensics forensics = new Forensics(issuer, null, cause, damage);
                                EntityDieAtRegionEvent event = new EntityDieAtRegionEvent(tmp, forensics, region);

                                if (event.isCancelled()) {
                                    e.setCancelled(true);
                                } else {
                                    Set<Entity> cached = (Set<Entity>) entity_cache.getOrDefault(region, Collections.newSetFromMap(new ConcurrentHashMap<>()));
                                    cached.remove(tmp);

                                    entity_cache.put(region, cached);
                                }
                            }
                        }, 20);
                    }
                }
            });
        }
    }

    /**
     * Event listener
     *
     * @param e the event
     */
    @EventHandler(priority = EventPriority.LOWEST)
    public void entityDieBlock(EntityDamageByBlockEvent e) {
        if (!e.isCancelled()) {
            Block issuer = e.getDamager();
            Entity tmp = e.getEntity();

            EntityDamageEvent.DamageCause cause = e.getCause();
            double damage = e.getFinalDamage();

            Cuboid.getRegions().forEach((region) -> {
                if (region.isInside(issuer) || region.isInside(tmp)) {
                    if (tmp instanceof LivingEntity) {
                        LivingEntity entity = (LivingEntity) tmp;

                        if (damage >= entity.getHealth()) {
                            //We can assume the entity died as the damage caused is more than the entity's health
                            Forensics forensics = new Forensics(null, issuer, cause, damage);
                            EntityDieAtRegionEvent event = new EntityDieAtRegionEvent(entity, forensics, region);

                            if (event.isCancelled()) {
                                e.setCancelled(true);
                            } else {
                                Set<Entity> cached = (Set<Entity>) entity_cache.getOrDefault(region, Collections.newSetFromMap(new ConcurrentHashMap<>()));
                                cached.remove(entity);

                                entity_cache.put(region, cached);
                            }
                        }
                    } else {
                        plugin.getServer().getScheduler().runTaskLater(plugin, () -> {
                            if (tmp.isDead()) {
                                Forensics forensics = new Forensics(null, issuer, cause, damage);
                                EntityDieAtRegionEvent event = new EntityDieAtRegionEvent(tmp, forensics, region);

                                if (event.isCancelled()) {
                                    e.setCancelled(true);
                                } else {
                                    Set<Entity> cached = (Set<Entity>) entity_cache.getOrDefault(region, Collections.newSetFromMap(new ConcurrentHashMap<>()));
                                    cached.remove(tmp);

                                    entity_cache.put(region, cached);
                                }
                            }
                        }, 20);
                    }
                }
            });
        }
    }

    /**
     * Event listener
     *
     * @param e the event
     */
    @EventHandler(priority = EventPriority.LOWEST)
    public void entityDie(EntityDamageEvent e) {
        if (!e.isCancelled()) {
            Entity tmp = e.getEntity();

            EntityDamageEvent.DamageCause cause = e.getCause();
            double damage = e.getFinalDamage();

            Cuboid.getRegions().forEach((region) -> {
                if (region.isInside(tmp)) {
                    if (tmp instanceof LivingEntity) {
                        LivingEntity entity = (LivingEntity) tmp;

                        if (damage >= entity.getHealth()) {
                            //We can assume the entity died as the damage caused is more than the entity's health
                            Forensics forensics = new Forensics(null, null, cause, damage);
                            EntityDieAtRegionEvent event = new EntityDieAtRegionEvent(entity, forensics, region);

                            if (event.isCancelled()) {
                                e.setCancelled(true);
                            } else {
                                Set<Entity> cached = (Set<Entity>) entity_cache.getOrDefault(region, Collections.newSetFromMap(new ConcurrentHashMap<>()));
                                cached.remove(entity);

                                entity_cache.put(region, cached);
                            }
                        }
                    } else {
                        plugin.getServer().getScheduler().runTaskLater(plugin, () -> {
                            if (tmp.isDead()) {
                                Forensics forensics = new Forensics(null, null, cause, damage);
                                EntityDieAtRegionEvent event = new EntityDieAtRegionEvent(tmp, forensics, region);

                                if (event.isCancelled()) {
                                    e.setCancelled(true);
                                } else {
                                    Set<Entity> cached = (Set<Entity>) entity_cache.getOrDefault(region, Collections.newSetFromMap(new ConcurrentHashMap<>()));
                                    cached.remove(tmp);

                                    entity_cache.put(region, cached);
                                }
                            }
                        }, 20);
                    }
                }
            });
        }
    }
}
