package ml.karmaconfigs.api.bukkit.region.event.player;

/*
 * This file is part of KarmaAPI, licensed under the MIT License.
 *
 *  Copyright (c) karma (KarmaDev) <karmaconfigs@gmail.com>
 *  Copyright (c) contributors
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in all
 *  copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 *  SOFTWARE.
 */

import ml.karmaconfigs.api.bukkit.region.Cuboid;
import ml.karmaconfigs.api.bukkit.region.event.InteractAction;
import org.bukkit.block.Block;
import org.bukkit.entity.Player;
import org.bukkit.event.Cancellable;
import org.bukkit.event.HandlerList;
import org.bukkit.event.player.PlayerEvent;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Entity interact directly or indirectly
 * with a region
 */
public class PlayerActionWithRegionEvent extends PlayerEvent implements Cancellable {

    private final static HandlerList HANDLER_LIST = new HandlerList();

    private boolean cancelled = false;

    private final Cuboid region;
    private final Block block;
    private final InteractAction action;

    /**
     * Initialize the entity interact at region event
     *
     * @param ent the entity
     * @param target the target block ( if exists )
     * @param trigger the action that triggered the event
     * @param rg the region the entity
     *           has interacted in
     */
    public PlayerActionWithRegionEvent(final Player ent, final Block target, final InteractAction trigger, final Cuboid rg) {
        super(ent);

        block = target;
        action = trigger;

        region = rg;
    }

    /**
     * Gets the cancellation state of this event. A cancelled event will not
     * be executed in the server, but will still pass to other plugins
     *
     * @return true if this event is cancelled
     */
    @Override
    public boolean isCancelled() {
        return cancelled;
    }

    /**
     * the block that is involved
     * in the event
     *
     * @return the block
     */
    @Nullable
    public Block getBlock() {
        return block;
    }

    /**
     * Get the event trigger
     *
     * @return the event trigger
     */
    public InteractAction getTrigger() {
        return action;
    }

    /**
     * Get the region the entity is entering to
     *
     * @return the event region
     */
    public Cuboid getRegion() {
        return region;
    }

    /**
     * Get event handler list
     *
     * @return event handler list
     */
    @NotNull
    @Override
    public HandlerList getHandlers() {
        return HANDLER_LIST;
    }

    /**
     * Sets the cancellation state of this event. A cancelled event will not
     * be executed in the server, but will still pass to other plugins.
     *
     * @param cancel true if you wish to cancel this event
     */
    @Override
    public void setCancelled(final boolean cancel) {
        cancelled = cancel;
    }

    /**
     * Get a list of event handlers
     *
     * @return a list of event handlers
     */
    public static HandlerList getHandlerList() {
        return HANDLER_LIST;
    }
}
