package ml.karmaconfigs.remote.messaging.util.message;

import ml.karmaconfigs.api.common.utils.string.StringUtils;
import ml.karmaconfigs.remote.messaging.util.message.type.DataType;
import org.jetbrains.annotations.Nullable;

import java.io.Serializable;
import java.util.Base64;

public class MessageDataInput extends MessageInput {

    private final MessageOutput input;

    /**
     * Initialize the message input
     *
     * @param data the serialized base64 message output
     */
    public MessageDataInput(final byte[] data) {
        super(data);
        String serialized = new String(data);
        input = StringUtils.loadUnsafe(serialized);
    }

    /**
     * Initialize the message input
     *
     * @param out the message output
     */
    public MessageDataInput(final MessageOutput out) {
        super(out);
        input = out;
    }

    /**
     * Get the data object
     *
     * @param key the data key
     * @return the object
     */
    @Override
    public @Nullable Serializable getSerialized(final String key) {
        return input.get(key, DataType.SERIALIZABLE);
    }

    /**
     * Get the data object
     *
     * @param key the data key
     * @return the object
     */
    @Override
    public @Nullable CharSequence getSequence(final String key) {
        return input.get(key, DataType.SEQUENCE);
    }

    /**
     * Get the data object
     *
     * @param key the data key
     * @return the object
     */
    @Override
    public boolean getBoolean(final String key) {
        return Boolean.TRUE.equals(input.get(key, DataType.BOOLEAN));
    }

    /**
     * Get the data object
     *
     * @param key the data key
     * @return the object
     */
    @Override
    public @Nullable Number getNumber(final String key) {
        return input.get(key, DataType.NUMBER);
    }

    /**
     * Get the data object
     *
     * @param key the data key
     * @return the object
     */
    @Override
    public char[] getCharacters(final String key) {
        return input.get(key, DataType.CHARACTER);
    }

    /**
     * Get the data object
     *
     * @param key the data key
     * @return the object
     */
    @Override
    public byte[] getBytes(final String key) {
        return input.get(key, DataType.BYTE);
    }
}
