package mn.lambda.paypro.sdk.printer

import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Paint.Align
import android.util.Base64
import android.util.Log
import com.google.zxing.BarcodeFormat
import com.google.zxing.WriterException
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel

class BitmapPrinter(val context: Context, options: PrinterOptions) {
    private val bitmapDraw: BitmapDraw = BitmapDraw(context, options)
    private var log = StringBuilder()

    fun print(context: Context, receipt: Receipt): Bitmap {
        log = StringBuilder()
        drawBitmap(receipt.getPrintLines(context))
        Log.d("PayPRO SDK","Print receipt:\n$log")
        return bitmapDraw.bitmap!!
    }

    private fun getAlign(align: Int): Align {
        return when (align) {
            PrintItemAlign.LEFT -> {
                Align.LEFT
            }
            PrintItemAlign.CENTER -> {
                Align.CENTER
            }
            else -> {
                Align.RIGHT
            }
        }
    }

    /**
     * draw printing data to bitmap
     *
     * @param printItems draws the data item to be printed.
     */
    private fun drawBitmap(printItems: List<List<PrintItem>>) {
        for (lineItems in printItems) {
            for (printItem in lineItems) {
                when(printItem.type) {
                    PrintItemType.TEXT -> {
                        val item = printItem as PrintTextItem
                        //print text
                        bitmapDraw.addText(
                            item.text,
                            item.textSize,
                            item.isBold,
                            getAlign(item.align),
                            item.widthPercent
                        )
                        log.append(item.text)
                    }
                    PrintItemType.IMAGE -> {
                        val item = printItem as PrintImageItem
                        //print image
                        val imageBytes: ByteArray = Base64.decode(
                            item.base64Image.toByteArray(),
                            Base64.NO_WRAP or Base64.NO_PADDING or Base64.URL_SAFE
                        )
                        val decodedImage = BitmapFactory.decodeByteArray(imageBytes, 0, imageBytes.size)
                        bitmapDraw.addImage(
                            decodedImage,
                            getAlign(item.align),
                            item.widthPercent
                        )
                    }
                    PrintItemType.QRCODE -> {
                        val item = printItem as PrintQrCodeItem
                        try {
                            val bitmap: Bitmap = QRCodeUtils.create2dCode(
                                item.qrCode,
                                item.height,
                                ErrorCorrectionLevel.L
                            )
                            bitmapDraw.addImage(
                                bitmap,
                                getAlign(item.align),
                                item.widthPercent
                            )
                            log.append("[!qrCode ").append(item.qrCode).append("]")
                        } catch (e: WriterException) {
                            Log.e("PayPRO SDK","Create QR code bitmap failed. ${item.qrCode}")
                        }
                    }
                    PrintItemType.BARCODE -> {
                        val item = printItem as PrintBarCodeItem
                        val bitmap: Bitmap? = BarCodeUtils.createBarCodeWithText(
                            item.barcode,
                            BarcodeFormat.CODE_128,
                            item.width, item.height
                        )
                        if (bitmap == null) {
                            Log.e("PayPRO SDK","Create barcode bitmap failed. ${item.barcode}")
                        } else {
                            bitmapDraw.addImage(
                                bitmap,
                                getAlign(item.align),
                                item.widthPercent
                            )
                            log.append("[!barCode ").append(item.barcode).append("]")
                        }
                    }
                    PrintItemType.SKIP -> {
                        val item = printItem as PrintPaperSkipItem
                        bitmapDraw.feedPaper(item.height)
                        log.append(" ").append("  ")
                    }
                }
            }
            bitmapDraw.newline()
            log.append("\n")
        }
    }

}
