package mn.lambda.paypro.sdk.printer

import android.content.Context
import android.graphics.*
import androidx.annotation.IntRange

class BitmapDraw(context: Context, val options: PrinterOptions) {
    /**
     * Canvas
     */
    private var mCanvas: Canvas? = null
    private var receiptBmp: Bitmap? = null
    private var topY = 0f
    private var bottomY = 0f
    private var leftX = 0f
    private val boldTypeface: Typeface
    private val normalTypeface: Typeface

    init {
        boldTypeface = Typeface.createFromAsset(context.assets, options.boldFont)
        normalTypeface = Typeface.createFromAsset(context.assets, options.normalFont)
    }

    /**
     * create a bitmap
     */
    val bitmap: Bitmap?
        get() {
            deleteExtraCanvas()
            mCanvas = null
            return receiptBmp
        }

    /**
     * Draw text
     *
     * @param text         the text to be drawn
     * @param textSzie     set the paint's text size in pixel units
     * @param bold         true to set the text to bold, false to normal.
     * @param align        set the ratio of text width to paper
     * @param widthPercent set the ratio of text width to paper,from 0-100,
     * 0 to automatically adapt the occupied width according to the text width.
     */
    fun addText(
        text: String,
        textSize: Float,
        bold: Boolean,
        align: Paint.Align?,
        @IntRange(from = 0, to = 100) widthPercent: Int
    ) {
        if (mCanvas == null) {
            init()
        }
        val paint = Paint()
        paint.isFakeBoldText = bold
        paint.textSize = textSize
        paint.textAlign = align
        paint.isAntiAlias = true
        //Calculate the width of the occupied by text
        val width: Float
        val measureTextWidth = paint.measureText(text)
        width = if (widthPercent == 0) {
            Math.min(PAPER_CANVAS_WIDTH - leftX, measureTextWidth)
        } else {
            PAPER_CANVAS_WIDTH * widthPercent / 100
        }
        // Draws text on the specified coordinates
        val y = drawText(leftX, topY, text, paint, width)
        // Reset bottomY and leftX
        bottomY = Math.max(bottomY, y)
        leftX += width
        // Deal with the situation that the current line has ended
        if (leftX >= PAPER_CANVAS_WIDTH) {
            newline()
        }
    }

    /**
     * Draw image bitmap
     *
     * @param image        the image to be drawn
     * @param align        set the ratio of text width to paper
     * @param widthPercent set the ratio of image width to paper,from 0-100,
     * 0 to automatically adapt the occupied width according to the image width.
     */
    fun addImage(
        image: Bitmap,
        align: Paint.Align?,
        @IntRange(from = 0, to = 100) widthPercent: Int
    ) {
        if (mCanvas == null) {
            init()
        }
        val paint = Paint()
        paint.textAlign = align
        val width: Float
        width = if (widthPercent == 0) {
            Math.min(PAPER_CANVAS_WIDTH, image.width.toFloat())
        } else {
            Math.max(
                image.width.toFloat(),
                PAPER_CANVAS_WIDTH * widthPercent / 100
            )
        }
        val y = drawImage(leftX, topY, image, paint, width)
        bottomY = Math.max(bottomY, y)
        leftX += width
        if (leftX >= PAPER_CANVAS_WIDTH) {
            newline()
        }
    }

    /**
     * Move start position to next line
     */
    fun newline() {
        if (mCanvas == null) {
            init()
        }
        leftX = 0f
        topY = bottomY + options.lineSpace
        bottomY = topY
    }

    /**
     * Feed the paper
     */
    fun feedPaper(height: Float) {
        if (mCanvas == null) {
            init()
        }
        bottomY += height
        resizeCanvas(bottomY)
        newline()
    }

    /**
     * init canvas
     */
    private fun init() {
        topY = 0f
        bottomY = 0f
        leftX = 0f
        receiptBmp = Bitmap.createBitmap(PAPER_CANVAS_WIDTH.toInt(), 40, Bitmap.Config.ARGB_8888)
        mCanvas = Canvas(receiptBmp!!)
        mCanvas!!.drawColor(Color.WHITE)
    }

    /**
     * Resize canvas
     *
     * @param height the minimum canvas height
     */
    private fun resizeCanvas(height: Float) {
        if (receiptBmp!!.height > height) {
            return
        }
        val temp = Bitmap.createBitmap(
            PAPER_CANVAS_WIDTH.toInt(),
            (height.toInt() * 1.2).toInt(),
            Bitmap.Config.ARGB_8888
        )
        mCanvas = Canvas(temp)
        mCanvas!!.drawColor(Color.WHITE)
        mCanvas!!.drawBitmap(receiptBmp!!, 0f, 0f, Paint())
        receiptBmp = temp
    }

    /**
     * Delete redundant canvas
     */
    private fun deleteExtraCanvas() {
        if (receiptBmp!!.height.toFloat() == bottomY) {
            return
        }
        val temp = Bitmap.createBitmap(
            PAPER_CANVAS_WIDTH.toInt(),
            bottomY.toInt(),
            Bitmap.Config.ARGB_8888
        )
        mCanvas = Canvas(temp)
        mCanvas!!.drawColor(Color.WHITE)
        mCanvas!!.drawBitmap(receiptBmp!!, 0f, 0f, Paint())
        receiptBmp = temp
    }

    /**
     * Execute to draw text
     *
     * @param leftX     the X coordinate of the left side of the drawing
     * @param topY      the Y coordinate of the top of the drawing
     * @param value     text content
     * @param paint     text paint tool
     * @param rectWidth rect width
     * @return The Y coordinate of the bottom of the drawing
     */
    private fun drawText(
        leftX: Float,
        topY: Float,
        value: String,
        paint: Paint,
        rectWidth: Float
    ): Float {
        paint.typeface = if(paint.isFakeBoldText) boldTypeface else normalTypeface
        val subIndex = paint.breakText(value, 0, value.length, true, rectWidth, null)
        if (subIndex == 0) {
            return topY
        }
        val splitText = value.substring(0, subIndex)
        val otherText = value.substring(subIndex)
        val offsetX: Float
        offsetX = when (paint.textAlign) {
            Paint.Align.CENTER -> rectWidth / 2
            Paint.Align.RIGHT -> rectWidth
            else -> 0f
        }
        val fontMetrics = paint.fontMetrics
        val textHeight = fontMetrics.descent - fontMetrics.ascent
        resizeCanvas(topY + textHeight)
        mCanvas!!.drawText(splitText, leftX + offsetX, topY - fontMetrics.ascent, paint)
        mCanvas!!.save()
        return if (otherText.isNotEmpty()) {
            drawText(
                leftX,
                topY + textHeight + options.lineSpace,
                otherText,
                paint,
                rectWidth
            )
        } else topY + textHeight
    }

    /**
     * Execute to draw image
     *
     * @param leftX     the X coordinate of the left side of the drawing
     * @param topY      the Y coordinate of the top of the drawing
     * @param bitmap    the image bitmap
     * @param paint     paint tool
     * @param rectWidth rect width
     * @return The Y coordinate of the bottom of the drawing
     */
    private fun drawImage(
        leftX: Float,
        topY: Float,
        bitmap: Bitmap,
        paint: Paint,
        rectWidth: Float
    ): Float {
        var offsetX = 0f
        when (paint.textAlign) {
            Paint.Align.CENTER -> offsetX = (rectWidth - bitmap.width) / 2
            Paint.Align.RIGHT -> offsetX = rectWidth - bitmap.width
            else -> {}
        }
        resizeCanvas(topY + bitmap.height)
        mCanvas!!.drawBitmap(bitmap, leftX + offsetX, topY, paint)
        mCanvas!!.save()
        return topY + bitmap.height
    }

    companion object {
        /**
         * Paper width
         */
        private const val PAPER_WIDTH_NORMAL = 384f

        /**
         * Large paper width
         */
        private const val PAPER_WIDTH_LARGE = 576f

        /**
         * Invalid area width on both sides of the paper.
         */
        private const val WIDTH_PADDING = 5

        /**
         * Bitmap canvas width.
         */
        private const val PAPER_CANVAS_WIDTH = PAPER_WIDTH_NORMAL - WIDTH_PADDING * 2
    }
}
