/*
Copyright © 2014 by eBusiness Information
All rights reserved. This source code or any portion thereof
may not be reproduced or used in any manner whatsoever
without the express written permission of eBusiness Information.
*/
package mobi.designmyapp.common.utils;

import com.mortennobel.imagescaling.DimensionConstrain;
import com.mortennobel.imagescaling.ResampleOp;
import freemarker.template.*;
import mobi.designmyapp.common.model.Density;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.util.List;
import java.util.UUID;
import java.util.regex.Pattern;


public final class FileManagementUtils {

  private static final Logger LOGGER = LoggerFactory.getLogger(FileManagementUtils.class);
  public static final String OS = System.getProperty("os.name").toLowerCase();
  private static final String DIGIT_REGEX = "^[0-9]+$";

  private static final String IMAGE_URL_REGEX = "^(http\\:\\/\\/[a-zA-Z0-9\\-\\.]+\\.[a-zA-Z]{2,3}(?:\\/\\S*)?(?:[a-zA-Z0-9_])+\\.(?:jpg|jpeg|gif|png))$";
  private static final String PATH_REGEX = "^(\\$\\..*)";
  private static final String EMAIL_REGEX = "^(([^<>()\\[\\]\\.,;:\\s@\\\"]+(\\.[^<>()\\[\\]\\.,;:\\s@\\\"]+)*)|(\\\".+\\\"))@((\\[[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}\\])|(([a-zA-Z\\-0-9]+\\.)+[a-zA-Z]{2,}))$";
  private static final String PHONE_NUMBER_REGEX = "^\\+?[0-9\\. ]+$";

  private static final int MAX_ANDROID_DENSITY_RATIO = 4;
  private static final int DEFAULT_ANDROID_ICON_SIZE = 192;

  private static final Density[] ANDROID_DENSITIES = new Density[]{
      new Density("ldpi", 0.75f),
      new Density("mdpi", 1),
      new Density("hdpi", 1.5f),
      new Density("xhdpi", 2),
      new Density("xxhdpi", 3)
  };
  private static final int DEFAULT_IOS_ICON_SIZE = 152;
  private static final Density[] IOS_DENSITIES = new Density[]{
      new Density("-40", 40f/76f),
      new Density("-50", 50f/76f),
      new Density("-60", 60f/76f),
      new Density("-72", 72f/76f),
      new Density("-76", 1f),
      new Density("-small", 29f/76f),
      new Density("",57f/76f)
  };
  private FileManagementUtils() {
    // Prevents instantiation
  }

  public static String generateKey() {
    return UUID.randomUUID().toString();
  }

  /**
   * Copy srcFile to destFile
   */
  public static void copyFile(File srcFile, File destFile) {
    try {
      FileUtils.copyFile(srcFile, destFile);
    } catch (IOException e) {
      throw new IllegalStateException("Unable to copy file.", e);
    }
  }

  /**
   * Copy srcFile to destDir
   */
  public static void copyFileToDirectory(File srcFile, File destDir) {
    try {
      FileUtils.copyFileToDirectory(srcFile, destDir);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Copy all files in sourceDir to destDir.
   */
  public static void copyDirectoryContent(File sourceDir, File destDir) {
    try {
      FileUtils.copyDirectory(sourceDir, destDir);
    } catch (IOException e) {
      throw new IllegalStateException("Unable to copy directory.", e);
    }
  }

  /**
   * Escapes single and double quotes
   */
  public static String escapeQuotes(String s) {
    String strippedSlashes = StringUtils.replaceEach(s, new String[]{"\\\\'", "\\\\\"", "\\'", "\\\""}, new String[]{"'", "\"", "'", "\""});
    return StringUtils.replaceEach(strippedSlashes, new String[]{"'", "\""}, new String[]{"\\'", "\\\""});
  }


  /**
   * Perform token replacement in multiple files
   * @param model the model object. Any attribute name matching a token will have its value replace the token
   * @param srcDirectory the source directory (read-only)
   * @param parseList the relative paths from the source directory
   * @param destDirectory the destination directory
   */
  public static void replaceTokens(Object model, File srcDirectory, List<String> parseList, File destDirectory) {
    Configuration configuration = new Configuration();
    configuration.setObjectWrapper(new DefaultObjectWrapper());
    configuration.setDefaultEncoding("UTF-8");
    configuration.setTemplateExceptionHandler(TemplateExceptionHandler.IGNORE_HANDLER);
    try {
      for (String filePathToBeParsed : parseList) {
        File fileOutput = locateFile(destDirectory, filePathToBeParsed);
        Writer output = new OutputStreamWriter(new FileOutputStream(fileOutput));
        configuration.setDirectoryForTemplateLoading(srcDirectory);
        final Template template = configuration.getTemplate(filePathToBeParsed);
        template.process(model, output);
        if(output != null)
          output.close();
      }
    } catch (IOException e) {
      throw new RuntimeException(e);
    } catch (TemplateException e) {
      throw new RuntimeException(e);
    }
  }


  /**
   * Parse all the given files with the given freemarker configuration and copy them
   */
  public static void parseAndCopyAll(Configuration configuration, Object model, File sourceDirectory, List<String> filesToBeParsedRelativePath, File targetDirectory) {
    try {
      for (String filePathToBeParsed : filesToBeParsedRelativePath) {
        File fileOutput = new File(targetDirectory,filePathToBeParsed);
        if(fileOutput.exists()) {
          Writer output = new OutputStreamWriter(new FileOutputStream(fileOutput));
          configuration.setDirectoryForTemplateLoading(sourceDirectory);
          final Template template = configuration.getTemplate(filePathToBeParsed);
          template.process(model, output);
          if(output != null)
            output.close();
        }
        else
          LOGGER.warn("File {} doesn't exist. Not parsing this file",fileOutput.getAbsolutePath());

      }
    } catch (IOException e) {
      throw new RuntimeException(e);
    } catch (TemplateException e) {
      throw new RuntimeException(e);
    }
  }

  public static File locateFile(File sourceDir, String fileRelativePath) {
    final File file = new File(sourceDir.getAbsolutePath() + "/" + fileRelativePath);
    if (!file.exists())
      throw new IllegalArgumentException("File " + file.getAbsolutePath() + " doesn't exist.");
    return file;
  }

  private static String normalizeForFileName(String value) {
    value = value.replaceAll("\\W", "_");

    if (value.length() > 80)
      value = value.substring(0, 80);

    return value;
  }

  public static String computePackage(String appName, String applicationPackage, String templateTag) {
    return new StringBuilder(applicationPackage).append(".").append(templateTag).append(".").append(normalizeForFileName(appName).toLowerCase()).toString();
  }

  public static void copyDataFromTmpToWork(File tmp, File work){
    File workAssetsDir = getDataDir(work);
    if(tmp.exists())
      copyDirectoryContent(tmp,workAssetsDir);
  }

  public static void copySourceFromTmpToWork(File tmp, File work){
    File workAssetsDir = getSourceDir(work);
    if(tmp.exists())
      copyDirectoryContent(tmp,workAssetsDir);
  }

  private static File getDataDir(File parent) {
    File dataDir = new File(parent, "build/apk/assets/data");

    if (!dataDir.exists())
      dataDir.mkdirs();

    return dataDir;
  }

  private static File getSourceDir(File parent) {
    File dataDir = new File(parent, "build/apk/assets/sources");

    if (!dataDir.exists())
      dataDir.mkdirs();

    return dataDir;
  }

  public static void assertCanRead(String dir) {
    if (!new File(dir).canRead())
      throw new IllegalStateException(dir + " should be readable.");
  }

  public static void assertCanWrite(String dir) {
    if (!new File(dir).canWrite())
      throw new IllegalStateException(dir + " should be readable.");
  }

  public static String getExtension(String name) {
    final String[] fileNameParts = name.split("\\.");
    String lastPart = fileNameParts[fileNameParts.length - 1];
    return "." + lastPart;
  }

  public static String getNameWithoutExtension(String name) {
    final String[] fileNameParts = name.split("\\.");
    String firstPart = fileNameParts[0];
    return firstPart;
  }

  public static void copyAndroidDrawablesFromTmpToWork(File tmpDirectory, File workDirectory) {
    for (int i = 0; i < ANDROID_DENSITIES.length; i++) {
      Density density = ANDROID_DENSITIES[i];
      File drawable_tmp = new File(tmpDirectory.getAbsolutePath()+File.separator+"image/drawable-"+density.getDensity());
      File drawable_work = new File(workDirectory.getAbsolutePath()+"/res/");
      if(!drawable_tmp.exists()) {
        continue;
      }
      try {
        FileUtils.copyDirectoryToDirectory(drawable_tmp, drawable_work);
      } catch (IOException e) {
        LOGGER.error("Error copying drawable directory", e);
        return;
      }
    }
    return;
  }

  public static void moveImagesFromTmpToAssetsPictures(String tmpPath, String workPath){
    try {
      for (File file : new File(tmpPath+File.separator+"image").listFiles()) {
        if (file.isFile())
          FileUtils.copyFileToDirectory(file, new File(workPath+"/assets/pictures"));
      }
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  public static void cleanDirectory(File directory) {
    try {
      FileUtils.cleanDirectory(directory);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  public static String readFileToString(File file) throws IOException{
    return FileUtils.readFileToString(file);
  }

  public static void copyInputStreamToFile(InputStream inputStream, File file) {
    try {
      FileUtils.copyInputStreamToFile(inputStream, file);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  public static void deleteDirectory(File file) {
    try {
      FileUtils.deleteDirectory(file);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  public static void deleteFile(File file) {
    try {
      FileUtils.forceDelete(file);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  public static void moveFile(File srcFile, File destFile) {
    try {
      FileUtils.moveFile(srcFile, destFile);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  private static int getSizeForDensity(int originalSize, float densityFactor) {
    return (int) (originalSize * densityFactor / 4);
  }

  public static BufferedImage resizeImage(BufferedImage originalImage, int maxWidth, int maxHeight) {
    DimensionConstrain dimenConstrain = DimensionConstrain.createMaxDimension(maxWidth, maxHeight, true);
    ResampleOp resampleOp = new ResampleOp(dimenConstrain);
    return resampleOp.filter(originalImage, null);
  }

  public static void writeBufferedImageToFile(BufferedImage image, File file) {
    try {
      ImageIO.write(image, "png", file);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Create drawable reference in res/values/public.xml
   */
  public static void createImageReference(String imageName, int drawableId, File tempDirectory) {
    String referenceLine = "<public type=\"drawable\" name=\"" + FilenameUtils.removeExtension(ImageUtils.normalizeImageName(imageName)) + "\" id=\"0x" + Integer.toHexString(drawableId) + "\" />";

    File referencesFile = new File(tempDirectory, "/res/values/public.xml");
    try {
      List<String> lines = FileUtils.readLines(referencesFile);
      lines.add(lines.size() - 1, referenceLine);
      FileUtils.writeLines(referencesFile, "UTF-8", lines);
    } catch (IOException e) {
      LOGGER.error("Error creating drawable", e);
    }
  }

  public static long minToMillis(long minutes) {
    return 1000 * 60 * minutes;
  }


  public static boolean isImageUrl(String value) {
    if (value == null)
      return false;

    return Pattern.compile(IMAGE_URL_REGEX).matcher(value).matches();
  }

  public static boolean isJsonPath(String value) {
    if (value == null)
      return false;

    return Pattern.compile(PATH_REGEX).matcher(value).matches();
  }

  /**
   * Prevents path hacks such as ../my/path or /opt/my/path
   * @param pathToTest
   * @return
   */
  public static String parsePath(String pathToTest) {
    while(pathToTest.startsWith("/"))
      pathToTest = pathToTest.substring(1);
    pathToTest = pathToTest.replaceAll("\\.\\.","");
    while(pathToTest.endsWith("/"))
      pathToTest = pathToTest.substring(0,pathToTest.length()-1);
    return pathToTest;
  }

  /**
   * Valid a filename, must be alpha numeric, and accept - and _ too.
   *
   * @param fileName
   * @return
   */
  public static boolean isValidFileName(String fileName) {
    String valid = "[\\p{Alnum}\\s_-]+";
    return fileName.matches(valid);
  }

  public static void eraseAndCopyDir(File sourceDir, File destDir) {
    if (destDir.exists()) {
      try {
        FileUtils.cleanDirectory(destDir);
      } catch (IOException e) {
        boolean success = FileUtils.deleteQuietly(destDir);
        LOGGER.debug("FileUtils.cleanDirectory failed; deleting file: " + success);
      }
    }

    if (sourceDir.exists()) {
      try {
        FileUtils.copyDirectory(sourceDir, destDir);
      } catch (IOException e) {
        throw new IllegalStateException("Unable to copy directory.");
      }
    }

  }

  public static boolean isEmail(String value) {
    if (value == null || value.trim().isEmpty())
      return false;

    return Pattern.compile(EMAIL_REGEX).matcher(value).matches();
  }

  public static boolean isValidPhoneNumber(String value) {
    if (value == null || value.trim().isEmpty())
      return true;

    return Pattern.compile(PHONE_NUMBER_REGEX).matcher(value).matches();
  }

  public static boolean noHtmlText(String value) {
    if (value == null)
      return true;

    if (value.contains("<") || value.contains(">") || value.contains("=") || value.matches(".*\\\".*"))
      return false;
    if (value.contains("&gt") || value.contains("&lt") || value.contains("&quot"))
      return false;

    return true;
  }


  public static String stringListBuilder(List<String> items, String start, String separator, String end) {
    StringBuilder sb = new StringBuilder();
    if(items == null)
      return sb.append(start).append(end).toString();
    else if(items.size() == 1)
      return sb.append(start).append(items.get(0)).append(end).toString();
    else {
      int size = items.size();
      for (int i = 0; i < size; i++) {
        String item = items.get(i);
        if (i == 0)
          sb.append(start);
        sb.append(item);
        if (i < size - 1)
          sb.append(separator);
        else
          sb.append(end);
      }
      return sb.toString();
    }
  }

  public static String createHash(File f) {
    try {
      return DigestUtils.shaHex(new FileInputStream(f));
    } catch (IOException e) {
      throw new IllegalStateException("File not found:"+f.getAbsolutePath());
    }
  }
}
