/*
Copyright © 2014 by eBusiness Information
All rights reserved. This source code or any portion thereof
may not be reproduced or used in any manner whatsoever
without the express written permission of eBusiness Information.
*/
package mobi.designmyapp.common.utils;

import com.mortennobel.imagescaling.DimensionConstrain;
import com.mortennobel.imagescaling.ResampleOp;
import mobi.designmyapp.common.model.Density;
import mobi.designmyapp.common.model.DensityPx;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;

/**
 * Created by loic on 31/07/2014.
 */
public class ImageUtils {

  private static final int MAX_ANDROID_DENSITY_RATIO = 4;
  private static final int DEFAULT_ANDROID_ICON_SIZE = 192;

  private static final Density[] ANDROID_DENSITIES = new Density[]{
      new Density("ldpi", 0.75f),
      new Density("mdpi", 1),
      new Density("hdpi", 1.5f),
      new Density("xhdpi", 2),
      new Density("xxhdpi", 3)
  };
  private static final int DEFAULT_IOS_ICON_SIZE = 152;
  private static final Density[] IOS_DENSITIES = new Density[]{
      new Density("-40", 40f/76f),
      new Density("-50", 50f/76f),
      new Density("-57", 57f/76f),
      new Density("-60", 60f/76f),
      new Density("-72", 72f/76f),
      new Density("-76", 1f),
      new Density("-small", 29f/76f),
      new Density("",57f/76f)
  };

  private static final DensityPx[] IOS_SPLASH_DENSITIES = new DensityPx[]{
      new DensityPx("-568h@2x", 640,1136),
      new DensityPx("-Portrait-667h@2x", 750,1334),
      new DensityPx("-Portrait-736@3x", 1242,2208),
      new DensityPx("-Portrait~iPad", 768,1004),
      new DensityPx("-Portrait", 768,1024),
      new DensityPx("-Portrait@2x", 1536,2048),
      new DensityPx("-iPad", 768,1004),
      new DensityPx("", 320,480),
      new DensityPx("@2x",640,960)
  };

  /**
   * Resize image to fit icon specs
   * @param stream the input stream of the image
   * @param destFile the destination file
   */
  public static void resizeImageToIcon(InputStream stream, File destFile) {
    try {
      // Read image
      BufferedImage originalImage = ImageIO.read(stream);

      BufferedImage resizedImage = resizeImage(originalImage, DEFAULT_ANDROID_ICON_SIZE, DEFAULT_ANDROID_ICON_SIZE);
      writeBufferedImageToFile(resizedImage,destFile);
    }
    catch (IOException e) {
      throw new RuntimeException(e);
    }
  }


  /**
   * Export to android icons (ldpi/mdpi/hdpi/xhdpi/xxhdpi/xxxhdpi)
   * @param icon the icon file
   * @param directory the destination directory
   */
  public static void saveAndroidIcons(File icon, File directory, String iconName) {
    try {
      // Read image
      BufferedImage originalImage = ImageIO.read(icon);

      for (int i = 0; i < ANDROID_DENSITIES.length; i++) {
        Density density = ANDROID_DENSITIES[i];

        int maxWidth = getSizeForDensity(DEFAULT_ANDROID_ICON_SIZE, density.getFactor());
        int maxHeight = getSizeForDensity(DEFAULT_ANDROID_ICON_SIZE, density.getFactor());

        BufferedImage resizedImage = resizeImage(originalImage, maxWidth, maxHeight);

        // Write image in its associate density folder
        File drawableDirectory = new File(directory, "drawable-" + density.getDensity());
        drawableDirectory.mkdirs();

        File resizedImageFile = new File(drawableDirectory, iconName);

        writeBufferedImageToFile(resizedImage, resizedImageFile);

      }
      return;
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Export to ios icons (icon,icon@2x,40,40@2x,50,50@2x,60,60@2x,72,72@2x,76,76@2x,small,small@2x)
   * @param icon the icon file
   * @param directory the destination directory
   */
  public static void saveIosIcons(File icon, File directory, String iconName) {
    try {
      // Read image
      BufferedImage originalImage = ImageIO.read(icon);

      String iconNameWithoutExtension = FileManagementUtils.getNameWithoutExtension(iconName);

      for (int i = 0; i < IOS_DENSITIES.length; i++) {
        Density density = IOS_DENSITIES[i];

        int maxWidth = (int) (DEFAULT_IOS_ICON_SIZE * density.getFactor());
        int maxHeight = (int) (DEFAULT_IOS_ICON_SIZE * density.getFactor());

        BufferedImage resizedImage = resizeImage(originalImage, maxWidth/2, maxHeight/2);
        BufferedImage resizedImage2x = resizeImage(originalImage, maxWidth, maxHeight);

        File resizedImageFile = new File(directory, iconNameWithoutExtension+density.getDensity()+".png");
        writeBufferedImageToFile(resizedImage, resizedImageFile);
        File resizedImageFile2x = new File(directory, iconNameWithoutExtension+density.getDensity()+"@2x.png");
        writeBufferedImageToFile(resizedImage2x, resizedImageFile2x);

      }
      return;
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Export to ios splashscreens
   * @param ss the splashscreen image file
   * @param directory the destination directory
   */
  public static void saveIosSplashScreens(File ss, File directory) {
    try {
      // Read image
      BufferedImage originalImage = ImageIO.read(ss);

      String splashscreenName = "Default";

      for (int i = 0; i < IOS_SPLASH_DENSITIES.length; i++) {
        DensityPx density = IOS_SPLASH_DENSITIES[i];

        int maxWidth = density.getWidth();
        int maxHeight = density.getHeight();

        BufferedImage resizedImage = cropImage(originalImage, maxWidth, maxHeight);

        File resizedImageFile = new File(directory, splashscreenName+density.getDensity()+".png");
        writeBufferedImageToFile(resizedImage, resizedImageFile);

      }
      return;
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Get the size in px of a dimension in function of the density factor
   * @param originalMaxSize the original size in px
   * @param densityFactor the density factor
   * @return
   */
  private static int getSizeForDensity(int originalMaxSize, float densityFactor) {
    return (int) (originalMaxSize * densityFactor / MAX_ANDROID_DENSITY_RATIO);
  }

  /**
   * Normalize image name replacing non-alphanumeric characters by "_"
   */
  public static String normalizeImageName(String value) {
    value = value.toLowerCase();
    value = value.replaceAll("[^a-z0-9._]", "_");
    //Check for points before the extension
    String[] parts = value.split("\\.");
    StringBuilder concat = new StringBuilder(parts[0]);
    for (int i = 1; i < parts.length - 1; i++) {
      concat.append("_" + parts[i]);
    }
    concat.append("." + parts[parts.length - 1]);

    return concat.toString();
  }

  /**
   * Resize image for all densities (ldpi/mdpi/hdpi/xhdpi/xxdpi)
   *
   * @param imageName
   * @param imageFile
   * @param tmpDirectory
   * @param force
   */
  @Deprecated
  public static void resizeImageForAllAndroidDensities(String imageName, File imageFile, File tmpDirectory, boolean force) {
    resizeImageForAllAndroidDensities(imageName, imageFile, tmpDirectory);
  }

  /**
   * Resize image for all densities (ldpi/mdpi/hdpi/xhdpi/xxdpi)
   *
   * @param imageName
   * @param imageFile
   * @param tmpDirectory
   */
  public static void resizeImageForAllAndroidDensities(String imageName, File imageFile, File tmpDirectory) {
    try {
      // Read image
      BufferedImage originalImage = ImageIO.read(new FileInputStream(imageFile));

      int originalWidth = originalImage.getWidth();
      int originalHeight = originalImage.getHeight();

      for (int i = 0; i < ANDROID_DENSITIES.length; i++) {
        Density density = ANDROID_DENSITIES[i];

        int maxWidth = getSizeForDensity(originalWidth, density.getFactor());
        int maxHeight = getSizeForDensity(originalHeight, density.getFactor());

        BufferedImage resizedImage = resizeImage(originalImage, maxWidth, maxHeight);

        saveImageInFolder(imageName, tmpDirectory, density, resizedImage);
      }
      return;
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  private static void saveImageInFolder(String imageName, File tempDirectory, Density density, BufferedImage resizedImage) {
    // Create drawable folder if it does not exist
    File drawable = new File(tempDirectory, "drawable-"+density.getDensity());
    if(!drawable.exists())
      drawable.mkdir();

    // Write image on its associate density folder
    File resizedImageFile = new File(drawable, ImageUtils.normalizeImageName(imageName));
    writeBufferedImageToFile(resizedImage, resizedImageFile);
  }

  /**
   * Resize image to desired size
   * @param originalImage the BufferedImage
   * @param maxWidth the maximum width in px
   * @param maxHeight the maximum height in px
   * @return the resulting BufferedImage fitting the dimensions provided
   */
  public static BufferedImage resizeImage(BufferedImage originalImage, int maxWidth, int maxHeight) {
    DimensionConstrain dimenConstrain = DimensionConstrain.createMaxDimension(maxWidth, maxHeight, true);
    ResampleOp resampleOp = new ResampleOp(dimenConstrain);
    return resampleOp.filter(originalImage, null);
  }

  /**
   * Crop image to desired size
   * @param originalImage the BufferedImage
   * @param width the width in px
   * @param height the height in px
   * @return the resulting BufferedImage fitting the dimensions provided
   */
  public static BufferedImage cropImage(BufferedImage originalImage, int width, int height) {
    DimensionConstrain dimenConstrain = DimensionConstrain.createAbsolutionDimension(width, height);
    ResampleOp resampleOp = new ResampleOp(dimenConstrain);
    return resampleOp.filter(originalImage, null);
  }

  /**
   * Resize image to desired size
   * @param originalImage the BufferedImage
   * @param size the target size in px
   * @return the resulting BufferedImage fitting the dimensions provided
   */
  public static BufferedImage resizeImageSquare(BufferedImage originalImage, int size) {
    DimensionConstrain dimenConstrain = DimensionConstrain.createAbsolutionDimension(size, size);
    ResampleOp resampleOp = new ResampleOp(dimenConstrain);
    return resampleOp.filter(originalImage, null);
  }

  /**
   * Resize image to desired size
   * @param originalImageFile the File
   * @param size the target size in px
   * @return the resulting BufferedImage fitting the dimensions provided
   */
  public static BufferedImage resizeImageSquare(File originalImageFile, int size) {
    try {
      return resizeImageSquare(ImageIO.read(originalImageFile), size);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Resize image to desired size
   * @param originalImageFile the File
   * @param maxWidth the maximum width in px
   * @param maxHeight the maximum height in px
   * @return the resulting BufferedImage fitting the dimensions provided
   */
  public static BufferedImage resizeImage(File originalImageFile, int maxWidth, int maxHeight) {
    try {
      return resizeImage(ImageIO.read(originalImageFile),maxWidth,maxHeight);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Convert image to png
   * @param srcFile the source file
   * @param dstFile the destination png file
   */
  public static void convertToPng(File srcFile, File dstFile) {
    try {
      writeBufferedImageToFile(ImageIO.read(srcFile),dstFile);
      FileManagementUtils.deleteFile(srcFile);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Write buffered image to file
   * @param image the image
   * @param file the file
   */
  public static void writeBufferedImageToFile(BufferedImage image, File file) {
    try {
      if(file.exists())
        file.delete();
      ImageIO.write(image, "png", file);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }
}
