/*
 * Copyright 2012 Heiko Seeberger
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package name.heikoseeberger.lapislazuli

import LapislazuliParser._
import org.parboiled.errors.ErrorUtils
import org.pegdown.PegDownProcessor

object LapislazuliProcessor {

  val markdownProcessor = new PegDownProcessor

  def apply(input: String): Either[String, String] = {
    val result = LapislazuliParser(input)
    result.result map (node => Right(nodeToHtml(node))) getOrElse Left(ErrorUtils.printParseErrors(result))
  }

  def nodeToHtml(node: Node): String =
    node match {
      case Elem(name, attrs, content, children) =>
        val attrsString = attrsToString(attrs)
        val contentString = content getOrElse ""
        val childrenString = (children map nodeToHtml).mkString
        s"<$name$attrsString>$contentString$childrenString</$name>\n"
      case Plain(children) =>
        children map nodeToHtml mkString "\n"
      case Markdown(children) =>
        val input = children map nodeToHtml mkString "\n"
        val html = markdownProcessor markdownToHtml input
        html
      case Text(text: String, children: List[Text]) =>
        children.foldLeft(text.replace("----new-line----", ""))(_ + "\n" + nodeToHtml(_))
      case Comment(_) =>
        ""
    }

  def attrsToString(attrs: List[(String, String)]) =
    if (attrs.isEmpty)
      ""
    else
      (attrs map { case (name, value) => s"""$name="$value"""" }).mkString(" ", " ", "")
}
