/*
 * Copyright 2012 Heiko Seeberger
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package name.heikoseeberger.lapislazuli

import org.parboiled.Context
import org.parboiled.scala.parserunners.ReportingParseRunner
import org.parboiled.support.Position
import org.parboiled.scala._

object LapislazuliParser {

  def apply(input: String): ParsingResult[Node] =
    ReportingParseRunner((new LapislazuliParser).root) run input.transformIndents(4, "#~#", true)

  sealed trait Node {
    def children: List[Node]
  }

  case class Elem(
    name: String,
    attrs: List[(String, String)] = Nil,
    content: Option[String] = None,
    children: List[Node] = Nil) extends Node

  case class Comment(text: String) extends Node {
    val children: List[Node] = Nil
  }

  case class Plain(children: List[Text] = Nil) extends Node

  case class Markdown(children: List[Text] = Nil) extends Node

  object Text {
    def from(startColumn: => Int)(matchEndPosition: Position, children: List[Text], context: Context[Any]): Text = {
      val line = context.getInputBuffer.extractLine(matchEndPosition.line)
      val text = line.substring(startColumn - 1, matchEndPosition.column - 1)
      new Text(text, children)
    }
  }

  case class Text(text: String, children: List[Text] = Nil) extends Node
}

private class LapislazuliParser extends Parser {

  import LapislazuliParser._

  private var currentStartColumn: Int = _

  def root =
    rule(node ~ EOI)

  def node: Rule1[Node] =
    rule(elem | comment | plain | markdown)

  def elem =
    rule(name ~ attrs ~ content ~ eol ~ subNodes ~~> Elem)

  def comment =
    rule("-#" ~ chars ~> Comment ~ eol)

  def plain =
    rule(":plain" ~ eol ~ saveCurrentStartColumn _ ~ textSubNodes ~~> Plain)

  def markdown =
    rule(":markdown" ~ eol ~ saveCurrentStartColumn _ ~ textSubNodes ~~> Markdown)

  def saveCurrentStartColumn(context: Context[Any]): Unit =
    currentStartColumn = context.getPosition.column

  def subNodes =
    rule(INDENT ~ oneOrMore(node) ~ DEDENT | push(Nil))

  def textSubNodes: Rule1[List[Text]] =
    rule(INDENT ~ oneOrMore(textNode) ~ DEDENT | push(Nil))

  def textNode =
    rule(chars ~ pushFromContext(_.getPosition) ~ eol ~ textSubNodes ~~> withContext(Text.from(currentStartColumn) _))

  def attrs =
    rule(
      (idAttr ~ classAttr ~ commonAttrs ~~> (_ +: _ +: _)) |
        (idAttr ~ classAttr ~~> (List(_, _))) | (idAttr ~ commonAttrs ~~> (_ +: _)) | (classAttr ~ commonAttrs ~~> (_ +: _)) |
        idAttr ~~> (List(_)) | classAttr ~~> (List(_)) | commonAttrs |
        push(Nil)
    )

  def commonAttrs =
    rule("(" ~ zeroOrMore(attr, space) ~ ")")

  def attr =
    rule(name ~ "=\"" ~ unescapedChars ~ "\"")

  def idAttr =
    rule(("#" ~ name) ~~> ("id" -> _))

  def classAttr =
    rule(oneOrMore("." ~ name) ~~> (cs => "class" -> (cs mkString " ")))

  def content =
    rule(optional(space ~ oneOrMore(char) ~> identity))

  def name =
    rule(rule(alphaNumeric ~ zeroOrMore(alphaNumeric | "-" | "_")) ~> identity) // We need both rule calls, believe me!

  def alphaNumeric =
    rule("A" - "Z" | "a" - "z" | "0" - "9")

  def unescapedChars =
    rule(oneOrMore(unescapedChar) ~> identity)

  def chars =
    rule(zeroOrMore(char))

  def unescapedChar =
    rule(!("\"" | "\\") ~ char)

  def char =
    rule(!("\n" | EOI | INDENT | DEDENT) ~ ANY)

  def space =
    rule(oneOrMore(anyOf(" ")))

  def eol =
    rule("\n" | EOI)

}
