/*
 * Copyright 2017 Mikhail Lopatkin
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package name.mlopatkin.bitbucket.api;

import okhttp3.OkHttpClient;
import retrofit2.Retrofit;
import retrofit2.adapter.java8.Java8CallAdapterFactory;

import name.mlopatkin.bitbucket.api.rest.RepositoryEndpoint;

/**
 * Main entry point for the Bitbucket API.
 * <p>
 * This class allocates resources to serve network requests and therefore has to be explicitly
 * closed when not needed. Note that doing so will invalidate all objects retrieved via this class.
 */
public class Bitbucket implements AutoCloseable {
    /** Default Bitbucket's API endpoint */
    public static final String API_ENDPOINT = "https://api.bitbucket.org/2.0/";

    private final Retrofit mRetrofit;
    private final OkHttpClient mHttpClient;

    /**
     * Constructs an instance of Bitbucket API Facade with a given authenticator. The
     * authenticator is used to sign requests.
     *
     * @param authenticator the authenticator
     */
    public Bitbucket(RequestAuthenticator authenticator) {
        this(authenticator, API_ENDPOINT);
    }

    /**
     * Constructs an instance of Bitbucket API Facade with a given authenticator and a specified API
     * endpoint. The
     * authenticator is used to sign requests. The provided API endpoint overrides default
     * Bitbucket's API url (https://api.bitbucket.org/2.0/).
     *
     * This constructor can be used for functional testing with test servers or for proxying
     * requests.
     *
     * @param authenticator the authenticator
     * @param apiEndpoint the API endpoint URL
     */
    public Bitbucket(RequestAuthenticator authenticator, String apiEndpoint) {
        mHttpClient = new OkHttpClient.Builder()
                .addInterceptor(authenticator.makeInterceptor())
                .build();

        mRetrofit = new Retrofit.Builder()
                .baseUrl(apiEndpoint)
                .client(mHttpClient)
                .addCallAdapterFactory(Java8CallAdapterFactory.create())
                .build();
    }

    /**
     * Creates a handle object to access a repository on Bitbucket. Note that a valid {@link
     * Repository} object will be returned even if the repository with this username/repositoryName
     * doesn't exists on Bitbucket or authenticated account has no access to it. In such case most
     * methods of the returned objects will result in exception.
     *
     * @param username the username/teamname of the owner
     * @param repositoryName the name of the repository
     * @return the repository object
     */
    public Repository getRepository(String username, String repositoryName) {
        return new RepositoryImpl(
                mRetrofit.create(RepositoryEndpoint.class), username, repositoryName);
    }

    @Override
    public void close() {
        mHttpClient.dispatcher().executorService().shutdown();
    }
}
